package org.aspectj.bridge;

import org.aspectj.util.LangUtil;

import java.util.Enumeration;
import java.util.Hashtable;

public class CountingMessageHandler implements IMessageHandler {

    public final IMessageHandler delegate;

    public final CountingMessageHandler proxy;

    private final Hashtable<IMessage.Kind, IntHolder> counters;

    public static CountingMessageHandler makeCountingMessageHandler(IMessageHandler handler) {
        if (handler instanceof CountingMessageHandler) {
            return (CountingMessageHandler) handler;
        } else {
            return new CountingMessageHandler(handler);
        }
    }

    public CountingMessageHandler(IMessageHandler delegate) {
        LangUtil.throwIaxIfNull(delegate, "delegate");
        this.delegate = delegate;
        this.counters = new Hashtable<>();
        proxy = (delegate instanceof CountingMessageHandler ? (CountingMessageHandler) delegate : null);
    }

    public boolean handleMessage(IMessage message) throws AbortException {
        if (null != proxy) {
            return proxy.handleMessage(message);
        }
        if (null != message) {
            IMessage.Kind kind = message.getKind();
            if (!isIgnoring(kind)) {
                increment(kind);
            }
        }
        return delegate.handleMessage(message);
    }

    public boolean isIgnoring(IMessage.Kind kind) {
        return delegate.isIgnoring(kind);
    }

    public void dontIgnore(IMessage.Kind kind) {
        delegate.dontIgnore(kind);
    }

    public void ignore(IMessage.Kind kind) {
        delegate.ignore(kind);
    }

    public String toString() {
        return delegate.toString();
    }

    public int numMessages(IMessage.Kind kind, boolean orGreater) {
        if (null != proxy) {
            return proxy.numMessages(kind, orGreater);
        }
        int result = 0;
        if (null == kind) {
            for (Enumeration<IntHolder> enu = counters.elements(); enu.hasMoreElements(); ) {
                result += (enu.nextElement()).count;
            }
        } else if (!orGreater) {
            result = numMessages(kind);
        } else {
            for (IMessage.Kind k : IMessage.KINDS) {
                if (kind.isSameOrLessThan(k)) {
                    result += numMessages(k);
                }
            }
        }
        return result;
    }

    public boolean hasErrors() {
        return (0 < numMessages(IMessage.ERROR, true));
    }

    private int numMessages(IMessage.Kind kind) {
        if (null != proxy) {
            return proxy.numMessages(kind);
        }
        IntHolder counter = counters.get(kind);
        return (null == counter ? 0 : counter.count);
    }

    private void increment(IMessage.Kind kind) {
        if (null != proxy) {
            throw new IllegalStateException("not called when proxying");
        }
        IntHolder counter = counters.get(kind);
        if (null == counter) {
            counter = new IntHolder();
            counters.put(kind, counter);
        }
        counter.count++;
    }

    private static class IntHolder {

        int count;
    }

    public void reset() {
        if (proxy != null) {
            proxy.reset();
        }
        counters.clear();
    }
}
