package org.aspectj.bridge;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public interface IMessage {

    IMessage[] RA_IMessage = new IMessage[0];

    Kind WEAVEINFO = new Kind("weaveinfo", 5);

    Kind INFO = new Kind("info", 10);

    Kind USAGE = new Kind("usage", 15);

    Kind DEBUG = new Kind("debug", 20);

    Kind TASKTAG = new Kind("task", 25);

    Kind WARNING = new Kind("warning", 30);

    Kind ERROR = new Kind("error", 40);

    Kind FAIL = new Kind("fail", 50);

    Kind ABORT = new Kind("abort", 60);

    List<Kind> KINDS = Collections.unmodifiableList(Arrays.asList(new Kind[]{WEAVEINFO, INFO, USAGE, DEBUG, TASKTAG, WARNING, ERROR, FAIL, ABORT}));

    String getMessage();

    Kind getKind();

    boolean isError();

    boolean isWarning();

    boolean isDebug();

    boolean isUsage();

    boolean isInfo();

    boolean isAbort();

    boolean isTaskTag();

    boolean isFailed();

    boolean getDeclared();

    int getID();

    int getSourceStart();

    int getSourceEnd();

    Throwable getThrown();

    ISourceLocation getSourceLocation();

    final class Kind implements Comparable<IMessage.Kind> {

        public static final Comparator<IMessage.Kind> COMPARATOR = new Comparator<IMessage.Kind>() {

            public int compare(IMessage.Kind one, IMessage.Kind two) {
                if (null == one) {
                    return (null == two ? 0 : -1);
                } else if (null == two) {
                    return 1;
                } else if (one == two) {
                    return 0;
                } else {
                    return (one.precedence - two.precedence);
                }
            }
        };

        public boolean isSameOrLessThan(Kind kind) {
            return (0 >= COMPARATOR.compare(this, kind));
        }

        public int compareTo(IMessage.Kind other) {
            return COMPARATOR.compare(this, other);
        }

        private final int precedence;

        private final String name;

        private Kind(String name, int precedence) {
            this.name = name;
            this.precedence = precedence;
        }

        public String toString() {
            return name;
        }
    }

    String getDetails();

    List<ISourceLocation> getExtraSourceLocations();
}
