package org.aspectj.bridge;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class Message implements IMessage {

    private final String message;

    private final IMessage.Kind kind;

    private final Throwable thrown;

    private final ISourceLocation sourceLocation;

    private final String details;

    private final List<ISourceLocation> extraSourceLocations;

    private final boolean declared;

    private final int id;

    private final int sourceStart, sourceEnd;

    public Message(String message, ISourceLocation location, boolean isError) {
        this(message, (isError ? IMessage.ERROR : IMessage.WARNING), null, location);
    }

    public Message(String message, ISourceLocation location, boolean isError, ISourceLocation[] extraSourceLocations) {
        this(message, "", (isError ? IMessage.ERROR : IMessage.WARNING), location, null, (extraSourceLocations.length > 0 ? extraSourceLocations : null));
    }

    public Message(String message, String details, IMessage.Kind kind, ISourceLocation sourceLocation, Throwable thrown, ISourceLocation[] extraSourceLocations) {
        this(message, details, kind, sourceLocation, thrown, extraSourceLocations, false, 0, -1, -1);
    }

    public Message(String message, String details, IMessage.Kind kind, ISourceLocation sLoc, Throwable thrown, ISourceLocation[] otherLocs, boolean declared, int id, int sourcestart, int sourceend) {
        this.details = details;
        this.id = id;
        this.sourceStart = sourcestart;
        this.sourceEnd = sourceend;
        this.message = ((message != null) ? message : ((thrown == null) ? null : thrown.getMessage()));
        this.kind = kind;
        this.sourceLocation = sLoc;
        this.thrown = thrown;
        if (otherLocs != null) {
            this.extraSourceLocations = Collections.unmodifiableList(Arrays.asList(otherLocs));
        } else {
            this.extraSourceLocations = Collections.emptyList();
        }
        if (null == this.kind) {
            throw new IllegalArgumentException("null kind");
        }
        if (null == this.message) {
            throw new IllegalArgumentException("null message");
        }
        this.declared = declared;
    }

    public Message(String message, IMessage.Kind kind, Throwable thrown, ISourceLocation sourceLocation) {
        this(message, "", kind, sourceLocation, thrown, null);
    }

    public IMessage.Kind getKind() {
        return kind;
    }

    public boolean isError() {
        return kind == IMessage.ERROR;
    }

    public boolean isWarning() {
        return kind == IMessage.WARNING;
    }

    public boolean isDebug() {
        return kind == IMessage.DEBUG;
    }

    public boolean isUsage() {
        return kind == IMessage.USAGE;
    }

    public boolean isTaskTag() {
        return kind == IMessage.TASKTAG;
    }

    public boolean isInfo() {
        return kind == IMessage.INFO;
    }

    public boolean isAbort() {
        return kind == IMessage.ABORT;
    }

    public boolean getDeclared() {
        return declared;
    }

    public boolean isFailed() {
        return kind == IMessage.FAIL;
    }

    final public String getMessage() {
        return message;
    }

    final public Throwable getThrown() {
        return thrown;
    }

    final public ISourceLocation getSourceLocation() {
        return sourceLocation;
    }

    public String toString() {
        return MessageUtil.renderMessage(this, false);
    }

    public String getDetails() {
        return details;
    }

    public List<ISourceLocation> getExtraSourceLocations() {
        return extraSourceLocations;
    }

    public int getID() {
        return id;
    }

    public int getSourceStart() {
        return sourceStart;
    }

    public int getSourceEnd() {
        return sourceEnd;
    }
}
