package org.aspectj.bridge;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class MessageHandler implements IMessageHolder {

    protected final ArrayList<IMessage> messages;

    protected final List<IMessage.Kind> ignoring;

    protected boolean handleMessageResult;

    protected IMessageHandler interceptor;

    public MessageHandler() {
        this(false);
    }

    public MessageHandler(boolean accumulateOnly) {
        messages = new ArrayList<>();
        ignoring = new ArrayList<>();
        init(accumulateOnly);
        ignore(IMessage.WEAVEINFO);
    }

    public void init() {
        init(false);
    }

    public void init(boolean accumulateOnly) {
        handleMessageResult = accumulateOnly;
        if (0 < messages.size()) {
            messages.clear();
        }
        if (0 < ignoring.size()) {
            boolean ignoringWeaveMessages = isIgnoring(IMessage.WEAVEINFO);
            ignoring.clear();
            if (ignoringWeaveMessages) {
                ignore(IMessage.WEAVEINFO);
            }
        }
        if (null != interceptor) {
            interceptor = null;
        }
    }

    public void clearMessages() {
        if (0 < messages.size()) {
            messages.clear();
        }
    }

    public boolean handleMessage(IMessage message) {
        if ((null != interceptor) && (interceptor.handleMessage(message))) {
            return true;
        }
        if (null == message) {
            throw new IllegalArgumentException("null message");
        }
        if (!ignoring.contains(message.getKind())) {
            messages.add(message);
        }
        return handleMessageResult;
    }

    public boolean isIgnoring(IMessage.Kind kind) {
        return ((null != kind) && (ignoring.contains(kind)));
    }

    public void ignore(IMessage.Kind kind) {
        if ((null != kind) && (!ignoring.contains(kind))) {
            ignoring.add(kind);
        }
    }

    public void dontIgnore(IMessage.Kind kind) {
        if (null != kind) {
            ignoring.remove(kind);
        }
    }

    public boolean hasAnyMessage(final IMessage.Kind kind, final boolean orGreater) {
        if (null == kind) {
            return (0 < messages.size());
        }
        if (!orGreater) {
            for (IMessage m : messages) {
                if (kind == m.getKind()) {
                    return true;
                }
            }
        } else {
            for (IMessage m : messages) {
                if (kind.isSameOrLessThan(m.getKind())) {
                    return true;
                }
            }
        }
        return false;
    }

    public int numMessages(IMessage.Kind kind, final boolean orGreater) {
        if (null == kind) {
            return messages.size();
        }
        int result = 0;
        if (!orGreater) {
            for (IMessage m : messages) {
                if (kind == m.getKind()) {
                    result++;
                }
            }
        } else {
            for (IMessage m : messages) {
                if (kind.isSameOrLessThan(m.getKind())) {
                    result++;
                }
            }
        }
        return result;
    }

    public List<IMessage> getUnmodifiableListView() {
        return Collections.unmodifiableList(messages);
    }

    public IMessage[] getMessages(IMessage.Kind kind, final boolean orGreater) {
        if (null == kind) {
            return messages.toArray(IMessage.RA_IMessage);
        }
        ArrayList<IMessage> result = new ArrayList<>();
        if (!orGreater) {
            for (IMessage m : messages) {
                if (kind == m.getKind()) {
                    result.add(m);
                }
            }
        } else {
            for (IMessage m : messages) {
                if (kind.isSameOrLessThan(m.getKind())) {
                    result.add(m);
                }
            }
        }
        if (0 == result.size()) {
            return IMessage.RA_IMessage;
        }
        return result.toArray(IMessage.RA_IMessage);
    }

    public IMessage[] getErrors() {
        return getMessages(IMessage.ERROR, false);
    }

    public IMessage[] getWarnings() {
        return getMessages(IMessage.WARNING, false);
    }

    public void setInterceptor(IMessageHandler interceptor) {
        this.interceptor = interceptor;
    }

    public String toString() {
        if (0 == messages.size()) {
            return "MessageHandler: no messages";
        } else {
            return "MessageHandler: " + messages;
        }
    }
}
