package org.aspectj.util;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

public class FileUtil {

    public static final File DEFAULT_PARENT = new File(".");

    public static final List<String> SOURCE_SUFFIXES = Collections.unmodifiableList(Arrays.asList(new String[]{".java", ".aj"}));

    public static final FileFilter ZIP_FILTER = new FileFilter() {

        public boolean accept(File file) {
            return isZipFile(file);
        }

        public String toString() {
            return "ZIP_FILTER";
        }
    };

    final static int[] INT_RA = new int[0];

    public static final FileFilter ALL = new FileFilter() {

        public boolean accept(File f) {
            return true;
        }
    };

    public static final FileFilter DIRS_AND_WRITABLE_CLASSES = new FileFilter() {

        public boolean accept(File file) {
            return ((null != file) && (file.isDirectory() || (file.canWrite() && file.getName().toLowerCase().endsWith(".class"))));
        }
    };

    private static final boolean PERMIT_CVS;

    static {
        String name = FileUtil.class.getName() + ".PERMIT_CVS";
        PERMIT_CVS = LangUtil.getBoolean(name, false);
    }

    public static boolean isZipFile(File file) {
        if (file == null)
            return false;
        try (ZipFile zipFile = new ZipFile(file)) {
            return true;
        } catch (IOException e) {
            return false;
        }
    }

    public static int zipSuffixLength(File file) {
        return (null == file ? 0 : zipSuffixLength(file.getPath()));
    }

    public static int zipSuffixLength(String path) {
        if ((null != path) && (4 < path.length())) {
            String test = path.substring(path.length() - 4).toLowerCase();
            if (".zip".equals(test) || ".jar".equals(test)) {
                return 4;
            }
        }
        return 0;
    }

    public static boolean hasSourceSuffix(File file) {
        return ((null != file) && hasSourceSuffix(file.getPath()));
    }

    public static boolean hasSourceSuffix(String path) {
        return ((null != path) && (0 != sourceSuffixLength(path)));
    }

    public static int sourceSuffixLength(File file) {
        return (null == file ? 0 : sourceSuffixLength(file.getPath()));
    }

    public static int sourceSuffixLength(String path) {
        if (LangUtil.isEmpty(path)) {
            return 0;
        }
        for (String suffix : SOURCE_SUFFIXES) {
            if (path.endsWith(suffix) || path.toLowerCase().endsWith(suffix)) {
                return suffix.length();
            }
        }
        return 0;
    }

    public static boolean canReadDir(File dir) {
        return ((null != dir) && dir.canRead() && dir.isDirectory());
    }

    public static boolean canReadFile(File file) {
        return ((null != file) && file.canRead() && file.isFile());
    }

    public static boolean canWriteDir(File dir) {
        return ((null != dir) && dir.canWrite() && dir.isDirectory());
    }

    public static boolean canWriteFile(File file) {
        return ((null != file) && file.canWrite() && file.isFile());
    }

    public static void throwIaxUnlessCanReadDir(File dir, String label) {
        if (!canReadDir(dir)) {
            throw new IllegalArgumentException(label + " not readable dir: " + dir);
        }
    }

    public static void throwIaxUnlessCanWriteFile(File file, String label) {
        if (!canWriteFile(file)) {
            throw new IllegalArgumentException(label + " not writable file: " + file);
        }
    }

    public static void throwIaxUnlessCanWriteDir(File dir, String label) {
        if (!canWriteDir(dir)) {
            throw new IllegalArgumentException(label + " not writable dir: " + dir);
        }
    }

    public static String[] getPaths(File[] files) {
        if ((null == files) || (0 == files.length)) {
            return new String[0];
        }
        String[] result = new String[files.length];
        for (int i = 0; i < result.length; i++) {
            if (null != files[i]) {
                result[i] = files[i].getPath();
            }
        }
        return result;
    }

    public static String[] getPaths(List<File> files) {
        final int size = (null == files ? 0 : files.size());
        if (0 == size) {
            return new String[0];
        }
        String[] result = new String[size];
        for (int i = 0; i < size; i++) {
            File file = files.get(i);
            if (null != file) {
                result[i] = file.getPath();
            }
        }
        return result;
    }

    public static String fileToClassName(File basedir, File classFile) {
        LangUtil.throwIaxIfNull(classFile, "classFile");
        String classFilePath = normalizedPath(classFile);
        if (!classFilePath.endsWith(".class")) {
            String m = classFile + " does not end with .class";
            throw new IllegalArgumentException(m);
        }
        classFilePath = classFilePath.substring(0, classFilePath.length() - 6);
        if (null != basedir) {
            String basePath = normalizedPath(basedir);
            if (!classFilePath.startsWith(basePath)) {
                String m = classFile + " does not start with " + basedir;
                throw new IllegalArgumentException(m);
            }
            classFilePath = classFilePath.substring(basePath.length() + 1);
        } else {
            final String[] suffixes = new String[]{"com", "org", "java", "javax"};
            boolean found = false;
            for (int i = 0; !found && (i < suffixes.length); i++) {
                int loc = classFilePath.indexOf(suffixes[i] + "/");
                if ((0 == loc) || ((-1 != loc) && ('/' == classFilePath.charAt(loc - 1)))) {
                    classFilePath = classFilePath.substring(loc);
                    found = true;
                }
            }
            if (!found) {
                int loc = classFilePath.lastIndexOf("/");
                if (-1 != loc) {
                    classFilePath = classFilePath.substring(loc + 1);
                }
            }
        }
        return classFilePath.replace('/', '.');
    }

    public static String normalizedPath(File file, File basedir) {
        String filePath = normalizedPath(file);
        if (null != basedir) {
            String basePath = normalizedPath(basedir);
            if (filePath.startsWith(basePath)) {
                filePath = filePath.substring(basePath.length());
                if (filePath.startsWith("/")) {
                    filePath = filePath.substring(1);
                }
            }
        }
        return filePath;
    }

    public static String flatten(File[] files, String infix) {
        if (LangUtil.isEmpty(files)) {
            return "";
        }
        return flatten(getPaths(files), infix);
    }

    public static String flatten(String[] paths, String infix) {
        if (null == infix) {
            infix = File.pathSeparator;
        }
        StringBuilder result = new StringBuilder();
        boolean first = true;
        for (String path : paths) {
            if (null == path) {
                continue;
            }
            if (first) {
                first = false;
            } else {
                result.append(infix);
            }
            result.append(path);
        }
        return result.toString();
    }

    public static String normalizedPath(File file) {
        return (null == file ? "" : weakNormalize(file.getAbsolutePath()));
    }

    public static String weakNormalize(String path) {
        if (null != path) {
            path = path.replace('\\', '/').trim();
        }
        return path;
    }

    public static File getBestFile(String[] paths) {
        if (null == paths) {
            return null;
        }
        File result = null;
        for (int i = 0; (null == result) && (i < paths.length); i++) {
            String path = paths[i];
            if (null == path) {
                continue;
            }
            if (path.startsWith("sp:")) {
                try {
                    path = System.getProperty(path.substring(3));
                } catch (Throwable t) {
                    path = null;
                }
                if (null == path) {
                    continue;
                }
            }
            try {
                File f = new File(path);
                if (f.exists() && f.canRead()) {
                    result = FileUtil.getBestFile(f);
                }
            } catch (Throwable t) {
            }
        }
        return result;
    }

    public static File getBestFile(String[] paths, boolean mustBeJar) {
        if (null == paths) {
            return null;
        }
        File result = null;
        for (int i = 0; (null == result) && (i < paths.length); i++) {
            String path = paths[i];
            if (null == path) {
                continue;
            }
            if (path.startsWith("sp:")) {
                try {
                    path = System.getProperty(path.substring(3));
                } catch (Throwable t) {
                    path = null;
                }
                if (null == path) {
                    continue;
                }
            }
            try {
                File f = new File(path);
                if (f.exists() && f.canRead()) {
                    if (mustBeJar && !f.isDirectory()) {
                        result = FileUtil.getBestFile(f);
                    }
                }
            } catch (Throwable t) {
            }
        }
        return result;
    }

    public static File getBestFile(File file) {
        LangUtil.throwIaxIfNull(file, "file");
        if (file.exists()) {
            try {
                return file.getCanonicalFile();
            } catch (IOException e) {
                return file.getAbsoluteFile();
            }
        } else {
            return file;
        }
    }

    public static String getBestPath(File file) {
        LangUtil.throwIaxIfNull(file, "file");
        if (file.exists()) {
            try {
                return file.getCanonicalPath();
            } catch (IOException e) {
                return file.getAbsolutePath();
            }
        } else {
            return file.getPath();
        }
    }

    public static String[] getAbsolutePaths(File[] files) {
        if ((null == files) || (0 == files.length)) {
            return new String[0];
        }
        String[] result = new String[files.length];
        for (int i = 0; i < result.length; i++) {
            if (null != files[i]) {
                result[i] = files[i].getAbsolutePath();
            }
        }
        return result;
    }

    public static int deleteContents(File dir) {
        return deleteContents(dir, ALL);
    }

    public static int deleteContents(File dir, FileFilter filter) {
        return deleteContents(dir, filter, true);
    }

    public static int deleteContents(File dir, FileFilter filter, boolean deleteEmptyDirs) {
        if (null == dir) {
            throw new IllegalArgumentException("null dir");
        }
        if ((!dir.exists()) || (!dir.canWrite())) {
            return 0;
        }
        if (!dir.isDirectory()) {
            dir.delete();
            return 1;
        }
        String[] fromFiles = dir.list();
        if (fromFiles == null) {
            return 0;
        }
        int result = 0;
        for (String string : fromFiles) {
            File file = new File(dir, string);
            if ((null == filter) || filter.accept(file)) {
                if (file.isDirectory()) {
                    result += deleteContents(file, filter, deleteEmptyDirs);
                    String[] fileContent = file.list();
                    if (deleteEmptyDirs && fileContent != null && 0 == fileContent.length) {
                        file.delete();
                    }
                } else {
                    file.delete();
                    result++;
                }
            }
        }
        return result;
    }

    public static int copyDir(File fromDir, File toDir) throws IOException {
        return copyDir(fromDir, toDir, null, null);
    }

    public static int copyDir(File fromDir, File toDir, final String fromSuffix, String toSuffix) throws IOException {
        return copyDir(fromDir, toDir, fromSuffix, toSuffix, (FileFilter) null);
    }

    public static int copyDir(File fromDir, File toDir, final String fromSuffix, final String toSuffix, final FileFilter delegate) throws IOException {
        if ((null == fromDir) || (!fromDir.canRead())) {
            return 0;
        }
        final boolean haveSuffix = ((null != fromSuffix) && (0 < fromSuffix.length()));
        final int slen = (!haveSuffix ? 0 : fromSuffix.length());
        if (!toDir.exists()) {
            toDir.mkdirs();
        }
        final String[] fromFiles;
        if (!haveSuffix) {
            fromFiles = fromDir.list();
        } else {
            FilenameFilter filter = new FilenameFilter() {

                public boolean accept(File dir, String name) {
                    return (new File(dir, name).isDirectory() || (name.endsWith(fromSuffix)));
                }
            };
            fromFiles = fromDir.list(filter);
        }
        int result = 0;
        final int MAX = (null == fromFiles ? 0 : fromFiles.length);
        for (int i = 0; i < MAX; i++) {
            String filename = fromFiles[i];
            File fromFile = new File(fromDir, filename);
            if (fromFile.canRead()) {
                if (fromFile.isDirectory()) {
                    result += copyDir(fromFile, new File(toDir, filename), fromSuffix, toSuffix, delegate);
                } else if (fromFile.isFile()) {
                    if (haveSuffix) {
                        filename = filename.substring(0, filename.length() - slen);
                    }
                    if (null != toSuffix) {
                        filename = filename + toSuffix;
                    }
                    File targetFile = new File(toDir, filename);
                    if ((null == delegate) || delegate.accept(targetFile)) {
                        copyFile(fromFile, targetFile);
                    }
                    result++;
                }
            }
        }
        return result;
    }

    public static String[] listFiles(File srcDir) {
        ArrayList<String> result = new ArrayList<>();
        if ((null != srcDir) && srcDir.canRead()) {
            listFiles(srcDir, null, result);
        }
        return result.toArray(new String[0]);
    }

    public static final FileFilter aspectjSourceFileFilter = new FileFilter() {

        public boolean accept(File pathname) {
            String name = pathname.getName().toLowerCase();
            return name.endsWith(".java") || name.endsWith(".aj");
        }
    };

    public static File[] listFiles(File srcDir, FileFilter fileFilter) {
        ArrayList<File> result = new ArrayList<>();
        if ((null != srcDir) && srcDir.canRead()) {
            listFiles(srcDir, result, fileFilter);
        }
        return result.toArray(new File[0]);
    }

    public static List<File> listClassFiles(File dir) {
        ArrayList<File> result = new ArrayList<>();
        if ((null != dir) && dir.canRead()) {
            listClassFiles(dir, result);
        }
        return result;
    }

    public static File[] getBaseDirFiles(File basedir, String[] paths) {
        return getBaseDirFiles(basedir, paths, (String[]) null);
    }

    public static File[] getBaseDirFiles(File basedir, String[] paths, String[] suffixes) {
        LangUtil.throwIaxIfNull(basedir, "basedir");
        LangUtil.throwIaxIfNull(paths, "paths");
        File[] result = null;
        if (!LangUtil.isEmpty(suffixes)) {
            ArrayList<File> list = new ArrayList<>();
            for (String path : paths) {
                for (String suffix : suffixes) {
                    if (path.endsWith(suffix)) {
                        list.add(new File(basedir, path));
                        break;
                    }
                }
            }
            result = list.toArray(new File[0]);
        } else {
            result = new File[paths.length];
            for (int i = 0; i < result.length; i++) {
                result[i] = newFile(basedir, paths[i]);
            }
        }
        return result;
    }

    private static File newFile(File dir, String path) {
        if (".".equals(path)) {
            return dir;
        } else if ("..".equals(path)) {
            File parentDir = dir.getParentFile();
            if (null != parentDir) {
                return parentDir;
            } else {
                return new File(dir, "..");
            }
        } else {
            return new File(dir, path);
        }
    }

    public static File[] copyFiles(File srcDir, String[] relativePaths, File destDir) throws IllegalArgumentException, IOException {
        final String[] paths = relativePaths;
        throwIaxUnlessCanReadDir(srcDir, "srcDir");
        throwIaxUnlessCanWriteDir(destDir, "destDir");
        LangUtil.throwIaxIfNull(paths, "relativePaths");
        File[] result = new File[paths.length];
        for (int i = 0; i < paths.length; i++) {
            String path = paths[i];
            LangUtil.throwIaxIfNull(path, "relativePaths-entry");
            File src = newFile(srcDir, paths[i]);
            File dest = newFile(destDir, path);
            File destParent = dest.getParentFile();
            if (!destParent.exists()) {
                destParent.mkdirs();
            }
            LangUtil.throwIaxIfFalse(canWriteDir(destParent), "dest-entry-parent");
            copyFile(src, dest);
            result[i] = dest;
        }
        return result;
    }

    public static void copyFile(File fromFile, File toFile) throws IOException {
        LangUtil.throwIaxIfNull(fromFile, "fromFile");
        LangUtil.throwIaxIfNull(toFile, "toFile");
        LangUtil.throwIaxIfFalse(!toFile.equals(fromFile), "same file");
        if (toFile.isDirectory()) {
            throwIaxUnlessCanWriteDir(toFile, "toFile");
            if (fromFile.isFile()) {
                File targFile = new File(toFile, fromFile.getName());
                copyValidFiles(fromFile, targFile);
            } else if (fromFile.isDirectory()) {
                copyDir(fromFile, toFile);
            } else {
                LangUtil.throwIaxIfFalse(false, "not dir or file: " + fromFile);
            }
        } else if (toFile.isFile()) {
            if (fromFile.isDirectory()) {
                LangUtil.throwIaxIfFalse(false, "can't copy to file dir: " + fromFile);
            }
            copyValidFiles(fromFile, toFile);
        } else {
            ensureParentWritable(toFile);
            if (fromFile.isFile()) {
                copyValidFiles(fromFile, toFile);
            } else if (fromFile.isDirectory()) {
                toFile.mkdirs();
                throwIaxUnlessCanWriteDir(toFile, "toFile");
                copyDir(fromFile, toFile);
            } else {
                LangUtil.throwIaxIfFalse(false, "not dir or file: " + fromFile);
            }
        }
    }

    public static File ensureParentWritable(File path) {
        LangUtil.throwIaxIfNull(path, "path");
        File pathParent = path.getParentFile();
        if (null == pathParent) {
            pathParent = DEFAULT_PARENT;
        }
        if (!pathParent.canWrite()) {
            pathParent.mkdirs();
        }
        throwIaxUnlessCanWriteDir(pathParent, "pathParent");
        return pathParent;
    }

    public static void copyValidFiles(File fromFile, File toFile) throws IOException {
        try (FileInputStream in = new FileInputStream(fromFile);
             FileOutputStream out = new FileOutputStream(toFile)) {
            copyStream(in, out);
        }
    }

    @SuppressWarnings("deprecation")
    public static void copyStream(DataInputStream in, PrintStream out) throws IOException {
        LangUtil.throwIaxIfNull(in, "in");
        LangUtil.throwIaxIfNull(in, "out");
        String s;
        while (null != (s = in.readLine())) {
            out.println(s);
        }
    }

    public static void copyStream(InputStream in, OutputStream out) throws IOException {
        final int MAX = 4096;
        byte[] buf = new byte[MAX];
        for (int bytesRead = in.read(buf, 0, MAX); bytesRead != -1; bytesRead = in.read(buf, 0, MAX)) {
            out.write(buf, 0, bytesRead);
        }
    }

    public static void copyStream(Reader in, Writer out) throws IOException {
        final int MAX = 4096;
        char[] buf = new char[MAX];
        for (int bytesRead = in.read(buf, 0, MAX); bytesRead != -1; bytesRead = in.read(buf, 0, MAX)) {
            out.write(buf, 0, bytesRead);
        }
    }

    public static File makeNewChildDir(File parent, String child) {
        if (null == parent || !parent.canWrite() || !parent.isDirectory()) {
            throw new IllegalArgumentException("bad parent: " + parent);
        } else if (null == child) {
            child = "makeNewChildDir";
        } else if (!isValidFileName(child)) {
            throw new IllegalArgumentException("bad child: " + child);
        }
        File result = new File(parent, child);
        int safety = 1000;
        for (String suffix = FileUtil.randomFileString(); ((0 < --safety) && result.exists()); suffix = FileUtil.randomFileString()) {
            result = new File(parent, child + suffix);
        }
        if (result.exists()) {
            System.err.println("exhausted files for child dir in " + parent);
            return null;
        }
        return ((result.mkdirs() && result.exists()) ? result : null);
    }

    public static File getTempDir(String name) {
        if (null == name) {
            name = "FileUtil_getTempDir";
        } else if (!isValidFileName(name)) {
            throw new IllegalArgumentException(" invalid: " + name);
        }
        File result = null;
        File tempFile = null;
        try {
            tempFile = File.createTempFile("ignoreMe", ".txt");
            File tempParent = tempFile.getParentFile();
            result = makeNewChildDir(tempParent, name);
        } catch (IOException t) {
            result = makeNewChildDir(new File("."), name);
        } finally {
            if (null != tempFile) {
                tempFile.delete();
            }
        }
        return result;
    }

    public static URL[] getFileURLs(File[] files) {
        if ((null == files) || (0 == files.length)) {
            return new URL[0];
        }
        URL[] result = new URL[files.length];
        for (int i = 0; i < result.length; i++) {
            result[i] = getFileURL(files[i]);
        }
        return result;
    }

    @SuppressWarnings("deprecation")
    public static URL getFileURL(File file) {
        LangUtil.throwIaxIfNull(file, "file");
        URL result = null;
        try {
            result = file.toURL();
            if (null != result) {
                return result;
            }
            String url = "file:" + file.getAbsolutePath().replace('\\', '/');
            result = new URL(url + (file.isDirectory() ? "/" : ""));
        } catch (MalformedURLException e) {
            String m = "Util.makeURL(\"" + file.getPath() + "\" MUE " + e.getMessage();
            System.err.println(m);
        }
        return result;
    }

    public static String writeAsString(File file, String contents) {
        LangUtil.throwIaxIfNull(file, "file");
        if (null == contents) {
            contents = "";
        }
        Writer out = null;
        try {
            File parentDir = file.getParentFile();
            if (!parentDir.exists() && !parentDir.mkdirs()) {
                return "unable to make parent dir for " + file;
            }
            Reader in = new StringReader(contents);
            out = new FileWriter(file);
            FileUtil.copyStream(in, out);
            return null;
        } catch (IOException e) {
            return LangUtil.unqualifiedClassName(e) + " writing " + file + ": " + e.getMessage();
        } finally {
            if (null != out) {
                try {
                    out.close();
                } catch (IOException e) {
                }
            }
        }
    }

    public static boolean[] readBooleanArray(DataInputStream s) throws IOException {
        int len = s.readInt();
        boolean[] ret = new boolean[len];
        for (int i = 0; i < len; i++) {
            ret[i] = s.readBoolean();
        }
        return ret;
    }

    public static void writeBooleanArray(boolean[] a, DataOutputStream s) throws IOException {
        int len = a.length;
        s.writeInt(len);
        for (boolean b : a) {
            s.writeBoolean(b);
        }
    }

    public static int[] readIntArray(DataInputStream s) throws IOException {
        int len = s.readInt();
        int[] ret = new int[len];
        for (int i = 0; i < len; i++) {
            ret[i] = s.readInt();
        }
        return ret;
    }

    public static void writeIntArray(int[] a, DataOutputStream s) throws IOException {
        int len = a.length;
        s.writeInt(len);
        for (int j : a) {
            s.writeInt(j);
        }
    }

    public static String[] readStringArray(DataInputStream s) throws IOException {
        int len = s.readInt();
        String[] ret = new String[len];
        for (int i = 0; i < len; i++) {
            ret[i] = s.readUTF();
        }
        return ret;
    }

    public static void writeStringArray(String[] a, DataOutputStream s) throws IOException {
        if (a == null) {
            s.writeInt(0);
            return;
        }
        int len = a.length;
        s.writeInt(len);
        for (String value : a) {
            s.writeUTF(value);
        }
    }

    public static String readAsString(File file) throws IOException {
        BufferedReader r = new BufferedReader(new FileReader(file));
        StringBuilder b = new StringBuilder();
        while (true) {
            int ch = r.read();
            if (ch == -1) {
                break;
            }
            b.append((char) ch);
        }
        r.close();
        return b.toString();
    }

    public static List<String> readAsLines(File file) {
        try {
            return Files.readAllLines(Paths.get(file.toURI()));
        } catch (NoSuchFileException nsfe) {
            return Collections.emptyList();
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    public static byte[] readAsByteArray(File file) throws IOException {
        FileInputStream in = new FileInputStream(file);
        byte[] ret = FileUtil.readAsByteArray(in);
        in.close();
        return ret;
    }

    public static byte[] readAsByteArray(InputStream inStream) throws IOException {
        int size = 1024;
        byte[] ba = new byte[size];
        int readSoFar = 0;
        while (true) {
            int nRead = inStream.read(ba, readSoFar, size - readSoFar);
            if (nRead == -1) {
                break;
            }
            readSoFar += nRead;
            if (readSoFar == size) {
                int newSize = size * 2;
                byte[] newBa = new byte[newSize];
                System.arraycopy(ba, 0, newBa, 0, size);
                ba = newBa;
                size = newSize;
            }
        }
        byte[] newBa = new byte[readSoFar];
        System.arraycopy(ba, 0, newBa, 0, readSoFar);
        return newBa;
    }

    final static String FILECHARS = "abcdefghijklmnopqrstuvxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";

    static String randomFileString() {
        final double FILECHARS_length = FILECHARS.length();
        final int LEN = 6;
        final char[] result = new char[LEN];
        int index = (int) (Math.random() * 6d);
        for (int i = 0; i < LEN; i++) {
            if (index >= LEN) {
                index = 0;
            }
            result[index++] = FILECHARS.charAt((int) (Math.random() * FILECHARS_length));
        }
        return new String(result);
    }

    public static InputStream getStreamFromZip(String zipFile, String name) {
        try {
            ZipFile zf = new ZipFile(zipFile);
            try {
                ZipEntry entry = zf.getEntry(name);
                return zf.getInputStream(entry);
            } finally {
            }
        } catch (IOException ioe) {
            return null;
        }
    }

    public static List<String> lineSeek(String sought, List<String> sources, boolean listAll, PrintStream errorSink) {
        if (LangUtil.isEmpty(sought) || LangUtil.isEmpty(sources)) {
            return Collections.emptyList();
        }
        ArrayList<String> result = new ArrayList<>();
        for (String path : sources) {
            String error = lineSeek(sought, path, listAll, result);
            if ((null != error) && (null != errorSink)) {
                errorSink.println(error);
            }
        }
        return result;
    }

    public static String lineSeek(String sought, String sourcePath, boolean listAll, List<String> sink) {
        if (LangUtil.isEmpty(sought) || LangUtil.isEmpty(sourcePath)) {
            return "nothing sought";
        }
        if (LangUtil.isEmpty(sourcePath)) {
            return "no sourcePath";
        }
        final File file = new File(sourcePath);
        if (!file.canRead() || !file.isFile()) {
            return "sourcePath not a readable file";
        }
        int lineNum = 0;
        FileReader fin = null;
        try {
            fin = new FileReader(file);
            BufferedReader reader = new BufferedReader(fin);
            String line;
            while (null != (line = reader.readLine())) {
                lineNum++;
                int loc = line.indexOf(sought);
                if (-1 != loc) {
                    sink.add(sourcePath + ":" + lineNum + ":" + loc);
                    if (!listAll) {
                        break;
                    }
                }
            }
        } catch (IOException e) {
            return LangUtil.unqualifiedClassName(e) + " reading " + sourcePath + ":" + lineNum;
        } finally {
            try {
                if (null != fin) {
                    fin.close();
                }
            } catch (IOException e) {
            }
        }
        return null;
    }

    public static BufferedOutputStream makeOutputStream(File file) throws FileNotFoundException {
        File parent = file.getParentFile();
        if (parent != null) {
            parent.mkdirs();
        }
        return new BufferedOutputStream(new FileOutputStream(file));
    }

    public static boolean sleepPastFinalModifiedTime(File[] files) {
        if ((null == files) || (0 == files.length)) {
            return true;
        }
        long delayUntil = System.currentTimeMillis();
        for (File file : files) {
            if ((null == file) || !file.exists()) {
                continue;
            }
            long nextModTime = file.lastModified();
            if (nextModTime > delayUntil) {
                delayUntil = nextModTime;
            }
        }
        return LangUtil.sleepUntil(++delayUntil);
    }

    private static void listClassFiles(final File baseDir, ArrayList<File> result) {
        File[] files = baseDir.listFiles();
        for (File f : files) {
            if (f.isDirectory()) {
                listClassFiles(f, result);
            } else {
                if (f.getName().endsWith(".class")) {
                    result.add(f);
                }
            }
        }
    }

    private static void listFiles(final File baseDir, ArrayList<File> result, FileFilter filter) {
        File[] files = baseDir.listFiles();
        final boolean skipCVS = (!PERMIT_CVS && (filter == aspectjSourceFileFilter));
        for (File f : files) {
            if (f.isDirectory()) {
                if (skipCVS) {
                    String name = f.getName().toLowerCase();
                    if ("cvs".equals(name) || "sccs".equals(name)) {
                        continue;
                    }
                }
                listFiles(f, result, filter);
            } else {
                if (filter.accept(f)) {
                    result.add(f);
                }
            }
        }
    }

    private static boolean isValidFileName(String input) {
        return ((null != input) && (!input.contains(File.pathSeparator)));
    }

    private static void listFiles(final File baseDir, String dir, ArrayList<String> result) {
        final String dirPrefix = (null == dir ? "" : dir + "/");
        final File dirFile = (null == dir ? baseDir : new File(baseDir.getPath() + "/" + dir));
        final String[] files = dirFile.list();
        for (String file : files) {
            File f = new File(dirFile, file);
            String path = dirPrefix + file;
            if (f.isDirectory()) {
                listFiles(baseDir, path, result);
            } else {
                result.add(path);
            }
        }
    }

    private FileUtil() {
    }

    public static List<String> makeClasspath(URL[] urls) {
        List<String> ret = new LinkedList<>();
        if (urls != null) {
            for (URL url : urls) {
                ret.add(toPathString(url));
            }
        }
        return ret;
    }

    private static String toPathString(URL url) {
        try {
            return url.toURI().getPath();
        } catch (URISyntaxException e) {
            System.err.println("Warning!! Malformed URL may cause problems: " + url);
            return url.getPath();
        }
    }

    public static class Pipe implements Runnable {

        private final InputStream in;

        private final OutputStream out;

        private final long sleep;

        private ByteArrayOutputStream snoop;

        private long totalWritten;

        private Throwable thrown;

        private boolean halt;

        private final boolean closeInput;

        private final boolean closeOutput;

        private boolean finishStream;

        private boolean done;

        Pipe(InputStream in, OutputStream out) {
            this(in, out, 100l, false, false);
        }

        Pipe(InputStream in, OutputStream out, long sleep, boolean closeInput, boolean closeOutput) {
            LangUtil.throwIaxIfNull(in, "in");
            LangUtil.throwIaxIfNull(out, "out");
            this.in = in;
            this.out = out;
            this.closeInput = closeInput;
            this.closeOutput = closeOutput;
            this.sleep = Math.min(0l, Math.max(60l * 1000l, sleep));
        }

        public void setSnoop(ByteArrayOutputStream snoop) {
            this.snoop = snoop;
        }

        public void run() {
            totalWritten = 0;
            if (halt) {
                return;
            }
            try {
                final int MAX = 4096;
                byte[] buf = new byte[MAX];
                int count = in.read(buf, 0, MAX);
                ByteArrayOutputStream mySnoop;
                while ((halt && finishStream && (0 < count)) || (!halt && (-1 != count))) {
                    out.write(buf, 0, count);
                    mySnoop = snoop;
                    if (null != mySnoop) {
                        mySnoop.write(buf, 0, count);
                    }
                    totalWritten += count;
                    if (halt && !finishStream) {
                        break;
                    }
                    if (!halt && (0 < sleep)) {
                        Thread.sleep(sleep);
                    }
                    if (halt && !finishStream) {
                        break;
                    }
                    count = in.read(buf, 0, MAX);
                }
            } catch (Throwable e) {
                thrown = e;
            } finally {
                halt = true;
                if (closeInput) {
                    try {
                        in.close();
                    } catch (IOException e) {
                    }
                }
                if (closeOutput) {
                    try {
                        out.close();
                    } catch (IOException e) {
                    }
                }
                done = true;
                completing(totalWritten, thrown);
            }
        }

        public boolean halt(boolean wait, boolean finishStream) {
            if (!halt) {
                halt = true;
            }
            if (wait) {
                while (!done) {
                    synchronized (this) {
                        notifyAll();
                    }
                    if (!done) {
                        try {
                            Thread.sleep(5l);
                        } catch (InterruptedException e) {
                            break;
                        }
                    }
                }
            }
            return halt;
        }

        public long totalWritten() {
            return totalWritten;
        }

        public Throwable getThrown() {
            return thrown;
        }

        protected void completing(long totalWritten, Throwable thrown) {
        }
    }
}
