package org.aspectj.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.security.PrivilegedActionException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

public class LangUtil {

    public static final String EOL;

    public static final String JRT_FS = "jrt-fs.jar";

    private static double vmVersion;

    public static String getVmVersionString() {
        return Double.toString(vmVersion);
    }

    public static double getVmVersion() {
        return vmVersion;
    }

    static {
        StringWriter buf = new StringWriter();
        PrintWriter writer = new PrintWriter(buf);
        writer.println("");
        String eol = "\n";
        try {
            buf.close();
            StringBuffer sb = buf.getBuffer();
            if (sb != null) {
                eol = buf.toString();
            }
        } catch (Throwable t) {
        }
        EOL = eol;
    }

    static {
        try {
            String vm = System.getProperty("java.version");
            if (vm == null) {
                vm = System.getProperty("java.runtime.version");
            }
            if (vm == null) {
                vm = System.getProperty("java.vm.version");
            }
            if (vm == null) {
                new RuntimeException("System properties appear damaged, cannot find: java.version/java.runtime.version/java.vm.version").printStackTrace(System.err);
                vmVersion = 1.5;
            } else {
                try {
                    List<Integer> numbers = getFirstNumbers(vm);
                    if (numbers.get(0) == 1) {
                        vmVersion = numbers.get(0) + (numbers.get(1) / 10d);
                    } else {
                        vmVersion = numbers.get(0);
                    }
                } catch (Throwable t) {
                    vmVersion = 1.5;
                }
            }
        } catch (Throwable t) {
            new RuntimeException("System properties appear damaged, cannot find: java.version/java.runtime.version/java.vm.version", t).printStackTrace(System.err);
            vmVersion = 1.5;
        }
    }

    private static List<Integer> getFirstNumbers(String vm) {
        List<Integer> result = new ArrayList<>();
        StringTokenizer st = new StringTokenizer(vm, ".-_");
        try {
            result.add(Integer.parseInt(st.nextToken()));
            result.add(Integer.parseInt(st.nextToken()));
        } catch (Exception e) {
        }
        return result;
    }

    @Deprecated
    public static boolean is1dot3VMOrGreater() {
        return 1.3 <= vmVersion;
    }

    @Deprecated
    public static boolean is1dot4VMOrGreater() {
        return 1.4 <= vmVersion;
    }

    @Deprecated
    public static boolean is1dot5VMOrGreater() {
        return 1.5 <= vmVersion;
    }

    @Deprecated
    public static boolean is1dot6VMOrGreater() {
        return 1.6 <= vmVersion;
    }

    @Deprecated
    public static boolean is1dot7VMOrGreater() {
        return 1.7 <= vmVersion;
    }

    public static boolean is1dot8VMOrGreater() {
        return 1.8 <= vmVersion;
    }

    public static boolean is9VMOrGreater() {
        return 9 <= vmVersion;
    }

    public static boolean is10VMOrGreater() {
        return 10 <= vmVersion;
    }

    public static boolean is11VMOrGreater() {
        return 11 <= vmVersion;
    }

    public static boolean is12VMOrGreater() {
        return 12 <= vmVersion;
    }

    public static boolean is13VMOrGreater() {
        return 13 <= vmVersion;
    }

    public static boolean is14VMOrGreater() {
        return 14 <= vmVersion;
    }

    public static boolean is15VMOrGreater() {
        return 15 <= vmVersion;
    }

    public static boolean is16VMOrGreater() {
        return 16 <= vmVersion;
    }

    public static boolean is17VMOrGreater() {
        return 17 <= vmVersion;
    }

    public static boolean is18VMOrGreater() {
        return 18 <= vmVersion;
    }

    public static boolean is19VMOrGreater() {
        return 19 <= vmVersion;
    }

    public static final void throwIaxIfNull(final Object o, final String name) {
        if (null == o) {
            String message = "null " + (null == name ? "input" : name);
            throw new IllegalArgumentException(message);
        }
    }

    public static final void throwIaxIfNotAssignable(final Object[] ra, final Class<?> c, final String name) {
        throwIaxIfNull(ra, name);
        String label = (null == name ? "input" : name);
        for (int i = 0; i < ra.length; i++) {
            if (null == ra[i]) {
                String m = " null " + label + "[" + i + "]";
                throw new IllegalArgumentException(m);
            } else if (null != c) {
                Class<?> actualClass = ra[i].getClass();
                if (!c.isAssignableFrom(actualClass)) {
                    String message = label + " not assignable to " + c.getName();
                    throw new IllegalArgumentException(message);
                }
            }
        }
    }

    public static final void throwIaxIfNotAssignable(final Object o, final Class<?> c, final String name) {
        throwIaxIfNull(o, name);
        if (null != c) {
            Class<?> actualClass = o.getClass();
            if (!c.isAssignableFrom(actualClass)) {
                String message = name + " not assignable to " + c.getName();
                throw new IllegalArgumentException(message);
            }
        }
    }

    public static final void throwIaxIfFalse(final boolean test, final String message) {
        if (!test) {
            throw new IllegalArgumentException(message);
        }
    }

    public static boolean isEmpty(String s) {
        return ((null == s) || (0 == s.length()));
    }

    public static boolean isEmpty(Object[] ra) {
        return ((null == ra) || (0 == ra.length));
    }

    public static boolean isEmpty(byte[] ra) {
        return ((null == ra) || (0 == ra.length));
    }

    public static boolean isEmpty(Collection<?> collection) {
        return ((null == collection) || (0 == collection.size()));
    }

    public static boolean isEmpty(Map<?, ?> map) {
        return ((null == map) || (0 == map.size()));
    }

    public static String[] split(String text) {
        return strings(text).toArray(new String[0]);
    }

    public static List<String> commaSplit(String input) {
        return anySplit(input, ",");
    }

    public static String[] splitClasspath(String classpath) {
        if (LangUtil.isEmpty(classpath)) {
            return new String[0];
        }
        StringTokenizer st = new StringTokenizer(classpath, File.pathSeparator);
        ArrayList<String> result = new ArrayList<>(st.countTokens());
        while (st.hasMoreTokens()) {
            String entry = st.nextToken();
            if (!LangUtil.isEmpty(entry)) {
                result.add(entry);
            }
        }
        return result.toArray(new String[0]);
    }

    public static boolean getBoolean(String propertyName, boolean defaultValue) {
        if (null != propertyName) {
            try {
                String value = System.getProperty(propertyName);
                if (null != value) {
                    return Boolean.parseBoolean(value);
                }
            } catch (Throwable t) {
            }
        }
        return defaultValue;
    }

    public static List<String> anySplit(String input, String delim) {
        if (null == input) {
            return Collections.emptyList();
        }
        List<String> result = new ArrayList<>();
        if (LangUtil.isEmpty(delim) || (!input.contains(delim))) {
            result.add(input.trim());
        } else {
            StringTokenizer st = new StringTokenizer(input, delim);
            while (st.hasMoreTokens()) {
                result.add(st.nextToken().trim());
            }
        }
        return result;
    }

    public static List<String> strings(String text) {
        if (LangUtil.isEmpty(text)) {
            return Collections.emptyList();
        }
        List<String> strings = new ArrayList<>();
        StringTokenizer tok = new StringTokenizer(text);
        while (tok.hasMoreTokens()) {
            strings.add(tok.nextToken());
        }
        return strings;
    }

    public static <T> List<T> safeList(List<T> list) {
        return (null == list ? Collections.<T>emptyList() : Collections.unmodifiableList(list));
    }

    public static String[][] copyStrings(String[][] in) {
        String[][] out = new String[in.length][];
        for (int i = 0; i < out.length; i++) {
            out[i] = new String[in[i].length];
            System.arraycopy(in[i], 0, out[i], 0, out[i].length);
        }
        return out;
    }

    public static String[] extractOptions(String[] args, String[][] options) {
        if (LangUtil.isEmpty(args) || LangUtil.isEmpty(options)) {
            return args;
        }
        BitSet foundSet = new BitSet();
        String[] result = new String[args.length];
        int resultIndex = 0;
        for (int j = 0; j < args.length; j++) {
            boolean found = false;
            for (int i = 0; !found && (i < options.length); i++) {
                String[] option = options[i];
                LangUtil.throwIaxIfFalse(!LangUtil.isEmpty(option), "options");
                String sought = option[0];
                found = sought.equals(args[j]);
                if (found) {
                    foundSet.set(i);
                    int doMore = option.length - 1;
                    if (0 < doMore) {
                        final int MAX = j + doMore;
                        if (MAX >= args.length) {
                            String s = "expecting " + doMore + " args after ";
                            throw new IllegalArgumentException(s + args[j]);
                        }
                        for (int k = 1; k < option.length; k++) {
                            option[k] = args[++j];
                        }
                    }
                }
            }
            if (!found) {
                result[resultIndex++] = args[j];
            }
        }
        for (int i = 0; i < options.length; i++) {
            if (!foundSet.get(i)) {
                options[i][0] = null;
            }
        }
        if (resultIndex < args.length) {
            String[] temp = new String[resultIndex];
            System.arraycopy(result, 0, temp, 0, resultIndex);
            args = temp;
        }
        return args;
    }

    public static Object[] safeCopy(Object[] source, Object[] sink) {
        final Class<?> sinkType = (null == sink ? Object.class : sink.getClass().getComponentType());
        final int sourceLength = (null == source ? 0 : source.length);
        final int sinkLength = (null == sink ? 0 : sink.length);
        final int resultSize;
        List<Object> result = null;
        if (0 == sourceLength) {
            resultSize = 0;
        } else {
            result = new ArrayList<>(sourceLength);
            for (int i = 0; i < sourceLength; i++) {
                if ((null != source[i]) && (sinkType.isAssignableFrom(source[i].getClass()))) {
                    result.add(source[i]);
                }
            }
            resultSize = result.size();
        }
        if (resultSize != sinkLength) {
            sink = (Object[]) Array.newInstance(sinkType, result.size());
        }
        if (0 < resultSize) {
            sink = result.toArray(sink);
        }
        return sink;
    }

    public static String unqualifiedClassName(Class<?> c) {
        if (null == c) {
            return "null";
        }
        String name = c.getName();
        int loc = name.lastIndexOf(".");
        if (-1 != loc) {
            name = name.substring(1 + loc);
        }
        return name;
    }

    public static String unqualifiedClassName(Object o) {
        return LangUtil.unqualifiedClassName(null == o ? null : o.getClass());
    }

    public static String replace(String in, String sought, String replace) {
        if (LangUtil.isEmpty(in) || LangUtil.isEmpty(sought)) {
            return in;
        }
        StringBuilder result = new StringBuilder();
        final int len = sought.length();
        int start = 0;
        int loc;
        while (-1 != (loc = in.indexOf(sought, start))) {
            result.append(in.substring(start, loc));
            if (!LangUtil.isEmpty(replace)) {
                result.append(replace);
            }
            start = loc + len;
        }
        result.append(in.substring(start));
        return result.toString();
    }

    public static String toSizedString(long i, int width) {
        String result = "" + i;
        int size = result.length();
        if (width > size) {
            final String pad = "                                              ";
            final int padLength = pad.length();
            if (width > padLength) {
                width = padLength;
            }
            int topad = width - size;
            result = pad.substring(0, topad) + result;
        }
        return result;
    }

    public static String renderExceptionShort(Throwable e) {
        if (null == e) {
            return "(Throwable) null";
        }
        return "(" + LangUtil.unqualifiedClassName(e) + ") " + e.getMessage();
    }

    public static String renderException(Throwable t) {
        return renderException(t, true);
    }

    public static String renderException(Throwable t, boolean elide) {
        if (null == t) {
            return "null throwable";
        }
        t = unwrapException(t);
        StringBuffer stack = stackToString(t, false);
        if (elide) {
            elideEndingLines(StringChecker.TEST_PACKAGES, stack, 100);
        }
        return stack.toString();
    }

    static void elideEndingLines(StringChecker checker, StringBuffer stack, int maxLines) {
        if (null == checker || (null == stack) || (0 == stack.length())) {
            return;
        }
        final LinkedList<String> lines = new LinkedList<>();
        StringTokenizer st = new StringTokenizer(stack.toString(), "\n\r");
        while (st.hasMoreTokens() && (0 < --maxLines)) {
            lines.add(st.nextToken());
        }
        st = null;
        String line;
        int elided = 0;
        while (!lines.isEmpty()) {
            line = lines.getLast();
            if (!checker.acceptString(line)) {
                break;
            } else {
                elided++;
                lines.removeLast();
            }
        }
        if ((elided > 0) || (maxLines < 1)) {
            final int EOL_LEN = EOL.length();
            int totalLength = 0;
            while (!lines.isEmpty()) {
                totalLength += EOL_LEN + lines.getFirst().length();
                lines.removeFirst();
            }
            if (stack.length() > totalLength) {
                stack.setLength(totalLength);
                if (elided > 0) {
                    stack.append("    (... " + elided + " lines...)");
                }
            }
        }
    }

    public static StringBuffer stackToString(Throwable throwable, boolean skipMessage) {
        if (null == throwable) {
            return new StringBuffer();
        }
        StringWriter buf = new StringWriter();
        PrintWriter writer = new PrintWriter(buf);
        if (!skipMessage) {
            writer.println(throwable.getMessage());
        }
        throwable.printStackTrace(writer);
        try {
            buf.close();
        } catch (IOException ioe) {
        }
        return buf.getBuffer();
    }

    public static Throwable unwrapException(Throwable t) {
        Throwable current = t;
        Throwable next = null;
        while (current != null) {
            if (current instanceof InvocationTargetException) {
                next = ((InvocationTargetException) current).getTargetException();
            } else if (current instanceof ClassNotFoundException) {
                next = ((ClassNotFoundException) current).getException();
            } else if (current instanceof ExceptionInInitializerError) {
                next = ((ExceptionInInitializerError) current).getException();
            } else if (current instanceof PrivilegedActionException) {
                next = ((PrivilegedActionException) current).getException();
            } else if (current instanceof SQLException) {
                next = ((SQLException) current).getNextException();
            }
            if (null == next) {
                break;
            } else {
                current = next;
                next = null;
            }
        }
        return current;
    }

    public static <T> List<T> arrayAsList(T[] array) {
        if ((null == array) || (1 > array.length)) {
            return Collections.emptyList();
        }
        List<T> list = new ArrayList<>(Arrays.asList(array));
        return list;
    }

    public static class StringChecker {

        static StringChecker TEST_PACKAGES = new StringChecker(new String[]{"org.aspectj.testing", "org.eclipse.jdt.internal.junit", "junit.framework.", "org.apache.tools.ant.taskdefs.optional.junit.JUnitTestRunner"});

        String[] infixes;

        StringChecker(String[] infixes) {
            this.infixes = infixes;
        }

        public boolean acceptString(String input) {
            boolean result = false;
            if (!LangUtil.isEmpty(input)) {
                for (int i = 0; !result && (i < infixes.length); i++) {
                    result = (input.contains(infixes[i]));
                }
            }
            return result;
        }
    }

    public static String // XXX dumb implementation
    makeClasspath(String bootclasspath, String classpath, String classesDir, String outputJar) {
        StringBuffer sb = new StringBuffer();
        addIfNotEmpty(bootclasspath, sb, File.pathSeparator);
        addIfNotEmpty(classpath, sb, File.pathSeparator);
        if (!addIfNotEmpty(classesDir, sb, File.pathSeparator)) {
            addIfNotEmpty(outputJar, sb, File.pathSeparator);
        }
        return sb.toString();
    }

    private static boolean addIfNotEmpty(String input, StringBuffer sink, String delimiter) {
        if (LangUtil.isEmpty(input) || (null == sink)) {
            return false;
        }
        sink.append(input);
        if (!LangUtil.isEmpty(delimiter)) {
            sink.append(delimiter);
        }
        return true;
    }

    public static ProcessController makeProcess(ProcessController controller, String classpath, String mainClass, String[] args) {
        File java = LangUtil.getJavaExecutable();
        ArrayList<String> cmd = new ArrayList<>();
        cmd.add(java.getAbsolutePath());
        cmd.add("-classpath");
        cmd.add(classpath);
        cmd.add(mainClass);
        if (!LangUtil.isEmpty(args)) {
            cmd.addAll(Arrays.asList(args));
        }
        String[] command = cmd.toArray(new String[0]);
        if (null == controller) {
            controller = new ProcessController();
        }
        controller.init(command, mainClass);
        return controller;
    }

    public static File getJavaExecutable() {
        String javaHome = null;
        File result = null;
        try {
            javaHome = System.getProperty("java.home");
        } catch (Throwable t) {
        }
        if (null != javaHome) {
            File binDir = new File(javaHome, "bin");
            if (binDir.isDirectory() && binDir.canRead()) {
                String[] execs = new String[]{"java", "java.exe"};
                for (String exec : execs) {
                    result = new File(binDir, exec);
                    if (result.canRead()) {
                        break;
                    }
                }
            }
        }
        return result;
    }

    public static boolean sleepUntil(long time) {
        if (time == 0) {
            return true;
        } else if (time < 0) {
            throw new IllegalArgumentException("negative: " + time);
        }
        long curTime = System.currentTimeMillis();
        for (int i = 0; (i < 100) && (curTime < time); i++) {
            try {
                Thread.sleep(time - curTime);
            } catch (InterruptedException e) {
            }
            curTime = System.currentTimeMillis();
        }
        return (curTime >= time);
    }

    public static class ProcessController {

        private String[] command;

        private String[] envp;

        private String label;

        private boolean init;

        private boolean started;

        private boolean completed;

        private boolean userStopped;

        private Process process;

        private FileUtil.Pipe errStream;

        private FileUtil.Pipe outStream;

        private FileUtil.Pipe inStream;

        private ByteArrayOutputStream errSnoop;

        private ByteArrayOutputStream outSnoop;

        private int result;

        private Thrown thrown;

        public ProcessController() {
        }

        public final void reinit() {
            if (!init) {
                throw new IllegalStateException("must init(..) before reinit()");
            }
            if (started && !completed) {
                throw new IllegalStateException("not completed - do stop()");
            }
            started = false;
            completed = false;
            result = Integer.MIN_VALUE;
            thrown = null;
            process = null;
            errStream = null;
            outStream = null;
            inStream = null;
        }

        public final void init(String classpath, String mainClass, String[] args) {
            init(LangUtil.getJavaExecutable(), classpath, mainClass, args);
        }

        public final void init(File java, String classpath, String mainClass, String[] args) {
            LangUtil.throwIaxIfNull(java, "java");
            LangUtil.throwIaxIfNull(mainClass, "mainClass");
            LangUtil.throwIaxIfNull(args, "args");
            ArrayList<String> cmd = new ArrayList<>();
            cmd.add(java.getAbsolutePath());
            cmd.add("-classpath");
            cmd.add(classpath);
            cmd.add(mainClass);
            if (!LangUtil.isEmpty(args)) {
                cmd.addAll(Arrays.asList(args));
            }
            init(cmd.toArray(new String[0]), mainClass);
        }

        public final void init(String[] command, String label) {
            this.command = (String[]) LangUtil.safeCopy(command, new String[0]);
            if (1 > this.command.length) {
                throw new IllegalArgumentException("empty command");
            }
            this.label = LangUtil.isEmpty(label) ? command[0] : label;
            init = true;
            reinit();
        }

        public final void setEnvp(String[] envp) {
            this.envp = (String[]) LangUtil.safeCopy(envp, new String[0]);
            if (1 > this.envp.length) {
                throw new IllegalArgumentException("empty envp");
            }
        }

        public final void setErrSnoop(ByteArrayOutputStream snoop) {
            errSnoop = snoop;
            if (null != errStream) {
                errStream.setSnoop(errSnoop);
            }
        }

        public final void setOutSnoop(ByteArrayOutputStream snoop) {
            outSnoop = snoop;
            if (null != outStream) {
                outStream.setSnoop(outSnoop);
            }
        }

        public final Thread start() {
            if (!init) {
                throw new IllegalStateException("not initialized");
            }
            synchronized (this) {
                if (started) {
                    throw new IllegalStateException("already started");
                }
                started = true;
            }
            try {
                process = Runtime.getRuntime().exec(command);
            } catch (IOException e) {
                stop(e, Integer.MIN_VALUE);
                return null;
            }
            errStream = new FileUtil.Pipe(process.getErrorStream(), System.err);
            if (null != errSnoop) {
                errStream.setSnoop(errSnoop);
            }
            outStream = new FileUtil.Pipe(process.getInputStream(), System.out);
            if (null != outSnoop) {
                outStream.setSnoop(outSnoop);
            }
            inStream = new FileUtil.Pipe(System.in, process.getOutputStream());
            Runnable processRunner = new Runnable() {

                @Override
                public void run() {
                    Throwable thrown = null;
                    int result = Integer.MIN_VALUE;
                    try {
                        new Thread(errStream).start();
                        new Thread(outStream).start();
                        new Thread(inStream).start();
                        process.waitFor();
                        result = process.exitValue();
                    } catch (Throwable e) {
                        thrown = e;
                    } finally {
                        stop(thrown, result);
                    }
                }
            };
            Thread result = new Thread(processRunner, label);
            result.start();
            return result;
        }

        public final synchronized void stop() {
            if (completed) {
                return;
            }
            userStopped = true;
            stop(null, Integer.MIN_VALUE);
        }

        public final String[] getCommand() {
            String[] toCopy = command;
            if (LangUtil.isEmpty(toCopy)) {
                return new String[0];
            }
            String[] result = new String[toCopy.length];
            System.arraycopy(toCopy, 0, result, 0, result.length);
            return result;
        }

        public final boolean completed() {
            return completed;
        }

        public final boolean started() {
            return started;
        }

        public final boolean userStopped() {
            return userStopped;
        }

        public final Thrown getThrown() {
            return makeThrown(null);
        }

        public final int getResult() {
            return result;
        }

        protected void doCompleting(Thrown thrown, int result) {
        }

        private final synchronized void stop(Throwable thrown, int result) {
            if (completed) {
                throw new IllegalStateException("already completed");
            } else if (null != this.thrown) {
                throw new IllegalStateException("already set thrown: " + thrown);
            }
            this.thrown = makeThrown(thrown);
            if (null != process) {
                process.destroy();
            }
            if (null != inStream) {
                inStream.halt(false, true);
                inStream = null;
            }
            if (null != outStream) {
                outStream.halt(true, true);
                outStream = null;
            }
            if (null != errStream) {
                errStream.halt(true, true);
                errStream = null;
            }
            if (Integer.MIN_VALUE != result) {
                this.result = result;
            }
            completed = true;
            doCompleting(this.thrown, result);
        }

        private final synchronized Thrown makeThrown(Throwable processThrown) {
            if (null != thrown) {
                return thrown;
            }
            return new Thrown(processThrown, (null == outStream ? null : outStream.getThrown()), (null == errStream ? null : errStream.getThrown()), (null == inStream ? null : inStream.getThrown()));
        }

        public static class Thrown {

            public final Throwable fromProcess;

            public final Throwable fromErrPipe;

            public final Throwable fromOutPipe;

            public final Throwable fromInPipe;

            public final boolean thrown;

            private Thrown(Throwable fromProcess, Throwable fromOutPipe, Throwable fromErrPipe, Throwable fromInPipe) {
                this.fromProcess = fromProcess;
                this.fromErrPipe = fromErrPipe;
                this.fromOutPipe = fromOutPipe;
                this.fromInPipe = fromInPipe;
                thrown = ((null != fromProcess) || (null != fromInPipe) || (null != fromOutPipe) || (null != fromErrPipe));
            }

            @Override
            public String toString() {
                StringBuilder sb = new StringBuilder();
                append(sb, fromProcess, "process");
                append(sb, fromOutPipe, " stdout");
                append(sb, fromErrPipe, " stderr");
                append(sb, fromInPipe, "  stdin");
                if (0 == sb.length()) {
                    return "Thrown (none)";
                } else {
                    return sb.toString();
                }
            }

            private void append(StringBuilder sb, Throwable thrown, String label) {
                if (null != thrown) {
                    sb.append("from " + label + ": ");
                    sb.append(LangUtil.renderExceptionShort(thrown));
                    sb.append(LangUtil.EOL);
                }
            }
        }
    }

    public static String getJrtFsFilePath() {
        return getJavaHome() + File.separator + "lib" + File.separator + JRT_FS;
    }

    public static String getJavaHome() {
        return System.getProperty("java.home");
    }
}
