package org.aspectj.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

public class PartialOrder {

    public interface PartialComparable {

        int compareTo(Object other);

        int fallbackCompareTo(Object other);
    }

    private static class SortObject<T extends PartialComparable> {

        T object;

        List<SortObject<T>> smallerObjects = new LinkedList<>();

        List<SortObject<T>> biggerObjects = new LinkedList<>();

        public SortObject(T o) {
            object = o;
        }

        boolean hasNoSmallerObjects() {
            return smallerObjects.size() == 0;
        }

        boolean removeSmallerObject(SortObject<T> o) {
            smallerObjects.remove(o);
            return hasNoSmallerObjects();
        }

        void addDirectedLinks(SortObject<T> other) {
            int cmp = object.compareTo(other.object);
            if (cmp == 0) {
                return;
            }
            if (cmp > 0) {
                this.smallerObjects.add(other);
                other.biggerObjects.add(this);
            } else {
                this.biggerObjects.add(other);
                other.smallerObjects.add(this);
            }
        }

        public String toString() {
            return object.toString();
        }
    }

    private static <T extends PartialComparable> void addNewPartialComparable(List<SortObject<T>> graph, T o) {
        SortObject<T> so = new SortObject<>(o);
        for (SortObject<T> other : graph) {
            so.addDirectedLinks(other);
        }
        graph.add(so);
    }

    private static <T extends PartialComparable> void removeFromGraph(List<SortObject<T>> graph, SortObject<T> o) {
        for (Iterator<SortObject<T>> i = graph.iterator(); i.hasNext(); ) {
            SortObject<T> other = i.next();
            if (o == other) {
                i.remove();
            }
            other.removeSmallerObject(o);
        }
    }

    public static <T extends PartialComparable> List<T> sort(List<T> objects) {
        if (objects.size() < 2) {
            return objects;
        }
        List<SortObject<T>> sortList = new LinkedList<>();
        for (T object : objects) {
            addNewPartialComparable(sortList, object);
        }
        final int N = objects.size();
        for (int index = 0; index < N; index++) {
            SortObject<T> leastWithNoSmallers = null;
            for (SortObject<T> so : sortList) {
                if (so.hasNoSmallerObjects()) {
                    if (leastWithNoSmallers == null || so.object.fallbackCompareTo(leastWithNoSmallers.object) < 0) {
                        leastWithNoSmallers = so;
                    }
                }
            }
            if (leastWithNoSmallers == null) {
                return null;
            }
            removeFromGraph(sortList, leastWithNoSmallers);
            objects.set(index, leastWithNoSmallers.object);
        }
        return objects;
    }

    static class Token implements PartialComparable {

        private String s;

        Token(String s) {
            this.s = s;
        }

        public int compareTo(Object other) {
            Token t = (Token) other;
            int cmp = s.charAt(0) - t.s.charAt(0);
            if (cmp == 1) {
                return 1;
            }
            if (cmp == -1) {
                return -1;
            }
            return 0;
        }

        public int fallbackCompareTo(Object other) {
            return -s.compareTo(((Token) other).s);
        }

        public String toString() {
            return s;
        }
    }

    public static void main(String[] args) {
        List<Token> l = new ArrayList<>();
        l.add(new Token("a1"));
        l.add(new Token("c2"));
        l.add(new Token("b3"));
        l.add(new Token("f4"));
        l.add(new Token("e5"));
        l.add(new Token("d6"));
        l.add(new Token("c7"));
        l.add(new Token("b8"));
        l.add(new Token("z"));
        l.add(new Token("x"));
        l.add(new Token("f9"));
        l.add(new Token("e10"));
        l.add(new Token("a11"));
        l.add(new Token("d12"));
        l.add(new Token("b13"));
        l.add(new Token("c14"));
        System.out.println(l);
        sort(l);
        System.out.println(l);
    }
}
