package org.aspectj.weaver;

import org.aspectj.bridge.ISourceLocation;
import org.aspectj.weaver.patterns.DeclareErrorOrWarning;
import org.aspectj.weaver.patterns.PerClause;
import org.aspectj.weaver.patterns.Pointcut;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

public class Checker extends ShadowMunger {

    private boolean isError;

    private String message;

    private volatile int hashCode = -1;

    @SuppressWarnings("unused")
    private Checker() {
    }

    public Checker(DeclareErrorOrWarning deow) {
        super(deow.getPointcut(), deow.getStart(), deow.getEnd(), deow.getSourceContext(), ShadowMungerDeow);
        this.message = deow.getMessage();
        this.isError = deow.isError();
    }

    private Checker(Pointcut pointcut, int start, int end, ISourceContext context, String message, boolean isError) {
        super(pointcut, start, end, context, ShadowMungerDeow);
        this.message = message;
        this.isError = isError;
    }

    public boolean isError() {
        return isError;
    }

    public String getMessage(Shadow shadow) {
        return format(this.message, shadow);
    }

    @Override
    public void specializeOn(Shadow shadow) {
        throw new IllegalStateException("Cannot call specializeOn(...) for a Checker");
    }

    @Override
    public boolean implementOn(Shadow shadow) {
        throw new IllegalStateException("Cannot call implementOn(...) for a Checker");
    }

    @Override
    public boolean match(Shadow shadow, World world) {
        if (super.match(shadow, world)) {
            world.reportCheckerMatch(this, shadow);
        }
        return false;
    }

    public int compareTo(Object other) {
        return 0;
    }

    @Override
    public boolean mustCheckExceptions() {
        return true;
    }

    @Override
    public Collection<ResolvedType> getThrownExceptions() {
        return Collections.emptyList();
    }

    @Override
    public boolean equals(Object other) {
        if (!(other instanceof Checker)) {
            return false;
        }
        Checker o = (Checker) other;
        return o.isError == isError && ((o.pointcut == null) ? (pointcut == null) : o.pointcut.equals(pointcut));
    }

    @Override
    public int hashCode() {
        if (hashCode == -1) {
            int result = 17;
            result = 37 * result + (isError ? 1 : 0);
            result = 37 * result + ((pointcut == null) ? 0 : pointcut.hashCode());
            hashCode = result;
        }
        return hashCode;
    }

    @Override
    public ShadowMunger parameterizeWith(ResolvedType declaringType, Map<String, UnresolvedType> typeVariableMap) {
        Checker ret = new Checker(this.pointcut.parameterizeWith(typeVariableMap, declaringType.getWorld()), this.start, this.end, this.sourceContext, this.message, this.isError);
        return ret;
    }

    @Override
    public ShadowMunger concretize(ResolvedType theAspect, World world, PerClause clause) {
        this.pointcut = this.pointcut.concretize(theAspect, getDeclaringType(), 0, this);
        this.hashCode = -1;
        return this;
    }

    @Override
    public ResolvedType getConcreteAspect() {
        return getDeclaringType();
    }

    private int nextCurly(String string, int pos) {
        do {
            int curlyIndex = string.indexOf('{', pos);
            if (curlyIndex == -1) {
                return -1;
            }
            if (curlyIndex == 0) {
                return 0;
            }
            if (string.charAt(curlyIndex - 1) != '\\') {
                return curlyIndex;
            }
            pos = curlyIndex + 1;
        } while (pos < string.length());
        return -1;
    }

    private String format(String msg, Shadow shadow) {
        int pos = 0;
        int curlyIndex = nextCurly(msg, 0);
        if (curlyIndex == -1) {
            if (msg.indexOf('{') != -1) {
                return msg.replace("\\{", "{");
            } else {
                return msg;
            }
        }
        StringBuilder ret = new StringBuilder();
        while (curlyIndex >= 0) {
            if (curlyIndex > 0) {
                ret.append(msg.substring(pos, curlyIndex).replace("\\{", "{"));
            }
            int endCurly = msg.indexOf('}', curlyIndex);
            if (endCurly == -1) {
                ret.append('{');
                pos = curlyIndex + 1;
            } else {
                ret.append(getValue(msg.substring(curlyIndex + 1, endCurly), shadow));
            }
            pos = endCurly + 1;
            curlyIndex = nextCurly(msg, pos);
        }
        ret.append(msg.substring(pos, msg.length()));
        return ret.toString();
    }

    private String getValue(String key, Shadow shadow) {
        if (key.equalsIgnoreCase("joinpoint")) {
            return shadow.toString();
        } else if (key.equalsIgnoreCase("joinpoint.kind")) {
            return shadow.getKind().getName();
        } else if (key.equalsIgnoreCase("joinpoint.enclosingclass")) {
            return shadow.getEnclosingType().getName();
        } else if (key.equalsIgnoreCase("joinpoint.enclosingmember.name")) {
            Member member = shadow.getEnclosingCodeSignature();
            if (member == null) {
                return "";
            } else {
                return member.getName();
            }
        } else if (key.equalsIgnoreCase("joinpoint.enclosingmember")) {
            Member member = shadow.getEnclosingCodeSignature();
            if (member == null) {
                return "";
            } else {
                return member.toString();
            }
        } else if (key.equalsIgnoreCase("joinpoint.signature")) {
            return shadow.getSignature().toString();
        } else if (key.equalsIgnoreCase("joinpoint.signature.declaringtype")) {
            return shadow.getSignature().getDeclaringType().toString();
        } else if (key.equalsIgnoreCase("joinpoint.signature.name")) {
            return shadow.getSignature().getName();
        } else if (key.equalsIgnoreCase("joinpoint.sourcelocation.sourcefile")) {
            ISourceLocation loc = shadow.getSourceLocation();
            if ((loc != null) && (loc.getSourceFile() != null)) {
                return loc.getSourceFile().toString();
            } else {
                return "UNKNOWN";
            }
        } else if (key.equalsIgnoreCase("joinpoint.sourcelocation.line")) {
            ISourceLocation loc = shadow.getSourceLocation();
            if (loc != null) {
                return Integer.toString(loc.getLine());
            } else {
                return "-1";
            }
        } else if (key.equalsIgnoreCase("advice.aspecttype")) {
            return getDeclaringType().getName();
        } else if (key.equalsIgnoreCase("advice.sourcelocation.line")) {
            ISourceLocation loc = getSourceLocation();
            if ((loc != null) && (loc.getSourceFile() != null)) {
                return Integer.toString(loc.getLine());
            } else {
                return "-1";
            }
        } else if (key.equalsIgnoreCase("advice.sourcelocation.sourcefile")) {
            ISourceLocation loc = getSourceLocation();
            if ((loc != null) && (loc.getSourceFile() != null)) {
                return loc.getSourceFile().toString();
            } else {
                return "UNKNOWN";
            }
        } else {
            return "UNKNOWN_KEY{" + key + "}";
        }
    }
}
