package org.aspectj.weaver;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.weaver.patterns.Declare;
import org.aspectj.weaver.patterns.DeclareAnnotation;
import org.aspectj.weaver.patterns.DeclareParents;
import org.aspectj.weaver.patterns.DeclareSoft;
import org.aspectj.weaver.patterns.DeclareTypeErrorOrWarning;
import org.aspectj.weaver.patterns.IVerificationRequired;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class CrosscuttingMembersSet {

    private transient World world;

    private final Map<ResolvedType, CrosscuttingMembers> members = new HashMap<>();

    private transient List<IVerificationRequired> verificationList = null;

    private List<ShadowMunger> shadowMungers = null;

    private List<ConcreteTypeMunger> typeMungers = null;

    private List<ConcreteTypeMunger> lateTypeMungers = null;

    private List<DeclareSoft> declareSofts = null;

    private List<DeclareParents> declareParents = null;

    private List<DeclareAnnotation> declareAnnotationOnTypes = null;

    private List<DeclareAnnotation> declareAnnotationOnFields = null;

    private List<DeclareAnnotation> declareAnnotationOnMethods = null;

    private List<DeclareTypeErrorOrWarning> declareTypeEows = null;

    private List<Declare> declareDominates = null;

    private boolean changedSinceLastReset = false;

    public CrosscuttingMembersSet(World world) {
        this.world = world;
    }

    public boolean addOrReplaceAspect(ResolvedType aspectType) {
        return addOrReplaceAspect(aspectType, true);
    }

    private boolean excludeDueToParentAspectHavingUnresolvedDependency(ResolvedType aspectType) {
        ResolvedType parent = aspectType.getSuperclass();
        boolean excludeDueToParent = false;
        while (parent != null) {
            if (parent.isAspect() && parent.isAbstract() && world.hasUnsatisfiedDependency(parent)) {
                if (!world.getMessageHandler().isIgnoring(IMessage.INFO)) {
                    world.getMessageHandler().handleMessage(MessageUtil.info("deactivating aspect '" + aspectType.getName() + "' as the parent aspect '" + parent.getName() + "' has unsatisfied dependencies"));
                }
                excludeDueToParent = true;
            }
            parent = parent.getSuperclass();
        }
        return excludeDueToParent;
    }

    public boolean addOrReplaceAspect(ResolvedType aspectType, boolean inWeavingPhase) {
        if (!world.isAspectIncluded(aspectType)) {
            return false;
        }
        if (world.hasUnsatisfiedDependency(aspectType)) {
            return false;
        }
        if (excludeDueToParentAspectHavingUnresolvedDependency(aspectType)) {
            return false;
        }
        boolean change = false;
        CrosscuttingMembers xcut = members.get(aspectType);
        if (xcut == null) {
            members.put(aspectType, aspectType.collectCrosscuttingMembers(inWeavingPhase));
            clearCaches();
            change = true;
        } else {
            if (xcut.replaceWith(aspectType.collectCrosscuttingMembers(inWeavingPhase), inWeavingPhase)) {
                clearCaches();
                change = true;
            } else {
                if (inWeavingPhase) {
                    shadowMungers = null;
                }
                change = false;
            }
        }
        if (aspectType.isAbstract()) {
            boolean ancestorChange = addOrReplaceDescendantsOf(aspectType, inWeavingPhase);
            change = change || ancestorChange;
        }
        changedSinceLastReset = changedSinceLastReset || change;
        return change;
    }

    private boolean addOrReplaceDescendantsOf(ResolvedType aspectType, boolean inWeavePhase) {
        Set<ResolvedType> knownAspects = members.keySet();
        Set<ResolvedType> toBeReplaced = new HashSet<>();
        for (ResolvedType candidateDescendant : knownAspects) {
            if ((candidateDescendant != aspectType) && (aspectType.isAssignableFrom(candidateDescendant, true))) {
                toBeReplaced.add(candidateDescendant);
            }
        }
        boolean change = false;
        for (ResolvedType next : toBeReplaced) {
            boolean thisChange = addOrReplaceAspect(next, inWeavePhase);
            change = change || thisChange;
        }
        return change;
    }

    public void addAdviceLikeDeclares(ResolvedType aspectType) {
        if (!members.containsKey(aspectType)) {
            return;
        }
        CrosscuttingMembers xcut = members.get(aspectType);
        xcut.addDeclares(aspectType.collectDeclares(true));
    }

    public boolean deleteAspect(UnresolvedType aspectType) {
        boolean isAspect = members.remove(aspectType) != null;
        clearCaches();
        return isAspect;
    }

    public boolean containsAspect(UnresolvedType aspectType) {
        return members.containsKey(aspectType);
    }

    public void addFixedCrosscuttingMembers(ResolvedType aspectType) {
        members.put(aspectType, aspectType.crosscuttingMembers);
        clearCaches();
    }

    private void clearCaches() {
        shadowMungers = null;
        typeMungers = null;
        lateTypeMungers = null;
        declareSofts = null;
        declareParents = null;
        declareAnnotationOnFields = null;
        declareAnnotationOnMethods = null;
        declareAnnotationOnTypes = null;
        declareDominates = null;
    }

    public List<ShadowMunger> getShadowMungers() {
        if (shadowMungers == null) {
            List<ShadowMunger> ret = new ArrayList<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getShadowMungers());
            }
            shadowMungers = ret;
        }
        return shadowMungers;
    }

    public List<ConcreteTypeMunger> getTypeMungers() {
        if (typeMungers == null) {
            List<ConcreteTypeMunger> ret = new ArrayList<>();
            for (CrosscuttingMembers xmembers : members.values()) {
                for (ConcreteTypeMunger mungerToAdd : xmembers.getTypeMungers()) {
                    ResolvedTypeMunger resolvedMungerToAdd = mungerToAdd.getMunger();
                    if (isNewStylePrivilegedAccessMunger(resolvedMungerToAdd)) {
                        String newFieldName = resolvedMungerToAdd.getSignature().getName();
                        boolean alreadyExists = false;
                        for (ConcreteTypeMunger existingMunger : ret) {
                            ResolvedTypeMunger existing = existingMunger.getMunger();
                            if (isNewStylePrivilegedAccessMunger(existing)) {
                                String existingFieldName = existing.getSignature().getName();
                                if (existingFieldName.equals(newFieldName) && existing.getSignature().getDeclaringType().equals(resolvedMungerToAdd.getSignature().getDeclaringType())) {
                                    alreadyExists = true;
                                    break;
                                }
                            }
                        }
                        if (!alreadyExists) {
                            ret.add(mungerToAdd);
                        }
                    } else {
                        ret.add(mungerToAdd);
                    }
                }
            }
            typeMungers = ret;
        }
        return typeMungers;
    }

    public List<ConcreteTypeMunger> getTypeMungersOfKind(ResolvedTypeMunger.Kind kind) {
        List<ConcreteTypeMunger> collected = null;
        for (ConcreteTypeMunger typeMunger : typeMungers) {
            if (typeMunger.getMunger() != null && typeMunger.getMunger().getKind() == kind) {
                if (collected == null) {
                    collected = new ArrayList<>();
                }
                collected.add(typeMunger);
            }
        }
        if (collected == null) {
            return Collections.emptyList();
        } else {
            return collected;
        }
    }

    private boolean isNewStylePrivilegedAccessMunger(ResolvedTypeMunger typeMunger) {
        boolean b = (typeMunger != null && typeMunger.getKind() == ResolvedTypeMunger.PrivilegedAccess && typeMunger.getSignature().getKind() == Member.FIELD);
        if (!b) {
            return b;
        }
        PrivilegedAccessMunger privAccessMunger = (PrivilegedAccessMunger) typeMunger;
        return privAccessMunger.shortSyntax;
    }

    public List<ConcreteTypeMunger> getLateTypeMungers() {
        if (lateTypeMungers == null) {
            List<ConcreteTypeMunger> ret = new ArrayList<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getLateTypeMungers());
            }
            lateTypeMungers = ret;
        }
        return lateTypeMungers;
    }

    public List<DeclareSoft> getDeclareSofts() {
        if (declareSofts == null) {
            Set<DeclareSoft> ret = new HashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareSofts());
            }
            declareSofts = new ArrayList<>();
            declareSofts.addAll(ret);
        }
        return declareSofts;
    }

    public List<DeclareParents> getDeclareParents() {
        if (declareParents == null) {
            Set<DeclareParents> ret = new HashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareParents());
            }
            declareParents = new ArrayList<>();
            declareParents.addAll(ret);
        }
        return declareParents;
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnTypes() {
        if (declareAnnotationOnTypes == null) {
            Set<DeclareAnnotation> ret = new LinkedHashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareAnnotationOnTypes());
            }
            declareAnnotationOnTypes = new ArrayList<>();
            declareAnnotationOnTypes.addAll(ret);
        }
        return declareAnnotationOnTypes;
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnFields() {
        if (declareAnnotationOnFields == null) {
            Set<DeclareAnnotation> ret = new LinkedHashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareAnnotationOnFields());
            }
            declareAnnotationOnFields = new ArrayList<>();
            declareAnnotationOnFields.addAll(ret);
        }
        return declareAnnotationOnFields;
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnMethods() {
        if (declareAnnotationOnMethods == null) {
            Set<DeclareAnnotation> ret = new LinkedHashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareAnnotationOnMethods());
            }
            declareAnnotationOnMethods = new ArrayList<>();
            declareAnnotationOnMethods.addAll(ret);
        }
        return declareAnnotationOnMethods;
    }

    public List<DeclareTypeErrorOrWarning> getDeclareTypeEows() {
        if (declareTypeEows == null) {
            Set<DeclareTypeErrorOrWarning> ret = new HashSet<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareTypeErrorOrWarning());
            }
            declareTypeEows = new ArrayList<>();
            declareTypeEows.addAll(ret);
        }
        return declareTypeEows;
    }

    public List<Declare> getDeclareDominates() {
        if (declareDominates == null) {
            List<Declare> ret = new ArrayList<>();
            for (CrosscuttingMembers crosscuttingMembers : members.values()) {
                ret.addAll(crosscuttingMembers.getDeclareDominates());
            }
            declareDominates = ret;
        }
        return declareDominates;
    }

    public ResolvedType findAspectDeclaringParents(DeclareParents p) {
        Set<ResolvedType> keys = this.members.keySet();
        for (ResolvedType element : keys) {
            for (DeclareParents dp : members.get(element).getDeclareParents()) {
                if (dp.equals(p)) {
                    return element;
                }
            }
        }
        return null;
    }

    public void reset() {
        verificationList = null;
        changedSinceLastReset = false;
    }

    public boolean hasChangedSinceLastReset() {
        return changedSinceLastReset;
    }

    public void recordNecessaryCheck(IVerificationRequired verification) {
        if (verificationList == null) {
            verificationList = new ArrayList<>();
        }
        verificationList.add(verification);
    }

    public void verify() {
        if (verificationList == null) {
            return;
        }
        for (IVerificationRequired element : verificationList) {
            element.verify();
        }
        verificationList = null;
    }

    public int serializationVersion = 1;

    public void write(CompressingDataOutputStream stream) throws IOException {
        stream.writeInt(shadowMungers.size());
        for (ShadowMunger shadowMunger : shadowMungers) {
            shadowMunger.write(stream);
        }
    }
}
