package org.aspectj.weaver;

import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.context.CompilationAndWeavingContext;

import java.util.Collections;
import java.util.List;

public class MissingResolvedTypeWithKnownSignature extends ResolvedType {

    private boolean issuedCantFindTypeError = false;

    private boolean issuedJoinPointWarning = false;

    private boolean issuedMissingInterfaceWarning = false;

    public MissingResolvedTypeWithKnownSignature(String signature, World world) {
        super(signature, world);
    }

    @Override
    public boolean isMissing() {
        return true;
    }

    public MissingResolvedTypeWithKnownSignature(String signature, String signatureErasure, World world) {
        super(signature, signatureErasure, world);
    }

    @Override
    public ResolvedMember[] getDeclaredFields() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_FIELDS);
        return ResolvedMember.NONE;
    }

    @Override
    public ResolvedMember[] getDeclaredMethods() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_METHODS);
        return ResolvedMember.NONE;
    }

    @Override
    public AnnotationAJ[] getAnnotations() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_ANNOTATION);
        return AnnotationAJ.EMPTY_ARRAY;
    }

    @Override
    public ResolvedType[] getDeclaredInterfaces() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_INTERFACES);
        return ResolvedType.EMPTY_RESOLVED_TYPE_ARRAY;
    }

    @Override
    public ResolvedMember[] getDeclaredPointcuts() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_POINTCUTS);
        return ResolvedMember.NONE;
    }

    @Override
    public ResolvedType getSuperclass() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_SUPERCLASS);
        return ResolvedType.MISSING;
    }

    @Override
    public int getModifiers() {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_MODIFIERS);
        return 0;
    }

    @Override
    public ISourceContext getSourceContext() {
        return new ISourceContext() {

            @Override
            public ISourceLocation makeSourceLocation(IHasPosition position) {
                return null;
            }

            @Override
            public ISourceLocation makeSourceLocation(int line, int offset) {
                return null;
            }

            @Override
            public int getOffset() {
                return 0;
            }

            @Override
            public void tidy() {
            }
        };
    }

    @Override
    public boolean isAssignableFrom(ResolvedType other) {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_ASSIGNABLE, other.getName());
        return false;
    }

    @Override
    public boolean isAssignableFrom(ResolvedType other, boolean allowMissing) {
        if (allowMissing) {
            return false;
        } else {
            return isAssignableFrom(other);
        }
    }

    @Override
    public boolean isCoerceableFrom(ResolvedType other) {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_COERCEABLE, other.getName());
        return false;
    }

    @Override
    public boolean hasAnnotation(UnresolvedType ofType) {
        raiseCantFindType(WeaverMessages.CANT_FIND_TYPE_ANNOTATION);
        return false;
    }

    @Override
    public List getInterTypeMungers() {
        return Collections.EMPTY_LIST;
    }

    @Override
    public List getInterTypeMungersIncludingSupers() {
        return Collections.EMPTY_LIST;
    }

    @Override
    public List getInterTypeParentMungers() {
        return Collections.EMPTY_LIST;
    }

    @Override
    public List getInterTypeParentMungersIncludingSupers() {
        return Collections.EMPTY_LIST;
    }

    @Override
    protected void collectInterTypeMungers(List collector) {
        return;
    }

    public void raiseWarningOnJoinPointSignature(String signature) {
        if (issuedJoinPointWarning) {
            return;
        }
        String message = WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE_JOINPOINT, getName(), signature);
        message += "\n" + CompilationAndWeavingContext.getCurrentContext();
        world.getLint().cantFindTypeAffectingJoinPointMatch.signal(message, null);
        issuedJoinPointWarning = true;
    }

    public void raiseWarningOnMissingInterfaceWhilstFindingMethods() {
        if (issuedMissingInterfaceWarning) {
            return;
        }
        String message = WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE_INTERFACE_METHODS, getName(), signature);
        message += "\n" + CompilationAndWeavingContext.getCurrentContext();
        world.getLint().cantFindTypeAffectingJoinPointMatch.signal(message, null);
        issuedMissingInterfaceWarning = true;
    }

    private void raiseCantFindType(String key) {
        if (!world.getLint().cantFindType.isEnabled()) {
            return;
        }
        if (issuedCantFindTypeError) {
            return;
        }
        String message = WeaverMessages.format(key, getName());
        message += "\n" + CompilationAndWeavingContext.getCurrentContext();
        world.getLint().cantFindType.signal(message, null);
        issuedCantFindTypeError = true;
    }

    private void raiseCantFindType(String key, String insert) {
        if (issuedCantFindTypeError) {
            return;
        }
        String message = WeaverMessages.format(key, getName(), insert);
        message += "\n" + CompilationAndWeavingContext.getCurrentContext();
        world.getLint().cantFindType.signal(message, null);
        issuedCantFindTypeError = true;
    }
}
