package org.aspectj.weaver;

import org.aspectj.bridge.ISourceLocation;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ResolvedMemberImpl extends MemberImpl implements IHasPosition, ResolvedMember {

    private String[] parameterNames = null;

    private boolean isResolved = false;

    protected UnresolvedType[] checkedExceptions = UnresolvedType.NONE;

    protected ResolvedMember backingGenericMember = null;

    protected AnnotationAJ[] annotations = null;

    protected ResolvedType[] annotationTypes = null;

    protected AnnotationAJ[][] parameterAnnotations = null;

    protected ResolvedType[][] parameterAnnotationTypes = null;

    private boolean isAnnotatedElsewhere = false;

    private boolean isAjSynthetic = false;

    protected TypeVariable[] typeVariables;

    protected int start, end;

    protected ISourceContext sourceContext = null;

    public ResolvedMemberImpl(MemberKind kind, UnresolvedType declaringType, int modifiers, UnresolvedType returnType, String name, UnresolvedType[] parameterTypes) {
        super(kind, declaringType, modifiers, returnType, name, parameterTypes);
    }

    public ResolvedMemberImpl(MemberKind kind, UnresolvedType declaringType, int modifiers, UnresolvedType returnType, String name, UnresolvedType[] parameterTypes, UnresolvedType[] checkedExceptions) {
        super(kind, declaringType, modifiers, returnType, name, parameterTypes);
        this.checkedExceptions = checkedExceptions;
    }

    public ResolvedMemberImpl(MemberKind kind, UnresolvedType declaringType, int modifiers, UnresolvedType returnType, String name, UnresolvedType[] parameterTypes, UnresolvedType[] checkedExceptions, ResolvedMember backingGenericMember) {
        this(kind, declaringType, modifiers, returnType, name, parameterTypes, checkedExceptions);
        this.backingGenericMember = backingGenericMember;
        this.isAjSynthetic = backingGenericMember.isAjSynthetic();
    }

    public ResolvedMemberImpl(MemberKind kind, UnresolvedType declaringType, int modifiers, String name, String signature) {
        super(kind, declaringType, modifiers, name, signature);
    }

    public static JoinPointSignature[] getJoinPointSignatures(Member joinPointSignature, World inAWorld) {
        ResolvedType originalDeclaringType = joinPointSignature.getDeclaringType().resolve(inAWorld);
        ResolvedMemberImpl firstDefiningMember = (ResolvedMemberImpl) joinPointSignature.resolve(inAWorld);
        if (firstDefiningMember == null) {
            return JoinPointSignature.EMPTY_ARRAY;
        }
        ResolvedType firstDefiningType = firstDefiningMember.getDeclaringType().resolve(inAWorld);
        if (firstDefiningType != originalDeclaringType) {
            if (joinPointSignature.getKind() == Member.CONSTRUCTOR) {
                return JoinPointSignature.EMPTY_ARRAY;
            }
        }
        List<ResolvedType> declaringTypes = new ArrayList<>();
        accumulateTypesInBetween(originalDeclaringType, firstDefiningType, declaringTypes);
        Set<ResolvedMember> memberSignatures = new LinkedHashSet<>();
        for (ResolvedType declaringType : declaringTypes) {
            memberSignatures.add(new JoinPointSignature(firstDefiningMember, declaringType));
        }
        if (shouldWalkUpHierarchyFor(firstDefiningMember)) {
            Iterator<ResolvedType> superTypeIterator = firstDefiningType.getDirectSupertypes();
            List<ResolvedType> typesAlreadyVisited = new ArrayList<>();
            accumulateMembersMatching(firstDefiningMember, superTypeIterator, typesAlreadyVisited, memberSignatures, false);
        }
        JoinPointSignature[] ret = new JoinPointSignature[memberSignatures.size()];
        memberSignatures.toArray(ret);
        return ret;
    }

    private static boolean shouldWalkUpHierarchyFor(Member aMember) {
        if (aMember.getKind() == Member.CONSTRUCTOR) {
            return false;
        }
        if (aMember.getKind() == Member.FIELD) {
            return false;
        }
        if (Modifier.isStatic(aMember.getModifiers())) {
            return false;
        }
        return true;
    }

    private static void accumulateTypesInBetween(ResolvedType subType, ResolvedType superType, List<ResolvedType> types) {
        types.add(subType);
        if (subType == superType) {
            return;
        } else {
            for (Iterator<ResolvedType> iter = subType.getDirectSupertypes(); iter.hasNext(); ) {
                ResolvedType parent = iter.next();
                if (superType.isAssignableFrom(parent)) {
                    accumulateTypesInBetween(parent, superType, types);
                }
            }
        }
    }

    private static void accumulateMembersMatching(ResolvedMemberImpl memberToMatch, Iterator<ResolvedType> typesToLookIn, List<ResolvedType> typesAlreadyVisited, Set<ResolvedMember> foundMembers, boolean ignoreGenerics) {
        while (typesToLookIn.hasNext()) {
            ResolvedType toLookIn = typesToLookIn.next();
            if (!typesAlreadyVisited.contains(toLookIn)) {
                typesAlreadyVisited.add(toLookIn);
                ResolvedMemberImpl foundMember = (ResolvedMemberImpl) toLookIn.lookupResolvedMember(memberToMatch, true, ignoreGenerics);
                if (foundMember != null && isVisibleTo(memberToMatch, foundMember)) {
                    List<ResolvedType> declaringTypes = new ArrayList<>();
                    ResolvedType resolvedDeclaringType = foundMember.getDeclaringType().resolve(toLookIn.getWorld());
                    accumulateTypesInBetween(toLookIn, resolvedDeclaringType, declaringTypes);
                    for (ResolvedType declaringType : declaringTypes) {
                        foundMembers.add(new JoinPointSignature(foundMember, declaringType));
                    }
                    if (!ignoreGenerics && toLookIn.isParameterizedType() && (foundMember.backingGenericMember != null)) {
                        foundMembers.add(new JoinPointSignature(foundMember.backingGenericMember, foundMember.declaringType.resolve(toLookIn.getWorld())));
                    }
                    accumulateMembersMatching(foundMember, toLookIn.getDirectSupertypes(), typesAlreadyVisited, foundMembers, ignoreGenerics);
                }
            }
        }
    }

    private static boolean isVisibleTo(ResolvedMember childMember, ResolvedMember parentMember) {
        if (childMember.getDeclaringType().equals(parentMember.getDeclaringType())) {
            return true;
        }
        if (Modifier.isPrivate(parentMember.getModifiers())) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public final int getModifiers(World world) {
        return modifiers;
    }

    @Override
    public final int getModifiers() {
        return modifiers;
    }

    @Override
    public final UnresolvedType[] getExceptions(World world) {
        return getExceptions();
    }

    public UnresolvedType[] getExceptions() {
        return checkedExceptions;
    }

    public ShadowMunger getAssociatedShadowMunger() {
        return null;
    }

    public boolean isAjSynthetic() {
        return isAjSynthetic;
    }

    protected void setAjSynthetic(boolean b) {
        isAjSynthetic = b;
    }

    public boolean hasAnnotations() {
        return (annotationTypes != null);
    }

    public boolean hasAnnotation(UnresolvedType ofType) {
        if (backingGenericMember != null) {
            if (annotationTypes != null) {
                throw new BCException("Unexpectedly found a backing generic member and a local set of annotations");
            }
            return backingGenericMember.hasAnnotation(ofType);
        }
        if (annotationTypes != null) {
            for (ResolvedType annotationType : annotationTypes) {
                if (annotationType.equals(ofType)) {
                    return true;
                }
            }
        }
        return false;
    }

    public ResolvedType[] getAnnotationTypes() {
        if (backingGenericMember != null) {
            if (annotationTypes != null) {
                throw new BCException("Unexpectedly found a backing generic member and a local set of annotations");
            }
            return backingGenericMember.getAnnotationTypes();
        }
        return annotationTypes;
    }

    public String getAnnotationDefaultValue() {
        throw new UnsupportedOperationException("You should resolve this member and call getAnnotationDefaultValue() on the result...");
    }

    @Override
    public AnnotationAJ[] getAnnotations() {
        if (backingGenericMember != null) {
            return backingGenericMember.getAnnotations();
        }
        if (annotations != null) {
            return annotations;
        }
        return super.getAnnotations();
    }

    public AnnotationAJ getAnnotationOfType(UnresolvedType ofType) {
        if (annotations != null) {
            for (AnnotationAJ annotation : annotations) {
                if (annotation.getType().equals(ofType)) {
                    return annotation;
                }
            }
            return null;
        }
        throw new UnsupportedOperationException("You should resolve this member and call getAnnotationOfType() on the result...");
    }

    public void setAnnotations(AnnotationAJ[] annotations) {
        this.annotations = annotations;
    }

    public void setAnnotationTypes(ResolvedType[] annotationTypes) {
        this.annotationTypes = annotationTypes;
    }

    public ResolvedType[][] getParameterAnnotationTypes() {
        return parameterAnnotationTypes;
    }

    public AnnotationAJ[][] getParameterAnnotations() {
        if (backingGenericMember != null) {
            return backingGenericMember.getParameterAnnotations();
        }
        throw new BCException("Cannot return parameter annotations for a " + this.getClass().getName() + " member");
    }

    public void addAnnotation(AnnotationAJ annotation) {
        if (annotationTypes == null) {
            annotationTypes = new ResolvedType[1];
            annotationTypes[0] = annotation.getType();
            annotations = new AnnotationAJ[1];
            annotations[0] = annotation;
        } else {
            int len = annotations.length;
            AnnotationAJ[] ret = new AnnotationAJ[len + 1];
            System.arraycopy(annotations, 0, ret, 0, len);
            ret[len] = annotation;
            annotations = ret;
            ResolvedType[] newAnnotationTypes = new ResolvedType[len + 1];
            System.arraycopy(annotationTypes, 0, newAnnotationTypes, 0, len);
            newAnnotationTypes[len] = annotation.getType();
            annotationTypes = newAnnotationTypes;
        }
    }

    public boolean isBridgeMethod() {
        return (modifiers & Constants.ACC_BRIDGE) != 0 && getKind().equals(METHOD);
    }

    public boolean isVarargsMethod() {
        return (modifiers & Constants.ACC_VARARGS) != 0;
    }

    public void setVarargsMethod() {
        modifiers = modifiers | Constants.ACC_VARARGS;
    }

    public boolean isSynthetic() {
        return (modifiers & 4096) != 0;
    }

    public void write(CompressingDataOutputStream s) throws IOException {
        getKind().write(s);
        s.writeBoolean(s.canCompress());
        if (s.canCompress()) {
            s.writeCompressedSignature(getDeclaringType().getSignature());
        } else {
            getDeclaringType().write(s);
        }
        s.writeInt(modifiers);
        if (s.canCompress()) {
            s.writeCompressedName(getName());
            s.writeCompressedSignature(getSignature());
        } else {
            s.writeUTF(getName());
            s.writeUTF(getSignature());
        }
        UnresolvedType.writeArray(getExceptions(), s);
        s.writeInt(getStart());
        s.writeInt(getEnd());
        s.writeBoolean(isVarargsMethod());
        if (typeVariables == null) {
            s.writeByte(0);
        } else {
            s.writeByte(typeVariables.length);
            for (TypeVariable typeVariable : typeVariables) {
                typeVariable.write(s);
            }
        }
        String gsig = getGenericSignature();
        if (getSignature().equals(gsig)) {
            s.writeByte(0xff);
        } else {
            s.writeByte(parameterTypes.length);
            for (UnresolvedType parameterType : parameterTypes) {
                if (s.canCompress()) {
                    s.writeCompressedSignature(parameterType.getSignature());
                } else {
                    UnresolvedType array_element = parameterType;
                    array_element.write(s);
                }
            }
            if (s.canCompress()) {
                s.writeCompressedSignature(returnType.getSignature());
            } else {
                returnType.write(s);
            }
        }
    }

    public String getSignatureForAttribute() {
        StringBuilder sb = new StringBuilder();
        if (typeVariables != null) {
            sb.append("<");
            for (TypeVariable typeVariable : typeVariables) {
                sb.append(typeVariable.getSignatureForAttribute());
            }
            sb.append(">");
        }
        sb.append("(");
        for (UnresolvedType parameterType : parameterTypes) {
            ResolvedType ptype = (ResolvedType) parameterType;
            sb.append(ptype.getSignatureForAttribute());
        }
        sb.append(")");
        sb.append(((ResolvedType) returnType).getSignatureForAttribute());
        return sb.toString();
    }

    public String getGenericSignature() {
        StringBuilder sb = new StringBuilder();
        if (typeVariables != null) {
            sb.append("<");
            for (TypeVariable typeVariable : typeVariables) {
                sb.append(typeVariable.getSignature());
            }
            sb.append(">");
        }
        sb.append("(");
        for (UnresolvedType ptype : parameterTypes) {
            sb.append(ptype.getSignature());
        }
        sb.append(")");
        sb.append(returnType.getSignature());
        return sb.toString();
    }

    public static void writeArray(ResolvedMember[] members, CompressingDataOutputStream s) throws IOException {
        s.writeInt(members.length);
        for (ResolvedMember member : members) {
            member.write(s);
        }
    }

    public static ResolvedMemberImpl readResolvedMember(VersionedDataInputStream s, ISourceContext sourceContext) throws IOException {
        MemberKind mk = MemberKind.read(s);
        boolean compressed = (s.isAtLeast169() ? s.readBoolean() : false);
        UnresolvedType declaringType = compressed ? UnresolvedType.forSignature(s.readUtf8(s.readShort())) : UnresolvedType.read(s);
        int modifiers = s.readInt();
        String name = compressed ? s.readUtf8(s.readShort()) : s.readUTF();
        String signature = compressed ? s.readUtf8(s.readShort()) : s.readUTF();
        ResolvedMemberImpl m = new ResolvedMemberImpl(mk, declaringType, modifiers, name, signature);
        m.checkedExceptions = UnresolvedType.readArray(s);
        m.start = s.readInt();
        m.end = s.readInt();
        m.sourceContext = sourceContext;
        if (s.getMajorVersion() >= AjAttribute.WeaverVersionInfo.WEAVER_VERSION_MAJOR_AJ150) {
            if (s.getMajorVersion() >= AjAttribute.WeaverVersionInfo.WEAVER_VERSION_MAJOR_AJ150M4) {
                boolean isvarargs = s.readBoolean();
                if (isvarargs) {
                    m.setVarargsMethod();
                }
            }
            int tvcount = s.isAtLeast169() ? s.readByte() : s.readInt();
            if (tvcount != 0) {
                m.typeVariables = new TypeVariable[tvcount];
                for (int i = 0; i < tvcount; i++) {
                    m.typeVariables[i] = TypeVariable.read(s);
                    m.typeVariables[i].setDeclaringElement(m);
                    m.typeVariables[i].setRank(i);
                }
            }
            if (s.getMajorVersion() >= AjAttribute.WeaverVersionInfo.WEAVER_VERSION_MAJOR_AJ150M4) {
                int pcount = -1;
                boolean hasAGenericSignature = false;
                if (s.isAtLeast169()) {
                    pcount = s.readByte();
                    hasAGenericSignature = (pcount >= 0 && pcount < 255);
                } else {
                    hasAGenericSignature = s.readBoolean();
                }
                if (hasAGenericSignature) {
                    int ps = (s.isAtLeast169() ? pcount : s.readInt());
                    UnresolvedType[] params = new UnresolvedType[ps];
                    for (int i = 0; i < params.length; i++) {
                        if (compressed) {
                            params[i] = TypeFactory.createTypeFromSignature(s.readSignature());
                        } else {
                            params[i] = TypeFactory.createTypeFromSignature(s.readUTF());
                        }
                    }
                    UnresolvedType rt = compressed ? TypeFactory.createTypeFromSignature(s.readSignature()) : TypeFactory.createTypeFromSignature(s.readUTF());
                    m.parameterTypes = params;
                    m.returnType = rt;
                }
            }
        }
        return m;
    }

    public static ResolvedMember[] readResolvedMemberArray(VersionedDataInputStream s, ISourceContext context) throws IOException {
        int len = s.readInt();
        ResolvedMember[] members = new ResolvedMember[len];
        for (int i = 0; i < len; i++) {
            members[i] = ResolvedMemberImpl.readResolvedMember(s, context);
        }
        return members;
    }

    @Override
    public ResolvedMember resolve(World world) {
        if (isResolved) {
            return this;
        }
        try {
            if (typeVariables != null && typeVariables.length > 0) {
                for (int i = 0; i < typeVariables.length; i++) {
                    typeVariables[i] = typeVariables[i].resolve(world);
                }
            }
            world.setTypeVariableLookupScope(this);
            declaringType = declaringType.resolve(world);
            if (declaringType.isRawType()) {
                declaringType = ((ReferenceType) declaringType).getGenericType();
            }
            if (parameterTypes != null && parameterTypes.length > 0) {
                for (int i = 0; i < parameterTypes.length; i++) {
                    parameterTypes[i] = parameterTypes[i].resolve(world);
                }
            }
            returnType = returnType.resolve(world);
        } finally {
            world.setTypeVariableLookupScope(null);
        }
        isResolved = true;
        return this;
    }

    public ISourceContext getSourceContext(World world) {
        return getDeclaringType().resolve(world).getSourceContext();
    }

    public String[] getParameterNames() {
        return parameterNames;
    }

    public final void setParameterNames(String[] pnames) {
        parameterNames = pnames;
    }

    @Override
    public final String[] getParameterNames(World world) {
        return getParameterNames();
    }

    public AjAttribute.EffectiveSignatureAttribute getEffectiveSignature() {
        return null;
    }

    public ISourceLocation getSourceLocation() {
        if (getSourceContext() == null) {
            return null;
        }
        return getSourceContext().makeSourceLocation(this);
    }

    public int getEnd() {
        return end;
    }

    public ISourceContext getSourceContext() {
        return sourceContext;
    }

    public int getStart() {
        return start;
    }

    public void setPosition(int sourceStart, int sourceEnd) {
        this.start = sourceStart;
        this.end = sourceEnd;
    }

    public void setDeclaringType(ReferenceType rt) {
        declaringType = rt;
    }

    public void setSourceContext(ISourceContext sourceContext) {
        this.sourceContext = sourceContext;
    }

    public boolean isAbstract() {
        return Modifier.isAbstract(modifiers);
    }

    public boolean isPublic() {
        return Modifier.isPublic(modifiers);
    }

    public boolean isDefault() {
        int mods = getModifiers();
        return !(Modifier.isPublic(mods) || Modifier.isProtected(mods) || Modifier.isPrivate(mods));
    }

    public boolean isVisible(ResolvedType fromType) {
        UnresolvedType declaringType = getDeclaringType();
        ResolvedType type = null;
        if (fromType.equals(declaringType)) {
            type = fromType;
        } else {
            World world = fromType.getWorld();
            type = declaringType.resolve(world);
        }
        return ResolvedType.isVisible(getModifiers(), type, fromType);
    }

    public void setCheckedExceptions(UnresolvedType[] checkedExceptions) {
        this.checkedExceptions = checkedExceptions;
    }

    public void setAnnotatedElsewhere(boolean b) {
        isAnnotatedElsewhere = b;
    }

    public boolean isAnnotatedElsewhere() {
        return isAnnotatedElsewhere;
    }

    @Override
    public UnresolvedType getGenericReturnType() {
        return getReturnType();
    }

    @Override
    public UnresolvedType[] getGenericParameterTypes() {
        return getParameterTypes();
    }

    public ResolvedMemberImpl parameterizedWith(UnresolvedType[] typeParameters, ResolvedType newDeclaringType, boolean isParameterized) {
        return parameterizedWith(typeParameters, newDeclaringType, isParameterized, null);
    }

    public ResolvedMemberImpl parameterizedWith(UnresolvedType[] typeParameters, ResolvedType newDeclaringType, boolean isParameterized, List<String> aliases) {
        if (!getDeclaringType().isGenericType() && !getDeclaringType().getName().contains("$")) {
            throw new IllegalStateException("Can't ask to parameterize a member of non-generic type: " + getDeclaringType() + "  kind(" + getDeclaringType().typeKind + ")");
        }
        TypeVariable[] typeVariables = getDeclaringType().getTypeVariables();
        if (isParameterized && (typeVariables.length != typeParameters.length)) {
            throw new IllegalStateException("Wrong number of type parameters supplied");
        }
        Map<String, UnresolvedType> typeMap = new HashMap<>();
        boolean typeParametersSupplied = typeParameters != null && typeParameters.length > 0;
        if (typeVariables != null) {
            for (int i = 0; i < typeVariables.length; i++) {
                UnresolvedType ut = (!typeParametersSupplied ? typeVariables[i].getFirstBound() : typeParameters[i]);
                typeMap.put(typeVariables[i].getName(), ut);
            }
        }
        if (aliases != null) {
            int posn = 0;
            for (String typeVariableAlias : aliases) {
                typeMap.put(typeVariableAlias, (!typeParametersSupplied ? typeVariables[posn].getFirstBound() : typeParameters[posn]));
                posn++;
            }
        }
        UnresolvedType parameterizedReturnType = parameterize(getGenericReturnType(), typeMap, isParameterized, newDeclaringType.getWorld());
        UnresolvedType[] parameterizedParameterTypes = new UnresolvedType[getGenericParameterTypes().length];
        UnresolvedType[] genericParameterTypes = getGenericParameterTypes();
        for (int i = 0; i < parameterizedParameterTypes.length; i++) {
            parameterizedParameterTypes[i] = parameterize(genericParameterTypes[i], typeMap, isParameterized, newDeclaringType.getWorld());
        }
        ResolvedMemberImpl ret = new ResolvedMemberImpl(getKind(), newDeclaringType, getModifiers(), parameterizedReturnType, getName(), parameterizedParameterTypes, getExceptions(), this);
        ret.setTypeVariables(getTypeVariables());
        ret.setSourceContext(getSourceContext());
        ret.setPosition(getStart(), getEnd());
        ret.setParameterNames(getParameterNames());
        return ret;
    }

    public ResolvedMember parameterizedWith(Map<String, UnresolvedType> m, World w) {
        declaringType = declaringType.resolve(w);
        if (declaringType.isRawType()) {
            declaringType = ((ResolvedType) declaringType).getGenericType();
        }
        UnresolvedType parameterizedReturnType = parameterize(getGenericReturnType(), m, true, w);
        UnresolvedType[] parameterizedParameterTypes = new UnresolvedType[getGenericParameterTypes().length];
        UnresolvedType[] genericParameterTypes = getGenericParameterTypes();
        for (int i = 0; i < parameterizedParameterTypes.length; i++) {
            parameterizedParameterTypes[i] = parameterize(genericParameterTypes[i], m, true, w);
        }
        ResolvedMemberImpl ret = new ResolvedMemberImpl(getKind(), declaringType, getModifiers(), parameterizedReturnType, getName(), parameterizedParameterTypes, getExceptions(), this);
        ret.setTypeVariables(getTypeVariables());
        ret.setSourceContext(getSourceContext());
        ret.setPosition(getStart(), getEnd());
        ret.setParameterNames(getParameterNames());
        return ret;
    }

    public void setTypeVariables(TypeVariable[] tvars) {
        typeVariables = tvars;
    }

    public TypeVariable[] getTypeVariables() {
        return typeVariables;
    }

    protected UnresolvedType parameterize(UnresolvedType aType, Map<String, UnresolvedType> typeVariableMap, boolean inParameterizedType, World w) {
        if (aType instanceof TypeVariableReference) {
            String variableName = ((TypeVariableReference) aType).getTypeVariable().getName();
            if (!typeVariableMap.containsKey(variableName)) {
                return aType;
            }
            return typeVariableMap.get(variableName);
        } else if (aType.isParameterizedType()) {
            if (inParameterizedType) {
                if (w != null) {
                    aType = aType.resolve(w);
                } else {
                    UnresolvedType dType = getDeclaringType();
                    aType = aType.resolve(((ResolvedType) dType).getWorld());
                }
                return aType.parameterize(typeVariableMap);
            } else {
                return aType.getRawType();
            }
        } else if (aType.isArray()) {
            int dims = 1;
            String sig = aType.getSignature();
            UnresolvedType arrayType = null;
            UnresolvedType componentSig = UnresolvedType.forSignature(sig.substring(dims));
            UnresolvedType parameterizedComponentSig = parameterize(componentSig, typeVariableMap, inParameterizedType, w);
            if (parameterizedComponentSig.isTypeVariableReference() && parameterizedComponentSig instanceof UnresolvedTypeVariableReferenceType && typeVariableMap.containsKey(((UnresolvedTypeVariableReferenceType) parameterizedComponentSig).getTypeVariable().getName())) {
                StringBuilder newsig = new StringBuilder();
                newsig.append("[T");
                newsig.append(((UnresolvedTypeVariableReferenceType) parameterizedComponentSig).getTypeVariable().getName());
                newsig.append(";");
                arrayType = UnresolvedType.forSignature(newsig.toString());
            } else {
                arrayType = ResolvedType.makeArray(parameterizedComponentSig, dims);
            }
            return arrayType;
        }
        return aType;
    }

    public boolean hasBackingGenericMember() {
        return backingGenericMember != null;
    }

    public ResolvedMember getBackingGenericMember() {
        return backingGenericMember;
    }

    public void resetName(String newName) {
        this.name = newName;
    }

    public void resetKind(MemberKind newKind) {
        this.kind = newKind;
    }

    public void resetModifiers(int newModifiers) {
        this.modifiers = newModifiers;
    }

    public void resetReturnTypeToObjectArray() {
        returnType = UnresolvedType.OBJECTARRAY;
    }

    public boolean matches(ResolvedMember aCandidateMatch, boolean ignoreGenerics) {
        ResolvedMemberImpl candidateMatchImpl = (ResolvedMemberImpl) aCandidateMatch;
        if (!getName().equals(aCandidateMatch.getName())) {
            return false;
        }
        UnresolvedType[] parameterTypes = getGenericParameterTypes();
        UnresolvedType[] candidateParameterTypes = aCandidateMatch.getGenericParameterTypes();
        if (parameterTypes.length != candidateParameterTypes.length) {
            return false;
        }
        boolean b = false;
        String myParameterSignature = getParameterSigWithBoundsRemoved();
        String candidateParameterSignature = candidateMatchImpl.getParameterSigWithBoundsRemoved();
        if (myParameterSignature.equals(candidateParameterSignature)) {
            b = true;
        } else {
            myParameterSignature = getParameterSignatureErased();
            candidateParameterSignature = candidateMatchImpl.getParameterSignatureErased();
            b = myParameterSignature.equals(candidateParameterSignature);
        }
        return b;
    }

    private String myParameterSignatureWithBoundsRemoved = null;

    private String myParameterSignatureErasure = null;

    private String getParameterSigWithBoundsRemoved() {
        if (myParameterSignatureWithBoundsRemoved != null) {
            return myParameterSignatureWithBoundsRemoved;
        }
        StringBuffer sig = new StringBuffer();
        UnresolvedType[] myParameterTypes = getGenericParameterTypes();
        for (UnresolvedType myParameterType : myParameterTypes) {
            appendSigWithTypeVarBoundsRemoved(myParameterType, sig, new HashSet<>());
        }
        myParameterSignatureWithBoundsRemoved = sig.toString();
        return myParameterSignatureWithBoundsRemoved;
    }

    public String getParameterSignatureErased() {
        if (myParameterSignatureErasure == null) {
            StringBuilder sig = new StringBuilder();
            for (UnresolvedType parameter : getParameterTypes()) {
                sig.append(parameter.getErasureSignature());
            }
            myParameterSignatureErasure = sig.toString();
        }
        return myParameterSignatureErasure;
    }

    public String getSignatureErased() {
        StringBuilder sb = new StringBuilder();
        sb.append("(");
        sb.append(getParameterSignatureErased());
        sb.append(")");
        sb.append(getReturnType().getErasureSignature());
        return sb.toString();
    }

    public static void appendSigWithTypeVarBoundsRemoved(UnresolvedType aType, StringBuffer toBuffer, Set<UnresolvedType> alreadyUsedTypeVars) {
        if (aType.isTypeVariableReference()) {
            TypeVariableReferenceType typeVariableRT = (TypeVariableReferenceType) aType;
            if (alreadyUsedTypeVars.contains(aType)) {
                toBuffer.append("...");
            } else {
                alreadyUsedTypeVars.add(aType);
                appendSigWithTypeVarBoundsRemoved(typeVariableRT.getTypeVariable().getFirstBound(), toBuffer, alreadyUsedTypeVars);
            }
        } else if (aType.isParameterizedType()) {
            toBuffer.append(aType.getRawType().getSignature());
            toBuffer.append("<");
            for (int i = 0; i < aType.getTypeParameters().length; i++) {
                appendSigWithTypeVarBoundsRemoved(aType.getTypeParameters()[i], toBuffer, alreadyUsedTypeVars);
            }
            toBuffer.append(">;");
        } else {
            toBuffer.append(aType.getSignature());
        }
    }

    public String toDebugString() {
        StringBuilder r = new StringBuilder();
        int mods = modifiers;
        if ((mods & 4096) > 0) {
            mods = mods - 4096;
        }
        if ((mods & 512) > 0) {
            mods = mods - 512;
        }
        if ((mods & 131072) > 0) {
            mods = mods - 131072;
        }
        String modsStr = Modifier.toString(mods);
        if (modsStr.length() != 0) {
            r.append(modsStr).append("(" + mods + ")").append(" ");
        }
        if (typeVariables != null && typeVariables.length > 0) {
            r.append("<");
            for (int i = 0; i < typeVariables.length; i++) {
                if (i > 0) {
                    r.append(",");
                }
                TypeVariable t = typeVariables[i];
                r.append(t.toDebugString());
            }
            r.append("> ");
        }
        r.append(getGenericReturnType().toDebugString());
        r.append(' ');
        r.append(declaringType.getName());
        r.append('.');
        r.append(name);
        if (kind != FIELD) {
            r.append("(");
            UnresolvedType[] params = getGenericParameterTypes();
            boolean parameterNamesExist = showParameterNames && parameterNames != null && parameterNames.length == params.length;
            if (params.length != 0) {
                for (int i = 0, len = params.length; i < len; i++) {
                    if (i > 0) {
                        r.append(", ");
                    }
                    r.append(params[i].toDebugString());
                    if (parameterNamesExist) {
                        r.append(" ").append(parameterNames[i]);
                    }
                }
            }
            r.append(")");
        }
        return r.toString();
    }

    public static boolean showParameterNames = true;

    public String toGenericString() {
        StringBuilder buf = new StringBuilder();
        buf.append(getGenericReturnType().getSimpleName());
        buf.append(' ');
        buf.append(declaringType.getName());
        buf.append('.');
        buf.append(name);
        if (kind != FIELD) {
            buf.append("(");
            UnresolvedType[] params = getGenericParameterTypes();
            if (params.length != 0) {
                buf.append(params[0].getSimpleName());
                for (int i = 1, len = params.length; i < len; i++) {
                    buf.append(", ");
                    buf.append(params[i].getSimpleName());
                }
            }
            buf.append(")");
        }
        return buf.toString();
    }

    public boolean isCompatibleWith(Member am) {
        if (kind != METHOD || am.getKind() != METHOD) {
            return true;
        }
        if (!name.equals(am.getName())) {
            return true;
        }
        if (!equalTypes(getParameterTypes(), am.getParameterTypes())) {
            return true;
        }
        return getReturnType().equals(am.getReturnType());
    }

    private static boolean equalTypes(UnresolvedType[] a, UnresolvedType[] b) {
        int len = a.length;
        if (len != b.length) {
            return false;
        }
        for (int i = 0; i < len; i++) {
            if (!a[i].equals(b[i])) {
                return false;
            }
        }
        return true;
    }

    public TypeVariable getTypeVariableNamed(String name) {
        if (typeVariables != null) {
            for (TypeVariable typeVariable : typeVariables) {
                if (typeVariable.getName().equals(name)) {
                    return typeVariable;
                }
            }
        }
        return declaringType.getTypeVariableNamed(name);
    }

    public void evictWeavingState() {
    }

    public boolean isEquivalentTo(Object other) {
        return this.equals(other);
    }

    public boolean isDefaultConstructor() {
        return false;
    }
}
