package org.aspectj.weaver;

import org.aspectj.weaver.patterns.Pointcut;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class ResolvedPointcutDefinition extends ResolvedMemberImpl {

    private Pointcut pointcut;

    public ResolvedPointcutDefinition(UnresolvedType declaringType, int modifiers, String name, UnresolvedType[] parameterTypes, Pointcut pointcut) {
        this(declaringType, modifiers, name, parameterTypes, UnresolvedType.VOID, pointcut);
    }

    public ResolvedPointcutDefinition(UnresolvedType declaringType, int modifiers, String name, UnresolvedType[] parameterTypes, UnresolvedType returnType, Pointcut pointcut) {
        super(POINTCUT, declaringType, modifiers, returnType, name, parameterTypes);
        this.pointcut = pointcut;
        checkedExceptions = UnresolvedType.NONE;
    }

    @Override
    public void write(CompressingDataOutputStream s) throws IOException {
        getDeclaringType().write(s);
        s.writeInt(getModifiers());
        s.writeUTF(getName());
        UnresolvedType.writeArray(getParameterTypes(), s);
        pointcut.write(s);
    }

    public static ResolvedPointcutDefinition read(VersionedDataInputStream s, ISourceContext context) throws IOException {
        ResolvedPointcutDefinition rpd = new ResolvedPointcutDefinition(UnresolvedType.read(s), s.readInt(), s.readUTF(), UnresolvedType.readArray(s), Pointcut.read(s, context));
        rpd.setSourceContext(context);
        return rpd;
    }

    @Override
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append("pointcut ");
        buf.append((getDeclaringType() == null ? "<nullDeclaringType>" : getDeclaringType().getName()));
        buf.append(".");
        buf.append(getName());
        buf.append("(");
        for (int i = 0; i < getParameterTypes().length; i++) {
            if (i > 0) {
                buf.append(", ");
            }
            buf.append(getParameterTypes()[i].toString());
        }
        buf.append(")");
        return buf.toString();
    }

    public Pointcut getPointcut() {
        return pointcut;
    }

    @Override
    public boolean isAjSynthetic() {
        return true;
    }

    @Override
    public ResolvedMemberImpl parameterizedWith(UnresolvedType[] typeParameters, ResolvedType newDeclaringType, boolean isParameterized) {
        TypeVariable[] typeVariables = getDeclaringType().resolve(newDeclaringType.getWorld()).getTypeVariables();
        if (isParameterized && (typeVariables.length != typeParameters.length)) {
            throw new IllegalStateException("Wrong number of type parameters supplied");
        }
        Map<String, UnresolvedType> typeMap = new HashMap<>();
        boolean typeParametersSupplied = typeParameters != null && typeParameters.length > 0;
        if (typeVariables != null) {
            for (int i = 0; i < typeVariables.length; i++) {
                UnresolvedType ut = (!typeParametersSupplied ? typeVariables[i].getFirstBound() : typeParameters[i]);
                typeMap.put(typeVariables[i].getName(), ut);
            }
        }
        UnresolvedType parameterizedReturnType = parameterize(getGenericReturnType(), typeMap, isParameterized, newDeclaringType.getWorld());
        UnresolvedType[] parameterizedParameterTypes = new UnresolvedType[getGenericParameterTypes().length];
        for (int i = 0; i < parameterizedParameterTypes.length; i++) {
            parameterizedParameterTypes[i] = parameterize(getGenericParameterTypes()[i], typeMap, isParameterized, newDeclaringType.getWorld());
        }
        ResolvedPointcutDefinition ret = new ResolvedPointcutDefinition(newDeclaringType, getModifiers(), getName(), parameterizedParameterTypes, parameterizedReturnType, pointcut.parameterizeWith(typeMap, newDeclaringType.getWorld()));
        ret.setTypeVariables(getTypeVariables());
        ret.setSourceContext(getSourceContext());
        ret.setPosition(getStart(), getEnd());
        ret.setParameterNames(getParameterNames());
        return ret;
    }

    public static final ResolvedPointcutDefinition DUMMY = new ResolvedPointcutDefinition(UnresolvedType.OBJECT, 0, "missing", UnresolvedType.NONE, Pointcut.makeMatchesNothing(Pointcut.RESOLVED));

    public static final ResolvedPointcutDefinition[] NO_POINTCUTS = new ResolvedPointcutDefinition[]{};

    public void setPointcut(Pointcut pointcut) {
        this.pointcut = pointcut;
    }
}
