package org.aspectj.weaver;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.bridge.SourceLocation;
import org.aspectj.util.FuzzyBoolean;
import org.aspectj.util.PartialOrder;
import org.aspectj.weaver.patterns.PerClause;
import org.aspectj.weaver.patterns.Pointcut;
import org.aspectj.weaver.patterns.TypePattern;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public abstract class ShadowMunger implements PartialOrder.PartialComparable, IHasPosition {

    public static final ShadowMunger[] NONE = new ShadowMunger[0];

    private static int VERSION_1 = 1;

    protected static final int ShadowMungerAdvice = 1;

    protected static final int ShadowMungerDeow = 2;

    public String handle = null;

    private int shadowMungerKind;

    protected int start, end;

    protected ISourceContext sourceContext;

    private ISourceLocation sourceLocation;

    private ISourceLocation binarySourceLocation;

    private File binaryFile;

    private ResolvedType declaringType;

    private boolean isBinary;

    private boolean checkedIsBinary;

    protected Pointcut pointcut;

    protected ShadowMunger() {
    }

    public ShadowMunger(Pointcut pointcut, int start, int end, ISourceContext sourceContext, int shadowMungerKind) {
        this.shadowMungerKind = shadowMungerKind;
        this.pointcut = pointcut;
        this.start = start;
        this.end = end;
        this.sourceContext = sourceContext;
    }

    public boolean match(Shadow shadow, World world) {
        if (world.isXmlConfigured() && world.isAspectIncluded(declaringType)) {
            TypePattern scoped = world.getAspectScope(declaringType);
            if (scoped != null) {
                Set<ResolvedType> excludedTypes = world.getExclusionMap().get(declaringType);
                ResolvedType type = shadow.getEnclosingType().resolve(world);
                if (excludedTypes != null && excludedTypes.contains(type)) {
                    return false;
                }
                boolean b = scoped.matches(type, TypePattern.STATIC).alwaysTrue();
                if (!b) {
                    if (!world.getMessageHandler().isIgnoring(IMessage.INFO)) {
                        world.getMessageHandler().handleMessage(MessageUtil.info("Type '" + type.getName() + "' not woven by aspect '" + declaringType.getName() + "' due to scope exclusion in XML definition"));
                    }
                    if (excludedTypes == null) {
                        excludedTypes = new HashSet<>();
                        excludedTypes.add(type);
                        world.getExclusionMap().put(declaringType, excludedTypes);
                    } else {
                        excludedTypes.add(type);
                    }
                    return false;
                }
            }
        }
        if (world.areInfoMessagesEnabled() && world.isTimingEnabled()) {
            long starttime = System.nanoTime();
            FuzzyBoolean isMatch = pointcut.match(shadow);
            long endtime = System.nanoTime();
            world.record(pointcut, endtime - starttime);
            return isMatch.maybeTrue();
        } else {
            FuzzyBoolean isMatch = pointcut.match(shadow);
            return isMatch.maybeTrue();
        }
    }

    public int fallbackCompareTo(Object other) {
        return toString().compareTo(toString());
    }

    public int getEnd() {
        return end;
    }

    public int getStart() {
        return start;
    }

    public ISourceLocation getSourceLocation() {
        if (sourceLocation == null) {
            if (sourceContext != null) {
                sourceLocation = sourceContext.makeSourceLocation(this);
            }
        }
        if (isBinary()) {
            if (binarySourceLocation == null) {
                binarySourceLocation = getBinarySourceLocation(sourceLocation);
            }
            return binarySourceLocation;
        }
        return sourceLocation;
    }

    public Pointcut getPointcut() {
        return pointcut;
    }

    public void setPointcut(Pointcut pointcut) {
        this.pointcut = pointcut;
    }

    public void setDeclaringType(ResolvedType aType) {
        declaringType = aType;
    }

    public ResolvedType getDeclaringType() {
        return declaringType;
    }

    public abstract ResolvedType getConcreteAspect();

    public ISourceLocation getBinarySourceLocation(ISourceLocation sl) {
        if (sl == null) {
            return null;
        }
        String sourceFileName = null;
        if (getDeclaringType() instanceof ReferenceType) {
            String s = ((ReferenceType) getDeclaringType()).getDelegate().getSourcefilename();
            int i = s.lastIndexOf('/');
            if (i != -1) {
                sourceFileName = s.substring(i + 1);
            } else {
                sourceFileName = s;
            }
        }
        ISourceLocation sLoc = new SourceLocation(getBinaryFile(), sl.getLine(), sl.getEndLine(), ((sl.getColumn() == 0) ? ISourceLocation.NO_COLUMN : sl.getColumn()), sl.getContext(), sourceFileName);
        return sLoc;
    }

    private File getBinaryFile() {
        if (binaryFile == null) {
            String binaryPath = getDeclaringType().getBinaryPath();
            if (binaryPath == null) {
                binaryPath = "classpath";
                getDeclaringType().setBinaryPath(binaryPath);
            }
            if (!binaryPath.contains("!")) {
                File f = getDeclaringType().getSourceLocation().getSourceFile();
                int i = f.getPath().lastIndexOf('.');
                String path = null;
                if (i != -1) {
                    path = f.getPath().substring(0, i) + ".class";
                } else {
                    path = f.getPath() + ".class";
                }
                binaryFile = new File(binaryPath + "!" + path);
            } else {
                binaryFile = new File(binaryPath);
            }
        }
        return binaryFile;
    }

    public boolean isBinary() {
        if (!checkedIsBinary) {
            ResolvedType rt = getDeclaringType();
            if (rt != null) {
                isBinary = ((rt.getBinaryPath() == null) ? false : true);
            }
            checkedIsBinary = true;
        }
        return isBinary;
    }

    public abstract ShadowMunger concretize(ResolvedType fromType, World world, PerClause clause);

    public abstract void specializeOn(Shadow shadow);

    public abstract boolean implementOn(Shadow shadow);

    public abstract ShadowMunger parameterizeWith(ResolvedType declaringType, Map<String, UnresolvedType> typeVariableMap);

    public abstract Collection<ResolvedType> getThrownExceptions();

    public abstract boolean mustCheckExceptions();

    public void write(CompressingDataOutputStream stream) throws IOException {
        stream.writeInt(VERSION_1);
        stream.writeInt(shadowMungerKind);
        stream.writeInt(start);
        stream.writeInt(end);
        PersistenceSupport.write(stream, sourceContext);
        PersistenceSupport.write(stream, sourceLocation);
        PersistenceSupport.write(stream, binarySourceLocation);
        PersistenceSupport.write(stream, binaryFile);
        declaringType.write(stream);
        stream.writeBoolean(isBinary);
        stream.writeBoolean(checkedIsBinary);
        pointcut.write(stream);
    }

    public boolean bindsProceedingJoinPoint() {
        return false;
    }

    public boolean isAroundAdvice() {
        return false;
    }
}
