package org.aspectj.weaver;

import java.util.ArrayList;
import java.util.List;

public class TypeFactory {

    public static ReferenceType createParameterizedType(ResolvedType aBaseType, UnresolvedType[] someTypeParameters, World inAWorld) {
        ResolvedType baseType = aBaseType;
        if (!aBaseType.isGenericType()) {
            if (someTypeParameters != null && someTypeParameters.length > 0) {
                if (!aBaseType.isRawType()) {
                    throw new IllegalStateException("Expecting raw type, but " + aBaseType + " is of type " + aBaseType.getTypekind());
                }
                baseType = baseType.getGenericType();
                if (baseType == null) {
                    throw new IllegalStateException("Raw type does not have generic type set");
                }
            }
        }
        ResolvedType[] resolvedParameters = inAWorld.resolve(someTypeParameters);
        ReferenceType existingType = ((ReferenceType) baseType).findDerivativeType(resolvedParameters);
        ReferenceType pType = null;
        if (existingType != null) {
            pType = existingType;
        } else {
            pType = new ReferenceType(baseType, resolvedParameters, inAWorld);
        }
        return (ReferenceType) pType.resolve(inAWorld);
    }

    public static UnresolvedType createUnresolvedParameterizedType(String sig, String erasuresig, UnresolvedType[] arguments) {
        return new UnresolvedType(sig, erasuresig, arguments);
    }

    static UnresolvedType convertSigToType(String aSignature) {
        UnresolvedType bound = null;
        int startOfParams = aSignature.indexOf('<');
        if (startOfParams == -1) {
            bound = UnresolvedType.forSignature(aSignature);
        } else {
            int endOfParams = aSignature.lastIndexOf('>');
            String signatureErasure = "L" + aSignature.substring(1, startOfParams) + ";";
            UnresolvedType[] typeParams = createTypeParams(aSignature.substring(startOfParams + 1, endOfParams));
            bound = new UnresolvedType("P" + aSignature.substring(1), signatureErasure, typeParams);
        }
        return bound;
    }

    public static UnresolvedType createTypeFromSignature(String signature) {
        char firstChar = signature.charAt(0);
        if (firstChar == 'P') {
            int startOfParams = signature.indexOf('<');
            if (startOfParams == -1) {
                String signatureErasure = "L" + signature.substring(1);
                return new UnresolvedType(signature, signatureErasure, UnresolvedType.NONE);
            } else {
                int endOfParams = locateMatchingEndAngleBracket(signature, startOfParams);
                StringBuilder erasureSig = new StringBuilder(signature);
                erasureSig.setCharAt(0, 'L');
                while (startOfParams != -1) {
                    erasureSig.delete(startOfParams, endOfParams + 1);
                    startOfParams = locateFirstBracket(erasureSig);
                    if (startOfParams != -1) {
                        endOfParams = locateMatchingEndAngleBracket(erasureSig, startOfParams);
                    }
                }
                String signatureErasure = erasureSig.toString();
                String lastType = null;
                int nestedTypePosition = signature.indexOf("$", endOfParams);
                if (nestedTypePosition != -1) {
                    lastType = signature.substring(nestedTypePosition + 1);
                } else {
                    lastType = new String(signature);
                }
                startOfParams = lastType.indexOf("<");
                UnresolvedType[] typeParams = UnresolvedType.NONE;
                if (startOfParams != -1) {
                    endOfParams = locateMatchingEndAngleBracket(lastType, startOfParams);
                    typeParams = createTypeParams(lastType.substring(startOfParams + 1, endOfParams));
                }
                StringBuilder s = new StringBuilder();
                int firstAngleBracket = signature.indexOf('<');
                s.append("P").append(signature.substring(1, firstAngleBracket));
                s.append('<');
                for (UnresolvedType typeParameter : typeParams) {
                    s.append(typeParameter.getSignature());
                }
                s.append(">;");
                signature = s.toString();
                return new UnresolvedType(signature, signatureErasure, typeParams);
            }
        } else if ((firstChar == '?' || firstChar == '*') && signature.length() == 1) {
            return WildcardedUnresolvedType.QUESTIONMARK;
        } else if (firstChar == '+') {
            UnresolvedType upperBound = convertSigToType(signature.substring(1));
            WildcardedUnresolvedType wildcardedUT = new WildcardedUnresolvedType(signature, upperBound, null);
            return wildcardedUT;
        } else if (firstChar == '-') {
            UnresolvedType lowerBound = convertSigToType(signature.substring(1));
            WildcardedUnresolvedType wildcardedUT = new WildcardedUnresolvedType(signature, null, lowerBound);
            return wildcardedUT;
        } else if (firstChar == 'T') {
            String typeVariableName = signature.substring(1);
            if (typeVariableName.endsWith(";")) {
                typeVariableName = typeVariableName.substring(0, typeVariableName.length() - 1);
            }
            return new UnresolvedTypeVariableReferenceType(new TypeVariable(typeVariableName));
        } else if (firstChar == '[') {
            int dims = 0;
            while (signature.charAt(dims) == '[') {
                dims++;
            }
            UnresolvedType componentType = createTypeFromSignature(signature.substring(dims));
            return new UnresolvedType(signature, signature.substring(0, dims) + componentType.getErasureSignature());
        } else if (signature.length() == 1) {
            switch (firstChar) {
                case 'V':
                    return UnresolvedType.VOID;
                case 'Z':
                    return UnresolvedType.BOOLEAN;
                case 'B':
                    return UnresolvedType.BYTE;
                case 'C':
                    return UnresolvedType.CHAR;
                case 'D':
                    return UnresolvedType.DOUBLE;
                case 'F':
                    return UnresolvedType.FLOAT;
                case 'I':
                    return UnresolvedType.INT;
                case 'J':
                    return UnresolvedType.LONG;
                case 'S':
                    return UnresolvedType.SHORT;
            }
        } else if (firstChar == '@') {
            return ResolvedType.MISSING;
        } else if (firstChar == 'L') {
            int leftAngleBracket = signature.indexOf('<');
            if (leftAngleBracket == -1) {
                return new UnresolvedType(signature);
            } else {
                int endOfParams = locateMatchingEndAngleBracket(signature, leftAngleBracket);
                StringBuilder erasureSig = new StringBuilder(signature);
                erasureSig.setCharAt(0, 'L');
                while (leftAngleBracket != -1) {
                    erasureSig.delete(leftAngleBracket, endOfParams + 1);
                    leftAngleBracket = locateFirstBracket(erasureSig);
                    if (leftAngleBracket != -1) {
                        endOfParams = locateMatchingEndAngleBracket(erasureSig, leftAngleBracket);
                    }
                }
                String signatureErasure = erasureSig.toString();
                String lastType = null;
                int nestedTypePosition = signature.indexOf("$", endOfParams);
                if (nestedTypePosition != -1) {
                    lastType = signature.substring(nestedTypePosition + 1);
                } else {
                    lastType = new String(signature);
                }
                leftAngleBracket = lastType.indexOf("<");
                UnresolvedType[] typeParams = UnresolvedType.NONE;
                if (leftAngleBracket != -1) {
                    endOfParams = locateMatchingEndAngleBracket(lastType, leftAngleBracket);
                    typeParams = createTypeParams(lastType.substring(leftAngleBracket + 1, endOfParams));
                }
                StringBuilder s = new StringBuilder();
                int firstAngleBracket = signature.indexOf('<');
                s.append("P").append(signature.substring(1, firstAngleBracket));
                s.append('<');
                for (UnresolvedType typeParameter : typeParams) {
                    s.append(typeParameter.getSignature());
                }
                s.append(">;");
                signature = s.toString();
                return new UnresolvedType(signature, signatureErasure, typeParams);
            }
        }
        return new UnresolvedType(signature);
    }

    private static int locateMatchingEndAngleBracket(CharSequence signature, int startOfParams) {
        if (startOfParams == -1) {
            return -1;
        }
        int count = 1;
        int idx = startOfParams;
        int max = signature.length();
        while (idx < max) {
            char ch = signature.charAt(++idx);
            if (ch == '<') {
                count++;
            } else if (ch == '>') {
                if (count == 1) {
                    break;
                }
                count--;
            }
        }
        return idx;
    }

    private static int locateFirstBracket(StringBuilder signature) {
        int idx = 0;
        int max = signature.length();
        while (idx < max) {
            if (signature.charAt(idx) == '<') {
                return idx;
            }
            idx++;
        }
        return -1;
    }

    private static UnresolvedType[] createTypeParams(String typeParameterSpecification) {
        String remainingToProcess = typeParameterSpecification;
        List<UnresolvedType> types = new ArrayList<>();
        while (remainingToProcess.length() != 0) {
            int endOfSig = 0;
            int anglies = 0;
            boolean hadAnglies = false;
            boolean sigFound = false;
            for (endOfSig = 0; (endOfSig < remainingToProcess.length()) && !sigFound; endOfSig++) {
                char thisChar = remainingToProcess.charAt(endOfSig);
                switch (thisChar) {
                    case '<':
                        anglies++;
                        hadAnglies = true;
                        break;
                    case '>':
                        anglies--;
                        break;
                    case '*':
                        if (anglies == 0) {
                            int nextCharPos = endOfSig + 1;
                            if (nextCharPos >= remainingToProcess.length()) {
                                sigFound = true;
                            } else {
                                char nextChar = remainingToProcess.charAt(nextCharPos);
                                if (!(nextChar == '+' || nextChar == '-')) {
                                    sigFound = true;
                                }
                            }
                        }
                        break;
                    case '[':
                        if (anglies == 0) {
                            int nextChar = endOfSig + 1;
                            while (remainingToProcess.charAt(nextChar) == '[') {
                                nextChar++;
                            }
                            if ("BCDFIJSZ".indexOf(remainingToProcess.charAt(nextChar)) != -1) {
                                sigFound = true;
                                endOfSig = nextChar;
                                break;
                            }
                        }
                        break;
                    case ';':
                        if (anglies == 0) {
                            sigFound = true;
                            break;
                        }
                }
            }
            String forProcessing = remainingToProcess.substring(0, endOfSig);
            if (hadAnglies && forProcessing.charAt(0) == 'L') {
                forProcessing = "P" + forProcessing.substring(1);
            }
            types.add(createTypeFromSignature(forProcessing));
            remainingToProcess = remainingToProcess.substring(endOfSig);
        }
        UnresolvedType[] typeParams = new UnresolvedType[types.size()];
        types.toArray(typeParams);
        return typeParams;
    }

    public static UnresolvedType createUnresolvedParameterizedType(String baseTypeSignature, UnresolvedType[] arguments) {
        StringBuilder parameterizedSig = new StringBuilder();
        parameterizedSig.append(ResolvedType.PARAMETERIZED_TYPE_IDENTIFIER);
        parameterizedSig.append(baseTypeSignature.substring(1, baseTypeSignature.length() - 1));
        if (arguments.length > 0) {
            parameterizedSig.append("<");
            for (UnresolvedType argument : arguments) {
                parameterizedSig.append(argument.getSignature());
            }
            parameterizedSig.append(">");
        }
        parameterizedSig.append(";");
        return createUnresolvedParameterizedType(parameterizedSig.toString(), baseTypeSignature, arguments);
    }
}
