package org.aspectj.weaver;

import java.util.Map;

public class TypeVariableReferenceType extends ReferenceType implements TypeVariableReference {

    private TypeVariable typeVariable;

    public TypeVariableReferenceType(TypeVariable typeVariable, World world) {
        super(typeVariable.getGenericSignature(), typeVariable.getErasureSignature(), world);
        this.typeVariable = typeVariable;
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof TypeVariableReferenceType) {
            return typeVariable == ((TypeVariableReferenceType) other).typeVariable;
        }
        return false;
    }

    @Override
    public int hashCode() {
        return typeVariable.hashCode();
    }

    @Override
    public ReferenceTypeDelegate getDelegate() {
        if (this.delegate == null) {
            ResolvedType resolvedFirstBound = typeVariable.getFirstBound().resolve(world);
            BoundedReferenceTypeDelegate brtd = null;
            if (resolvedFirstBound.isMissing()) {
                brtd = new BoundedReferenceTypeDelegate((ReferenceType) world.resolve(UnresolvedType.OBJECT));
                setDelegate(brtd);
                world.getLint().cantFindType.signal("Unable to find type for generic bound.  Missing type is " + resolvedFirstBound.getName(), getSourceLocation());
            } else {
                brtd = new BoundedReferenceTypeDelegate((ReferenceType) resolvedFirstBound);
                setDelegate(brtd);
            }
        }
        return this.delegate;
    }

    @Override
    public UnresolvedType parameterize(Map<String, UnresolvedType> typeBindings) {
        UnresolvedType ut = typeBindings.get(getName());
        if (ut != null) {
            return world.resolve(ut);
        }
        return this;
    }

    public TypeVariable getTypeVariable() {
        return typeVariable;
    }

    @Override
    public boolean isTypeVariableReference() {
        return true;
    }

    @Override
    public String toString() {
        return typeVariable.getName();
    }

    @Override
    public boolean isGenericWildcard() {
        return false;
    }

    @Override
    public boolean isAnnotation() {
        ReferenceType upper = (ReferenceType) typeVariable.getUpperBound();
        if (upper.isAnnotation()) {
            return true;
        }
        World world = upper.getWorld();
        typeVariable.resolve(world);
        ResolvedType annotationType = ResolvedType.ANNOTATION.resolve(world);
        UnresolvedType[] ifBounds = typeVariable.getSuperInterfaces();
        for (UnresolvedType ifBound : ifBounds) {
            if (((ReferenceType) ifBound).isAnnotation()) {
                return true;
            }
            if (ifBound.equals(annotationType)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public String getSignature() {
        StringBuilder sb = new StringBuilder();
        sb.append("T");
        sb.append(typeVariable.getName());
        sb.append(";");
        return sb.toString();
    }

    public String getTypeVariableName() {
        return typeVariable.getName();
    }

    public ReferenceType getUpperBound() {
        return (ReferenceType) typeVariable.resolve(world).getUpperBound();
    }

    public ResolvedType resolve(World world) {
        typeVariable.resolve(world);
        return this;
    }

    public boolean isTypeVariableResolved() {
        return typeVariable.isResolved;
    }
}
