package org.aspectj.weaver;

import org.aspectj.util.GenericSignature;
import org.aspectj.util.GenericSignature.ClassSignature;
import org.aspectj.util.GenericSignatureParser;
import org.aspectj.weaver.tools.Traceable;

import java.io.DataInputStream;
import java.io.IOException;
import java.util.Map;

public class UnresolvedType implements Traceable, TypeVariableDeclaringElement {

    public static final UnresolvedType[] NONE = new UnresolvedType[0];

    public static final UnresolvedType OBJECT = forSignature("Ljava/lang/Object;");

    public static final UnresolvedType OBJECTARRAY = forSignature("[Ljava/lang/Object;");

    public static final UnresolvedType CLONEABLE = forSignature("Ljava/lang/Cloneable;");

    public static final UnresolvedType SERIALIZABLE = forSignature("Ljava/io/Serializable;");

    public static final UnresolvedType THROWABLE = forSignature("Ljava/lang/Throwable;");

    public static final UnresolvedType RUNTIME_EXCEPTION = forSignature("Ljava/lang/RuntimeException;");

    public static final UnresolvedType ERROR = forSignature("Ljava/lang/Error;");

    public static final UnresolvedType AT_INHERITED = forSignature("Ljava/lang/annotation/Inherited;");

    public static final UnresolvedType AT_RETENTION = forSignature("Ljava/lang/annotation/Retention;");

    public static final UnresolvedType ENUM = forSignature("Ljava/lang/Enum;");

    public static final UnresolvedType ANNOTATION = forSignature("Ljava/lang/annotation/Annotation;");

    public static final UnresolvedType JL_CLASS = forSignature("Ljava/lang/Class;");

    public static final UnresolvedType JAVA_LANG_CLASS_ARRAY = forSignature("[Ljava/lang/Class;");

    public static final UnresolvedType JL_STRING = forSignature("Ljava/lang/String;");

    public static final UnresolvedType JL_EXCEPTION = forSignature("Ljava/lang/Exception;");

    public static final UnresolvedType JAVA_LANG_REFLECT_METHOD = forSignature("Ljava/lang/reflect/Method;");

    public static final UnresolvedType JAVA_LANG_REFLECT_FIELD = forSignature("Ljava/lang/reflect/Field;");

    public static final UnresolvedType JAVA_LANG_REFLECT_CONSTRUCTOR = forSignature("Ljava/lang/reflect/Constructor;");

    public static final UnresolvedType JAVA_LANG_ANNOTATION = forSignature("Ljava/lang/annotation/Annotation;");

    public static final UnresolvedType SUPPRESS_AJ_WARNINGS = forSignature("Lorg/aspectj/lang/annotation/SuppressAjWarnings;");

    public static final UnresolvedType AT_TARGET = forSignature("Ljava/lang/annotation/Target;");

    public static final UnresolvedType SOMETHING = new UnresolvedType("?");

    public static final UnresolvedType[] ARRAY_WITH_JUST_OBJECT = new UnresolvedType[]{OBJECT};

    public static final UnresolvedType JOINPOINT_STATICPART = forSignature("Lorg/aspectj/lang/JoinPoint$StaticPart;");

    public static final UnresolvedType JOINPOINT_ENCLOSINGSTATICPART = forSignature("Lorg/aspectj/lang/JoinPoint$EnclosingStaticPart;");

    public static final UnresolvedType AJC_PRIVILEGED = forSignature("Lorg/aspectj/internal/lang/annotation/ajcPrivileged;");

    public static final UnresolvedType PROCEEDING_JOINPOINT = forSignature("Lorg/aspectj/lang/ProceedingJoinPoint;");

    public static final UnresolvedType BOOLEAN = forPrimitiveType("Z");

    public static final UnresolvedType BYTE = forPrimitiveType("B");

    public static final UnresolvedType CHAR = forPrimitiveType("C");

    public static final UnresolvedType DOUBLE = forPrimitiveType("D");

    public static final UnresolvedType FLOAT = forPrimitiveType("F");

    public static final UnresolvedType INT = forPrimitiveType("I");

    public static final UnresolvedType LONG = forPrimitiveType("J");

    public static final UnresolvedType SHORT = forPrimitiveType("S");

    public static final UnresolvedType VOID = forPrimitiveType("V");

    public static final String MISSING_NAME = "@missing@";

    protected TypeKind typeKind = TypeKind.SIMPLE;

    protected String signature;

    protected String signatureErasure;

    private String packageName;

    private String className;

    protected UnresolvedType[] typeParameters;

    protected TypeVariable[] typeVariables;

    public boolean isPrimitiveType() {
        return typeKind == TypeKind.PRIMITIVE;
    }

    public boolean isVoid() {
        return signature.equals("V");
    }

    public boolean isSimpleType() {
        return typeKind == TypeKind.SIMPLE;
    }

    public boolean isRawType() {
        return typeKind == TypeKind.RAW;
    }

    public boolean isGenericType() {
        return typeKind == TypeKind.GENERIC;
    }

    public boolean isParameterizedType() {
        return typeKind == TypeKind.PARAMETERIZED;
    }

    public boolean isParameterizedOrGenericType() {
        return typeKind == TypeKind.GENERIC || typeKind == TypeKind.PARAMETERIZED;
    }

    public boolean isParameterizedOrRawType() {
        return typeKind == TypeKind.PARAMETERIZED || typeKind == TypeKind.RAW;
    }

    public boolean isTypeVariableReference() {
        return typeKind == TypeKind.TYPE_VARIABLE;
    }

    public boolean isGenericWildcard() {
        return typeKind == TypeKind.WILDCARD;
    }

    public TypeKind getTypekind() {
        return typeKind;
    }

    public final boolean isArray() {
        return signature.length() > 0 && signature.charAt(0) == '[';
    }

    @Override
    public boolean equals(Object other) {
        if (!(other instanceof UnresolvedType)) {
            return false;
        }
        return signature.equals(((UnresolvedType) other).signature);
    }

    @Override
    public int hashCode() {
        return signature.hashCode();
    }

    protected UnresolvedType(String signature) {
        this.signature = signature;
        this.signatureErasure = signature;
    }

    protected UnresolvedType(String signature, String signatureErasure) {
        this.signature = signature;
        this.signatureErasure = signatureErasure;
    }

    public UnresolvedType(String signature, String signatureErasure, UnresolvedType[] typeParams) {
        this.signature = signature;
        this.signatureErasure = signatureErasure;
        this.typeParameters = typeParams;
        if (typeParams != null) {
            this.typeKind = TypeKind.PARAMETERIZED;
        }
    }

    public int getSize() {
        return size;
    }

    private int size = 1;

    public static UnresolvedType forName(String name) {
        return forSignature(nameToSignature(name));
    }

    public static UnresolvedType[] forNames(String[] names) {
        UnresolvedType[] ret = new UnresolvedType[names.length];
        for (int i = 0, len = names.length; i < len; i++) {
            ret[i] = UnresolvedType.forName(names[i]);
        }
        return ret;
    }

    public static UnresolvedType forGenericType(String name, TypeVariable[] tvbs, String genericSig) {
        String sig = nameToSignature(name);
        UnresolvedType ret = UnresolvedType.forSignature(sig);
        ret.typeKind = TypeKind.GENERIC;
        ret.typeVariables = tvbs;
        ret.signatureErasure = sig;
        return ret;
    }

    public static UnresolvedType forGenericTypeSignature(String sig, String declaredGenericSig) {
        UnresolvedType ret = UnresolvedType.forSignature(sig);
        ret.typeKind = TypeKind.GENERIC;
        ClassSignature csig = new GenericSignatureParser().parseAsClassSignature(declaredGenericSig);
        GenericSignature.FormalTypeParameter[] ftps = csig.formalTypeParameters;
        ret.typeVariables = new TypeVariable[ftps.length];
        for (int i = 0; i < ftps.length; i++) {
            GenericSignature.FormalTypeParameter parameter = ftps[i];
            if (parameter.classBound instanceof GenericSignature.ClassTypeSignature) {
                GenericSignature.ClassTypeSignature cts = (GenericSignature.ClassTypeSignature) parameter.classBound;
                ret.typeVariables[i] = new TypeVariable(ftps[i].identifier, UnresolvedType.forSignature(cts.outerType.identifier + ";"));
            } else if (parameter.classBound instanceof GenericSignature.TypeVariableSignature) {
                GenericSignature.TypeVariableSignature tvs = (GenericSignature.TypeVariableSignature) parameter.classBound;
                UnresolvedTypeVariableReferenceType utvrt = new UnresolvedTypeVariableReferenceType(new TypeVariable(tvs.typeVariableName));
                ret.typeVariables[i] = new TypeVariable(ftps[i].identifier, utvrt);
            } else {
                throw new BCException("UnresolvedType.forGenericTypeSignature(): Do not know how to process type variable bound of type '" + parameter.classBound.getClass() + "'.  Full signature is '" + sig + "'");
            }
        }
        ret.signatureErasure = sig;
        ret.signature = ret.signatureErasure;
        return ret;
    }

    public static UnresolvedType forGenericTypeVariables(String sig, TypeVariable[] tVars) {
        UnresolvedType ret = UnresolvedType.forSignature(sig);
        ret.typeKind = TypeKind.GENERIC;
        ret.typeVariables = tVars;
        ret.signatureErasure = sig;
        ret.signature = ret.signatureErasure;
        return ret;
    }

    public static UnresolvedType forRawTypeName(String name) {
        UnresolvedType ret = UnresolvedType.forName(name);
        ret.typeKind = TypeKind.RAW;
        return ret;
    }

    public static UnresolvedType forPrimitiveType(String signature) {
        UnresolvedType ret = new UnresolvedType(signature);
        ret.typeKind = TypeKind.PRIMITIVE;
        if (signature.equals("J") || signature.equals("D")) {
            ret.size = 2;
        } else if (signature.equals("V")) {
            ret.size = 0;
        }
        return ret;
    }

    public static UnresolvedType[] add(UnresolvedType[] types, UnresolvedType end) {
        int len = types.length;
        UnresolvedType[] ret = new UnresolvedType[len + 1];
        System.arraycopy(types, 0, ret, 0, len);
        ret[len] = end;
        return ret;
    }

    public static UnresolvedType[] insert(UnresolvedType start, UnresolvedType[] types) {
        int len = types.length;
        UnresolvedType[] ret = new UnresolvedType[len + 1];
        ret[0] = start;
        System.arraycopy(types, 0, ret, 1, len);
        return ret;
    }

    public static UnresolvedType forSignature(String signature) {
        assert !(signature.startsWith("L") && signature.contains("<"));
        switch (signature.charAt(0)) {
            case 'B':
                return UnresolvedType.BYTE;
            case 'C':
                return UnresolvedType.CHAR;
            case 'D':
                return UnresolvedType.DOUBLE;
            case 'F':
                return UnresolvedType.FLOAT;
            case 'I':
                return UnresolvedType.INT;
            case 'J':
                return UnresolvedType.LONG;
            case 'L':
                return TypeFactory.createTypeFromSignature(signature);
            case 'P':
                return TypeFactory.createTypeFromSignature(signature);
            case 'S':
                return UnresolvedType.SHORT;
            case 'V':
                return UnresolvedType.VOID;
            case 'Z':
                return UnresolvedType.BOOLEAN;
            case '[':
                return TypeFactory.createTypeFromSignature(signature);
            case '+':
                return TypeFactory.createTypeFromSignature(signature);
            case '-':
                return TypeFactory.createTypeFromSignature(signature);
            case '?':
                return TypeFactory.createTypeFromSignature(signature);
            case 'T':
                return TypeFactory.createTypeFromSignature(signature);
            default:
                throw new BCException("Bad type signature " + signature);
        }
    }

    public static UnresolvedType[] forSignatures(String[] sigs) {
        UnresolvedType[] ret = new UnresolvedType[sigs.length];
        for (int i = 0, len = sigs.length; i < len; i++) {
            ret[i] = UnresolvedType.forSignature(sigs[i]);
        }
        return ret;
    }

    public String getName() {
        return signatureToName(signature);
    }

    public String getSimpleName() {
        String name = getRawName();
        int lastDot = name.lastIndexOf('.');
        if (lastDot != -1) {
            name = name.substring(lastDot + 1);
        }
        if (isParameterizedType()) {
            StringBuilder sb = new StringBuilder(name);
            sb.append("<");
            for (int i = 0; i < (typeParameters.length - 1); i++) {
                sb.append(typeParameters[i].getSimpleName());
                sb.append(",");
            }
            sb.append(typeParameters[typeParameters.length - 1].getSimpleName());
            sb.append(">");
            name = sb.toString();
        }
        return name;
    }

    public String getRawName() {
        return signatureToName((signatureErasure == null ? signature : signatureErasure));
    }

    public String getBaseName() {
        String name = getName();
        if (isParameterizedType() || isGenericType()) {
            if (typeParameters == null) {
                return name;
            } else {
                return name.substring(0, name.indexOf("<"));
            }
        } else {
            return name;
        }
    }

    public String getSimpleBaseName() {
        String name = getBaseName();
        int lastDot = name.lastIndexOf('.');
        if (lastDot != -1) {
            name = name.substring(lastDot + 1);
        }
        return name;
    }

    public static String[] getNames(UnresolvedType[] types) {
        String[] ret = new String[types.length];
        for (int i = 0, len = types.length; i < len; i++) {
            ret[i] = types[i].getName();
        }
        return ret;
    }

    public String getSignature() {
        return signature;
    }

    public String getErasureSignature() {
        if (signatureErasure == null) {
            return signature;
        }
        return signatureErasure;
    }

    private boolean needsModifiableDelegate = false;

    public boolean needsModifiableDelegate() {
        return needsModifiableDelegate;
    }

    public void setNeedsModifiableDelegate(boolean b) {
        this.needsModifiableDelegate = b;
    }

    public UnresolvedType getRawType() {
        return UnresolvedType.forSignature(getErasureSignature());
    }

    public UnresolvedType getOutermostType() {
        if (isArray() || isPrimitiveType()) {
            return this;
        }
        String sig = getErasureSignature();
        int dollar = sig.indexOf('$');
        if (dollar != -1) {
            return UnresolvedType.forSignature(sig.substring(0, dollar) + ';');
        } else {
            return this;
        }
    }

    public UnresolvedType getComponentType() {
        if (isArray()) {
            return forSignature(signature.substring(1));
        } else {
            return null;
        }
    }

    @Override
    public String toString() {
        return getName();
    }

    public String toDebugString() {
        return getName();
    }

    public ResolvedType resolve(World world) {
        return world.resolve(this);
    }

    private static String signatureToName(String signature) {
        switch (signature.charAt(0)) {
            case 'B':
                return "byte";
            case 'C':
                return "char";
            case 'D':
                return "double";
            case 'F':
                return "float";
            case 'I':
                return "int";
            case 'J':
                return "long";
            case 'L':
                String name = signature.substring(1, signature.length() - 1).replace('/', '.');
                return name;
            case 'T':
                StringBuilder nameBuff2 = new StringBuilder();
                int colon = signature.indexOf(";");
                String tvarName = signature.substring(1, colon);
                nameBuff2.append(tvarName);
                return nameBuff2.toString();
            case 'P':
                StringBuilder nameBuff = new StringBuilder();
                int paramNestLevel = 0;
                for (int i = 1; i < signature.length(); i++) {
                    char c = signature.charAt(i);
                    switch (c) {
                        case '/':
                            nameBuff.append('.');
                            break;
                        case '<':
                            nameBuff.append("<");
                            paramNestLevel++;
                            StringBuilder innerBuff = new StringBuilder();
                            while (paramNestLevel > 0) {
                                c = signature.charAt(++i);
                                if (c == '<') {
                                    paramNestLevel++;
                                }
                                if (c == '>') {
                                    paramNestLevel--;
                                }
                                if (paramNestLevel > 0) {
                                    innerBuff.append(c);
                                }
                                if (c == ';' && paramNestLevel == 1) {
                                    nameBuff.append(signatureToName(innerBuff.toString()));
                                    if (signature.charAt(i + 1) != '>') {
                                        nameBuff.append(',');
                                    }
                                    innerBuff = new StringBuilder();
                                }
                            }
                            nameBuff.append(">");
                            break;
                        case ';':
                            break;
                        default:
                            nameBuff.append(c);
                    }
                }
                return nameBuff.toString();
            case 'S':
                return "short";
            case 'V':
                return "void";
            case 'Z':
                return "boolean";
            case '[':
                return signatureToName(signature.substring(1, signature.length())) + "[]";
            case '+':
                return "? extends " + signatureToName(signature.substring(1, signature.length()));
            case '-':
                return "? super " + signatureToName(signature.substring(1, signature.length()));
            case '*':
                return "?";
            default:
                throw new BCException("Bad type signature: " + signature);
        }
    }

    private static String nameToSignature(String name) {
        int len = name.length();
        if (len < 8) {
            if (name.equals("int")) {
                return "I";
            }
            if (name.equals("void")) {
                return "V";
            }
            if (name.equals("long")) {
                return "J";
            }
            if (name.equals("boolean")) {
                return "Z";
            }
            if (name.equals("double")) {
                return "D";
            }
            if (name.equals("float")) {
                return "F";
            }
            if (name.equals("byte")) {
                return "B";
            }
            if (name.equals("short")) {
                return "S";
            }
            if (name.equals("char")) {
                return "C";
            }
            if (name.equals("?")) {
                return name;
            }
        }
        if (len == 0) {
            throw new BCException("Bad type name: " + name);
        }
        if (name.endsWith("[]")) {
            return "[" + nameToSignature(name.substring(0, name.length() - 2));
        }
        if (name.charAt(0) == '[') {
            return name.replace('.', '/');
        }
        if (!name.contains("<")) {
            return new StringBuilder("L").append(name.replace('.', '/')).append(';').toString();
        } else {
            StringBuilder nameBuff = new StringBuilder();
            int nestLevel = 0;
            nameBuff.append("P");
            for (int i = 0; i < len; i++) {
                char c = name.charAt(i);
                switch (c) {
                    case '.':
                        nameBuff.append('/');
                        break;
                    case '<':
                        nameBuff.append("<");
                        nestLevel++;
                        StringBuilder innerBuff = new StringBuilder();
                        while (nestLevel > 0) {
                            c = name.charAt(++i);
                            if (c == '<') {
                                nestLevel++;
                            } else if (c == '>') {
                                nestLevel--;
                            }
                            if (c == ',' && nestLevel == 1) {
                                nameBuff.append(nameToSignature(innerBuff.toString()));
                                innerBuff = new StringBuilder();
                            } else {
                                if (nestLevel > 0) {
                                    innerBuff.append(c);
                                }
                            }
                        }
                        nameBuff.append(nameToSignature(innerBuff.toString()));
                        nameBuff.append('>');
                        break;
                    default:
                        nameBuff.append(c);
                }
            }
            nameBuff.append(";");
            return nameBuff.toString();
        }
    }

    public final void write(CompressingDataOutputStream s) throws IOException {
        s.writeUTF(getSignature());
    }

    public static UnresolvedType read(DataInputStream s) throws IOException {
        String sig = s.readUTF();
        if (sig.equals(MISSING_NAME)) {
            return ResolvedType.MISSING;
        } else {
            return UnresolvedType.forSignature(sig);
        }
    }

    public String getNameAsIdentifier() {
        return getName().replace('.', '_');
    }

    public String getPackageNameAsIdentifier() {
        String name = getName();
        int index = name.lastIndexOf('.');
        if (index == -1) {
            return "";
        } else {
            return name.substring(0, index).replace('.', '_');
        }
    }

    public UnresolvedType[] getTypeParameters() {
        return typeParameters == null ? UnresolvedType.NONE : typeParameters;
    }

    public TypeVariable[] getTypeVariables() {
        return typeVariables;
    }

    public static class TypeKind {

        public final static TypeKind PRIMITIVE = new TypeKind("primitive");

        public final static TypeKind SIMPLE = new TypeKind("simple");

        public final static TypeKind RAW = new TypeKind("raw");

        public final static TypeKind GENERIC = new TypeKind("generic");

        public final static TypeKind PARAMETERIZED = new TypeKind("parameterized");

        public final static TypeKind TYPE_VARIABLE = new TypeKind("type_variable");

        public final static TypeKind WILDCARD = new TypeKind("wildcard");

        @Override
        public String toString() {
            return type;
        }

        private TypeKind(String type) {
            this.type = type;
        }

        private final String type;
    }

    @Override
    public TypeVariable getTypeVariableNamed(String name) {
        TypeVariable[] vars = getTypeVariables();
        if (vars == null || vars.length == 0) {
            return null;
        }
        for (TypeVariable aVar : vars) {
            if (aVar.getName().equals(name)) {
                return aVar;
            }
        }
        return null;
    }

    @Override
    public String toTraceString() {
        return getClass().getName() + "[" + getName() + "]";
    }

    public UnresolvedType parameterize(Map<String, UnresolvedType> typeBindings) {
        throw new UnsupportedOperationException("unable to parameterize unresolved type: " + signature);
    }

    public String getClassName() {
        if (className == null) {
            String name = getName();
            if (name.contains("<")) {
                name = name.substring(0, name.indexOf("<"));
            }
            int index = name.lastIndexOf('.');
            if (index == -1) {
                className = name;
            } else {
                className = name.substring(index + 1);
            }
        }
        return className;
    }

    public String getPackageName() {
        if (packageName == null) {
            String name = getName();
            int angly = name.indexOf('<');
            if (angly != -1) {
                name = name.substring(0, angly);
            }
            int index = name.lastIndexOf('.');
            if (index == -1) {
                packageName = "";
            } else {
                packageName = name.substring(0, index);
            }
        }
        return packageName;
    }

    public static void writeArray(UnresolvedType[] types, CompressingDataOutputStream stream) throws IOException {
        int len = types.length;
        stream.writeShort(len);
        for (UnresolvedType type : types) {
            type.write(stream);
        }
    }

    public static UnresolvedType[] readArray(DataInputStream s) throws IOException {
        int len = s.readShort();
        if (len == 0) {
            return UnresolvedType.NONE;
        }
        UnresolvedType[] types = new UnresolvedType[len];
        for (int i = 0; i < len; i++) {
            types[i] = UnresolvedType.read(s);
        }
        return types;
    }

    public static UnresolvedType makeArray(UnresolvedType base, int dims) {
        StringBuilder sig = new StringBuilder();
        for (int i = 0; i < dims; i++) {
            sig.append("[");
        }
        sig.append(base.getSignature());
        return UnresolvedType.forSignature(sig.toString());
    }
}
