package org.aspectj.weaver;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.IMessage.Kind;
import org.aspectj.bridge.IMessageHandler;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.Message;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.bridge.context.PinpointingMessageHandler;
import org.aspectj.util.IStructureModel;
import org.aspectj.weaver.ResolvedType.Primitive;
import org.aspectj.weaver.UnresolvedType.TypeKind;
import org.aspectj.weaver.patterns.Declare;
import org.aspectj.weaver.patterns.DeclareAnnotation;
import org.aspectj.weaver.patterns.DeclareParents;
import org.aspectj.weaver.patterns.DeclarePrecedence;
import org.aspectj.weaver.patterns.DeclareSoft;
import org.aspectj.weaver.patterns.DeclareTypeErrorOrWarning;
import org.aspectj.weaver.patterns.Pointcut;
import org.aspectj.weaver.patterns.TypePattern;
import org.aspectj.weaver.tools.PointcutDesignatorHandler;
import org.aspectj.weaver.tools.Trace;
import org.aspectj.weaver.tools.TraceFactory;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.WeakHashMap;

public abstract class World implements Dump.INode {

    private IMessageHandler messageHandler = IMessageHandler.SYSTEM_ERR;

    private ICrossReferenceHandler xrefHandler = null;

    private TypeVariableDeclaringElement typeVariableLookupScope;

    protected TypeMap typeMap = new TypeMap(this);

    private Set<PointcutDesignatorHandler> pointcutDesignators;

    public static boolean createInjarHierarchy = true;

    private final AspectPrecedenceCalculator precedenceCalculator;

    private final CrosscuttingMembersSet crosscuttingMembersSet = new CrosscuttingMembersSet(this);

    private IStructureModel model = null;

    private Lint lint = new Lint(this);

    private boolean XnoInline;

    private boolean XlazyTjp;

    private boolean XhasMember = false;

    private boolean Xpinpoint = false;

    private boolean behaveInJava5Way = false;

    private boolean timing = false;

    private boolean timingPeriodically = true;

    private boolean incrementalCompileCouldFollow = false;

    public static final RuntimeVersion RUNTIME_LEVEL_DEFAULT = RuntimeVersion.V1_5;

    private RuntimeVersion targetAspectjRuntimeLevel = RUNTIME_LEVEL_DEFAULT;

    private boolean optionalJoinpoint_ArrayConstruction = false;

    private boolean optionalJoinpoint_Synchronization = false;

    private boolean addSerialVerUID = false;

    private Properties extraConfiguration = null;

    private boolean checkedAdvancedConfiguration = false;

    private boolean synchronizationPointcutsInUse = false;

    private boolean runMinimalMemory = false;

    private boolean transientTjpFields = false;

    private boolean runMinimalMemorySet = false;

    private boolean shouldPipelineCompilation = true;

    private boolean shouldGenerateStackMaps = false;

    protected boolean bcelRepositoryCaching = xsetBCEL_REPOSITORY_CACHING_DEFAULT.equalsIgnoreCase("true");

    private boolean fastMethodPacking = false;

    private int itdVersion = 2;

    private boolean minimalModel = true;

    private boolean useFinal = true;

    private boolean targettingRuntime1_6_10 = false;

    private boolean completeBinaryTypes = false;

    private boolean overWeaving = false;

    private static boolean systemPropertyOverWeaving = false;

    public boolean forDEBUG_structuralChangesCode = false;

    public boolean forDEBUG_bridgingCode = false;

    public boolean optimizedMatching = true;

    public boolean generateNewLvts = true;

    protected long timersPerJoinpoint = 25000;

    protected long timersPerType = 250;

    public int infoMessagesEnabled = 0;

    private static Trace trace = TraceFactory.getTraceFactory().getTrace(World.class);

    private boolean errorThreshold;

    private boolean warningThreshold;

    private List<RuntimeException> dumpState_cantFindTypeExceptions = null;

    static {
        try {
            String value = System.getProperty("aspectj.overweaving", "false");
            if (value.equalsIgnoreCase("true")) {
                System.out.println("ASPECTJ: aspectj.overweaving=true: overweaving switched ON");
                systemPropertyOverWeaving = true;
            }
        } catch (Throwable t) {
            System.err.println("ASPECTJ: Unable to read system properties");
            t.printStackTrace();
        }
    }

    public final Primitive BYTE = new Primitive("B", 1, 0);

    public final Primitive CHAR = new Primitive("C", 1, 1);

    public final Primitive DOUBLE = new Primitive("D", 2, 2);

    public final Primitive FLOAT = new Primitive("F", 1, 3);

    public final Primitive INT = new Primitive("I", 1, 4);

    public final Primitive LONG = new Primitive("J", 2, 5);

    public final Primitive SHORT = new Primitive("S", 1, 6);

    public final Primitive BOOLEAN = new Primitive("Z", 1, 7);

    public final Primitive VOID = new Primitive("V", 0, 8);

    protected World() {
        super();
        typeMap.put("B", BYTE);
        typeMap.put("S", SHORT);
        typeMap.put("I", INT);
        typeMap.put("J", LONG);
        typeMap.put("F", FLOAT);
        typeMap.put("D", DOUBLE);
        typeMap.put("C", CHAR);
        typeMap.put("Z", BOOLEAN);
        typeMap.put("V", VOID);
        precedenceCalculator = new AspectPrecedenceCalculator(this);
    }

    @Override
    public void accept(Dump.IVisitor visitor) {
        visitor.visitObject("Shadow mungers:");
        visitor.visitList(crosscuttingMembersSet.getShadowMungers());
        visitor.visitObject("Type mungers:");
        visitor.visitList(crosscuttingMembersSet.getTypeMungers());
        visitor.visitObject("Late Type mungers:");
        visitor.visitList(crosscuttingMembersSet.getLateTypeMungers());
        if (dumpState_cantFindTypeExceptions != null) {
            visitor.visitObject("Cant find type problems:");
            visitor.visitList(dumpState_cantFindTypeExceptions);
            dumpState_cantFindTypeExceptions = null;
        }
    }

    public ResolvedType resolve(UnresolvedType ty) {
        return resolve(ty, false);
    }

    public ResolvedType resolve(UnresolvedType ty, ISourceLocation isl) {
        ResolvedType ret = resolve(ty, true);
        if (ResolvedType.isMissing(ty)) {
            getLint().cantFindType.signal(WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE, ty.getName()), isl);
        }
        return ret;
    }

    public ResolvedType[] resolve(UnresolvedType[] types) {
        if (types == null) {
            return ResolvedType.NONE;
        }
        ResolvedType[] ret = new ResolvedType[types.length];
        for (int i = 0; i < types.length; i++) {
            ret[i] = resolve(types[i]);
        }
        return ret;
    }

    public ResolvedType resolve(UnresolvedType ty, boolean allowMissing) {
        if (ty instanceof ResolvedType) {
            ResolvedType rty = (ResolvedType) ty;
            rty = resolve(rty);
            if (!rty.isTypeVariableReference() || ((TypeVariableReferenceType) rty).isTypeVariableResolved()) {
                return rty;
            }
        }
        if (ty.isTypeVariableReference()) {
            return ty.resolve(this);
        }
        String signature = ty.getSignature();
        ResolvedType ret = typeMap.get(signature);
        if (ret != null) {
            ret.world = this;
            return ret;
        } else if (signature.equals("?") || signature.equals("*")) {
            ResolvedType something = getWildcard();
            typeMap.put("?", something);
            return something;
        }
        synchronized (buildingTypeLock) {
            if (ty.isArray()) {
                ResolvedType componentType = resolve(ty.getComponentType(), allowMissing);
                ret = new ArrayReferenceType(signature, "[" + componentType.getErasureSignature(), this, componentType);
            } else {
                ret = resolveToReferenceType(ty, allowMissing);
                if (!allowMissing && ret.isMissing()) {
                    ret = handleRequiredMissingTypeDuringResolution(ty);
                }
                if (completeBinaryTypes) {
                    completeBinaryType(ret);
                }
            }
        }
        ResolvedType result = typeMap.get(signature);
        if (result == null && !ret.isMissing()) {
            ret = ensureRawTypeIfNecessary(ret);
            typeMap.put(signature, ret);
            return ret;
        }
        if (result == null) {
            return ret;
        } else {
            return result;
        }
    }

    private Object buildingTypeLock = new Object();

    private BoundedReferenceType wildcard;

    private BoundedReferenceType getWildcard() {
        if (wildcard == null) {
            wildcard = new BoundedReferenceType(this);
        }
        return wildcard;
    }

    protected void completeBinaryType(ResolvedType ret) {
    }

    public boolean isLocallyDefined(String classname) {
        return false;
    }

    private ResolvedType handleRequiredMissingTypeDuringResolution(UnresolvedType ty) {
        if (dumpState_cantFindTypeExceptions == null) {
            dumpState_cantFindTypeExceptions = new ArrayList<>();
        }
        if (dumpState_cantFindTypeExceptions.size() < 100) {
            dumpState_cantFindTypeExceptions.add(new RuntimeException("Can't find type " + ty.getName()));
        }
        return new MissingResolvedTypeWithKnownSignature(ty.getSignature(), this);
    }

    public ResolvedType resolve(ResolvedType ty) {
        if (ty.isTypeVariableReference()) {
            return ty;
        }
        ResolvedType resolved = typeMap.get(ty.getSignature());
        if (resolved == null) {
            resolved = ensureRawTypeIfNecessary(ty);
            typeMap.put(ty.getSignature(), resolved);
            resolved = ty;
        }
        resolved.world = this;
        return resolved;
    }

    private ResolvedType ensureRawTypeIfNecessary(ResolvedType type) {
        if (!isInJava5Mode() || type.isRawType()) {
            return type;
        }
        if (type instanceof ReferenceType && ((ReferenceType) type).getDelegate() != null && type.isGenericType()) {
            ReferenceType rawType = new ReferenceType(type.getSignature(), this);
            rawType.typeKind = UnresolvedType.TypeKind.RAW;
            ReferenceTypeDelegate delegate = ((ReferenceType) type).getDelegate();
            rawType.setDelegate(delegate);
            rawType.setGenericType((ReferenceType) type);
            return rawType;
        }
        return type;
    }

    public ResolvedType resolve(String name) {
        ResolvedType ret = resolve(UnresolvedType.forName(name));
        return ret;
    }

    public ReferenceType resolveToReferenceType(String name) {
        return (ReferenceType) resolve(name);
    }

    public ResolvedType resolve(String name, boolean allowMissing) {
        return resolve(UnresolvedType.forName(name), allowMissing);
    }

    private final ResolvedType resolveToReferenceType(UnresolvedType ty, boolean allowMissing) {
        if (ty.isParameterizedType()) {
            ResolvedType rt = resolveGenericTypeFor(ty, allowMissing);
            if (rt.isMissing()) {
                return rt;
            }
            ReferenceType genericType = (ReferenceType) rt;
            ReferenceType parameterizedType = TypeFactory.createParameterizedType(genericType, ty.typeParameters, this);
            return parameterizedType;
        } else if (ty.isGenericType()) {
            ResolvedType rt = resolveGenericTypeFor(ty, false);
            if (rt.isMissing()) {
                return rt;
            }
            ReferenceType genericType = (ReferenceType) rt;
            if (rt.isMissing()) {
                return rt;
            }
            return genericType;
        } else if (ty.isGenericWildcard()) {
            return resolveGenericWildcardFor((WildcardedUnresolvedType) ty);
        } else {
            String erasedSignature = ty.getErasureSignature();
            ReferenceType simpleOrRawType = new ReferenceType(erasedSignature, this);
            if (ty.needsModifiableDelegate()) {
                simpleOrRawType.setNeedsModifiableDelegate(true);
            }
            ReferenceTypeDelegate delegate = resolveDelegate(simpleOrRawType);
            if (delegate == null) {
                return new MissingResolvedTypeWithKnownSignature(ty.getSignature(), erasedSignature, this);
            }
            if (delegate.isGeneric() && behaveInJava5Way) {
                simpleOrRawType.typeKind = TypeKind.RAW;
                if (simpleOrRawType.hasNewInterfaces()) {
                    throw new IllegalStateException("Simple type promoted forced to raw, but it had new interfaces/superclass.  Type is " + simpleOrRawType.getName());
                }
                ReferenceType genericType = makeGenericTypeFrom(delegate, simpleOrRawType);
                simpleOrRawType.setDelegate(delegate);
                genericType.setDelegate(delegate);
                simpleOrRawType.setGenericType(genericType);
                return simpleOrRawType;
            } else {
                simpleOrRawType.setDelegate(delegate);
                return simpleOrRawType;
            }
        }
    }

    public ResolvedType resolveGenericTypeFor(UnresolvedType anUnresolvedType, boolean allowMissing) {
        String rawSignature = anUnresolvedType.getRawType().getSignature();
        ResolvedType rawType = typeMap.get(rawSignature);
        if (rawType == null) {
            rawType = resolve(UnresolvedType.forSignature(rawSignature), allowMissing);
            typeMap.put(rawSignature, rawType);
        }
        if (rawType.isMissing()) {
            return rawType;
        }
        ResolvedType genericType = rawType.getGenericType();
        if (rawType.isSimpleType() && (anUnresolvedType.typeParameters == null || anUnresolvedType.typeParameters.length == 0)) {
            rawType.world = this;
            return rawType;
        }
        if (genericType != null) {
            genericType.world = this;
            return genericType;
        } else {
            ReferenceTypeDelegate delegate = resolveDelegate((ReferenceType) rawType);
            ReferenceType genericRefType = makeGenericTypeFrom(delegate, ((ReferenceType) rawType));
            ((ReferenceType) rawType).setGenericType(genericRefType);
            genericRefType.setDelegate(delegate);
            ((ReferenceType) rawType).setDelegate(delegate);
            return genericRefType;
        }
    }

    private ReferenceType makeGenericTypeFrom(ReferenceTypeDelegate delegate, ReferenceType rawType) {
        String genericSig = delegate.getDeclaredGenericSignature();
        if (genericSig != null) {
            return new ReferenceType(UnresolvedType.forGenericTypeSignature(rawType.getSignature(), delegate.getDeclaredGenericSignature()), this);
        } else {
            return new ReferenceType(UnresolvedType.forGenericTypeVariables(rawType.getSignature(), delegate.getTypeVariables()), this);
        }
    }

    private ReferenceType resolveGenericWildcardFor(WildcardedUnresolvedType aType) {
        BoundedReferenceType ret = null;
        if (aType.isExtends()) {
            ResolvedType resolvedUpperBound = resolve(aType.getUpperBound());
            if (resolvedUpperBound.isMissing()) {
                return getWildcard();
            }
            ret = new BoundedReferenceType((ReferenceType) resolvedUpperBound, true, this);
        } else if (aType.isSuper()) {
            ResolvedType resolvedLowerBound = resolve(aType.getLowerBound());
            if (resolvedLowerBound.isMissing()) {
                return getWildcard();
            }
            ret = new BoundedReferenceType((ReferenceType) resolvedLowerBound, false, this);
        } else {
            ret = getWildcard();
        }
        return ret;
    }

    protected abstract ReferenceTypeDelegate resolveDelegate(ReferenceType ty);

    public ResolvedType getCoreType(UnresolvedType tx) {
        ResolvedType coreTy = resolve(tx, true);
        if (coreTy.isMissing()) {
            MessageUtil.error(messageHandler, WeaverMessages.format(WeaverMessages.CANT_FIND_CORE_TYPE, tx.getName()));
        }
        return coreTy;
    }

    public ReferenceType lookupOrCreateName(UnresolvedType ty) {
        String signature = ty.getSignature();
        ReferenceType ret = lookupBySignature(signature);
        if (ret == null) {
            ret = ReferenceType.fromTypeX(ty, this);
            typeMap.put(signature, ret);
        }
        return ret;
    }

    public ReferenceType lookupBySignature(String signature) {
        return (ReferenceType) typeMap.get(signature);
    }

    public ResolvedMember resolve(Member member) {
        ResolvedType declaring = member.getDeclaringType().resolve(this);
        if (declaring.isRawType()) {
            declaring = declaring.getGenericType();
        }
        ResolvedMember ret;
        if (member.getKind() == Member.FIELD) {
            ret = declaring.lookupField(member);
        } else {
            ret = declaring.lookupMethod(member);
        }
        if (ret != null) {
            return ret;
        }
        return declaring.lookupSyntheticMember(member);
    }

    private boolean allLintIgnored = false;

    public void setAllLintIgnored() {
        allLintIgnored = true;
    }

    public boolean areAllLintIgnored() {
        return allLintIgnored;
    }

    public abstract IWeavingSupport getWeavingSupport();

    public final Advice createAdviceMunger(AdviceKind kind, Pointcut p, Member signature, int extraParameterFlags, IHasSourceLocation loc, ResolvedType declaringAspect) {
        AjAttribute.AdviceAttribute attribute = new AjAttribute.AdviceAttribute(kind, p, extraParameterFlags, loc.getStart(), loc.getEnd(), loc.getSourceContext());
        return getWeavingSupport().createAdviceMunger(attribute, p, signature, declaringAspect);
    }

    public int compareByPrecedence(ResolvedType aspect1, ResolvedType aspect2) {
        return precedenceCalculator.compareByPrecedence(aspect1, aspect2);
    }

    public Integer getPrecedenceIfAny(ResolvedType aspect1, ResolvedType aspect2) {
        return precedenceCalculator.getPrecedenceIfAny(aspect1, aspect2);
    }

    public int compareByPrecedenceAndHierarchy(ResolvedType aspect1, ResolvedType aspect2) {
        return precedenceCalculator.compareByPrecedenceAndHierarchy(aspect1, aspect2);
    }

    public IMessageHandler getMessageHandler() {
        return messageHandler;
    }

    public void setMessageHandler(IMessageHandler messageHandler) {
        if (this.isInPinpointMode()) {
            this.messageHandler = new PinpointingMessageHandler(messageHandler);
        } else {
            this.messageHandler = messageHandler;
        }
    }

    public void showMessage(Kind kind, String message, ISourceLocation loc1, ISourceLocation loc2) {
        if (loc1 != null) {
            messageHandler.handleMessage(new Message(message, kind, null, loc1));
            if (loc2 != null) {
                messageHandler.handleMessage(new Message(message, kind, null, loc2));
            }
        } else {
            messageHandler.handleMessage(new Message(message, kind, null, loc2));
        }
    }

    public void setCrossReferenceHandler(ICrossReferenceHandler xrefHandler) {
        this.xrefHandler = xrefHandler;
    }

    public ICrossReferenceHandler getCrossReferenceHandler() {
        return xrefHandler;
    }

    public void setTypeVariableLookupScope(TypeVariableDeclaringElement scope) {
        typeVariableLookupScope = scope;
    }

    public TypeVariableDeclaringElement getTypeVariableLookupScope() {
        return typeVariableLookupScope;
    }

    public List<DeclareParents> getDeclareParents() {
        return crosscuttingMembersSet.getDeclareParents();
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnTypes() {
        return crosscuttingMembersSet.getDeclareAnnotationOnTypes();
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnFields() {
        return crosscuttingMembersSet.getDeclareAnnotationOnFields();
    }

    public List<DeclareAnnotation> getDeclareAnnotationOnMethods() {
        return crosscuttingMembersSet.getDeclareAnnotationOnMethods();
    }

    public List<DeclareTypeErrorOrWarning> getDeclareTypeEows() {
        return crosscuttingMembersSet.getDeclareTypeEows();
    }

    public List<DeclareSoft> getDeclareSoft() {
        return crosscuttingMembersSet.getDeclareSofts();
    }

    public CrosscuttingMembersSet getCrosscuttingMembersSet() {
        return crosscuttingMembersSet;
    }

    public IStructureModel getModel() {
        return model;
    }

    public void setModel(IStructureModel model) {
        this.model = model;
    }

    public Lint getLint() {
        return lint;
    }

    public void setLint(Lint lint) {
        this.lint = lint;
    }

    public boolean isXnoInline() {
        return XnoInline;
    }

    public void setXnoInline(boolean xnoInline) {
        XnoInline = xnoInline;
    }

    public boolean isXlazyTjp() {
        return XlazyTjp;
    }

    public void setXlazyTjp(boolean b) {
        XlazyTjp = b;
    }

    public boolean isHasMemberSupportEnabled() {
        return XhasMember;
    }

    public void setXHasMemberSupportEnabled(boolean b) {
        XhasMember = b;
    }

    public boolean isInPinpointMode() {
        return Xpinpoint;
    }

    public void setPinpointMode(boolean b) {
        Xpinpoint = b;
    }

    public boolean useFinal() {
        return useFinal;
    }

    public boolean isMinimalModel() {
        ensureAdvancedConfigurationProcessed();
        return minimalModel;
    }

    public boolean isTargettingRuntime1_6_10() {
        ensureAdvancedConfigurationProcessed();
        return targettingRuntime1_6_10;
    }

    public void setBehaveInJava5Way(boolean b) {
        behaveInJava5Way = b;
    }

    public void setTiming(boolean timersOn, boolean reportPeriodically) {
        timing = timersOn;
        timingPeriodically = reportPeriodically;
    }

    public void setErrorAndWarningThreshold(boolean errorThreshold, boolean warningThreshold) {
        this.errorThreshold = errorThreshold;
        this.warningThreshold = warningThreshold;
    }

    public boolean isIgnoringUnusedDeclaredThrownException() {
        return errorThreshold || warningThreshold;
    }

    public void performExtraConfiguration(String config) {
        if (config == null) {
            return;
        }
        extraConfiguration = new Properties();
        int pos = -1;
        while ((pos = config.indexOf(",")) != -1) {
            String nvpair = config.substring(0, pos);
            int pos2 = nvpair.indexOf("=");
            if (pos2 != -1) {
                String n = nvpair.substring(0, pos2);
                String v = nvpair.substring(pos2 + 1);
                extraConfiguration.setProperty(n, v);
            }
            config = config.substring(pos + 1);
        }
        if (config.length() > 0) {
            int pos2 = config.indexOf("=");
            if (pos2 != -1) {
                String n = config.substring(0, pos2);
                String v = config.substring(pos2 + 1);
                extraConfiguration.setProperty(n, v);
            }
        }
        ensureAdvancedConfigurationProcessed();
    }

    public boolean areInfoMessagesEnabled() {
        if (infoMessagesEnabled == 0) {
            infoMessagesEnabled = (messageHandler.isIgnoring(IMessage.INFO) ? 1 : 2);
        }
        return infoMessagesEnabled == 2;
    }

    public Properties getExtraConfiguration() {
        return extraConfiguration;
    }

    public final static String xsetAVOID_FINAL = "avoidFinal";

    public final static String xsetWEAVE_JAVA_PACKAGES = "weaveJavaPackages";

    public final static String xsetWEAVE_JAVAX_PACKAGES = "weaveJavaxPackages";

    public final static String xsetCAPTURE_ALL_CONTEXT = "captureAllContext";

    public final static String xsetRUN_MINIMAL_MEMORY = "runMinimalMemory";

    public final static String xsetDEBUG_STRUCTURAL_CHANGES_CODE = "debugStructuralChangesCode";

    public final static String xsetDEBUG_BRIDGING = "debugBridging";

    public final static String xsetTRANSIENT_TJP_FIELDS = "makeTjpFieldsTransient";

    public final static String xsetBCEL_REPOSITORY_CACHING = "bcelRepositoryCaching";

    public final static String xsetPIPELINE_COMPILATION = "pipelineCompilation";

    public final static String xsetGENERATE_STACKMAPS = "generateStackMaps";

    public final static String xsetPIPELINE_COMPILATION_DEFAULT = "true";

    public final static String xsetCOMPLETE_BINARY_TYPES = "completeBinaryTypes";

    public final static String xsetCOMPLETE_BINARY_TYPES_DEFAULT = "false";

    public final static String xsetTYPE_DEMOTION = "typeDemotion";

    public final static String xsetTYPE_DEMOTION_DEBUG = "typeDemotionDebug";

    public final static String xsetTYPE_REFS = "useWeakTypeRefs";

    public final static String xsetBCEL_REPOSITORY_CACHING_DEFAULT = "true";

    public final static String xsetFAST_PACK_METHODS = "fastPackMethods";

    public final static String xsetOVERWEAVING = "overWeaving";

    public final static String xsetOPTIMIZED_MATCHING = "optimizedMatching";

    public final static String xsetTIMERS_PER_JOINPOINT = "timersPerJoinpoint";

    public final static String xsetTIMERS_PER_FASTMATCH_CALL = "timersPerFastMatchCall";

    public final static String xsetITD_VERSION = "itdVersion";

    public final static String xsetITD_VERSION_ORIGINAL = "1";

    public final static String xsetITD_VERSION_2NDGEN = "2";

    public final static String xsetITD_VERSION_DEFAULT = xsetITD_VERSION_2NDGEN;

    public final static String xsetMINIMAL_MODEL = "minimalModel";

    public final static String xsetTARGETING_RUNTIME_1610 = "targetRuntime1_6_10";

    public final static String xsetGENERATE_NEW_LVTS = "generateNewLocalVariableTables";

    public boolean isInJava5Mode() {
        return behaveInJava5Way;
    }

    public boolean isTimingEnabled() {
        return timing;
    }

    public void setTargetAspectjRuntimeLevel(String s) {
        targetAspectjRuntimeLevel = RuntimeVersion.getVersionFor(s);
    }

    public void setOptionalJoinpoints(String jps) {
        if (jps == null) {
            return;
        }
        if (jps.contains("arrayconstruction")) {
            optionalJoinpoint_ArrayConstruction = true;
        }
        if (jps.contains("synchronization")) {
            optionalJoinpoint_Synchronization = true;
        }
    }

    public boolean isJoinpointArrayConstructionEnabled() {
        return optionalJoinpoint_ArrayConstruction;
    }

    public boolean isJoinpointSynchronizationEnabled() {
        return optionalJoinpoint_Synchronization;
    }

    public RuntimeVersion getTargetAspectjRuntimeLevel() {
        return targetAspectjRuntimeLevel;
    }

    public boolean isTargettingAspectJRuntime12() {
        boolean b = false;
        if (!isInJava5Mode()) {
            b = true;
        } else {
            b = (getTargetAspectjRuntimeLevel() == RuntimeVersion.V1_2);
        }
        return b;
    }

    public static class TypeMap {

        public final static int DONT_USE_REFS = 0;

        public final static int USE_WEAK_REFS = 1;

        public final static int USE_SOFT_REFS = 2;

        public List<String> addedSinceLastDemote;

        public List<String> writtenClasses;

        private static boolean debug = false;

        public static boolean useExpendableMap = true;

        private boolean demotionSystemActive;

        private boolean debugDemotion = false;

        public int policy = USE_WEAK_REFS;

        final Map<String, ResolvedType> tMap = new HashMap<>();

        final Map<String, Reference<ResolvedType>> expendableMap = Collections.synchronizedMap(new WeakHashMap<>());

        private final World w;

        private boolean memoryProfiling = false;

        private int maxExpendableMapSize = -1;

        private int collectedTypes = 0;

        private final ReferenceQueue<ResolvedType> rq = new ReferenceQueue<>();

        TypeMap(World w) {
            demotionSystemActive = w.isDemotionActive() && (w.isLoadtimeWeaving() || w.couldIncrementalCompileFollow());
            addedSinceLastDemote = new ArrayList<>();
            writtenClasses = new ArrayList<>();
            this.w = w;
            memoryProfiling = false;
        }

        public Map<String, Reference<ResolvedType>> getExpendableMap() {
            return expendableMap;
        }

        public Map<String, ResolvedType> getMainMap() {
            return tMap;
        }

        public int demote() {
            return demote(false);
        }

        public int demote(boolean atEndOfCompile) {
            if (!demotionSystemActive) {
                return 0;
            }
            if (debugDemotion) {
                System.out.println("Demotion running " + addedSinceLastDemote);
            }
            boolean isLtw = w.isLoadtimeWeaving();
            int demotionCounter = 0;
            if (isLtw) {
                for (String key : addedSinceLastDemote) {
                    ResolvedType type = tMap.get(key);
                    if (type != null && !type.isAspect() && !type.equals(UnresolvedType.OBJECT) && !type.isPrimitiveType()) {
                        List<ConcreteTypeMunger> typeMungers = type.getInterTypeMungers();
                        if (typeMungers == null || typeMungers.size() == 0) {
                            tMap.remove(key);
                            insertInExpendableMap(key, type);
                            demotionCounter++;
                        }
                    }
                }
                addedSinceLastDemote.clear();
            } else {
                List<String> forRemoval = new ArrayList<>();
                for (String key : addedSinceLastDemote) {
                    ResolvedType type = tMap.get(key);
                    if (type == null) {
                        forRemoval.add(key);
                        continue;
                    }
                    if (!writtenClasses.contains(type.getName())) {
                        continue;
                    }
                    if (type != null && !type.isAspect() && !type.equals(UnresolvedType.OBJECT) && !type.isPrimitiveType()) {
                        List<ConcreteTypeMunger> typeMungers = type.getInterTypeMungers();
                        if (typeMungers == null || typeMungers.size() == 0) {
                            ReferenceTypeDelegate delegate = ((ReferenceType) type).getDelegate();
                            boolean isWeavable = delegate == null ? false : delegate.isExposedToWeaver();
                            boolean hasBeenWoven = delegate == null ? false : delegate.hasBeenWoven();
                            if (!isWeavable || hasBeenWoven) {
                                if (debugDemotion) {
                                    System.out.println("Demoting " + key);
                                }
                                forRemoval.add(key);
                                tMap.remove(key);
                                insertInExpendableMap(key, type);
                                demotionCounter++;
                            }
                        } else {
                            writtenClasses.remove(type.getName());
                            forRemoval.add(key);
                        }
                    } else {
                        writtenClasses.remove(type.getName());
                        forRemoval.add(key);
                    }
                }
                addedSinceLastDemote.removeAll(forRemoval);
            }
            if (debugDemotion) {
                System.out.println("Demoted " + demotionCounter + " types.  Types remaining in fixed set #" + tMap.keySet().size() + ".  addedSinceLastDemote size is " + addedSinceLastDemote.size());
                System.out.println("writtenClasses.size() = " + writtenClasses.size() + ": " + writtenClasses);
            }
            if (atEndOfCompile) {
                if (debugDemotion) {
                    System.out.println("Clearing writtenClasses");
                }
                writtenClasses.clear();
            }
            return demotionCounter;
        }

        private void insertInExpendableMap(String key, ResolvedType type) {
            if (useExpendableMap) {
                Reference<ResolvedType> existingReference = expendableMap.get(key);
                if (existingReference == null || existingReference.get() == null) {
                    expendableMap.remove(key);
                    if (policy == USE_SOFT_REFS) {
                        expendableMap.put(key, new SoftReference<>(type));
                    } else {
                        expendableMap.put(key, new WeakReference<>(type));
                    }
                }
            }
        }

        public ResolvedType put(String key, ResolvedType type) {
            if (!type.isCacheable()) {
                return type;
            }
            if (type.isParameterizedType() && type.isParameterizedWithTypeVariable()) {
                if (debug) {
                    System.err.println("Not putting a parameterized type that utilises member declared type variables into the typemap: key=" + key + " type=" + type);
                }
                return type;
            }
            if (type.isTypeVariableReference()) {
                if (debug) {
                    System.err.println("Not putting a type variable reference type into the typemap: key=" + key + " type=" + type);
                }
                return type;
            }
            if (type instanceof BoundedReferenceType) {
                if (debug) {
                    System.err.println("Not putting a bounded reference type into the typemap: key=" + key + " type=" + type);
                }
                return type;
            }
            if (type instanceof MissingResolvedTypeWithKnownSignature) {
                if (debug) {
                    System.err.println("Not putting a missing type into the typemap: key=" + key + " type=" + type);
                }
                return type;
            }
            if ((type instanceof ReferenceType) && (((ReferenceType) type).getDelegate() == null) && w.isExpendable(type)) {
                if (debug) {
                    System.err.println("Not putting expendable ref type with null delegate into typemap: key=" + key + " type=" + type);
                }
                return type;
            }
            if ((type instanceof ReferenceType) && type.getWorld().isInJava5Mode() && (((ReferenceType) type).getDelegate() != null) && type.isGenericType()) {
                throw new BCException("Attempt to add generic type to typemap " + type.toString() + " (should be raw)");
            }
            if (w.isExpendable(type)) {
                if (useExpendableMap) {
                    expendableMap.remove(key);
                    if (policy == USE_WEAK_REFS) {
                        if (memoryProfiling) {
                            expendableMap.put(key, new WeakReference<>(type, rq));
                        } else {
                            expendableMap.put(key, new WeakReference<>(type));
                        }
                    } else if (policy == USE_SOFT_REFS) {
                        if (memoryProfiling) {
                            expendableMap.put(key, new SoftReference<>(type, rq));
                        } else {
                            expendableMap.put(key, new SoftReference<>(type));
                        }
                    }
                }
                if (memoryProfiling && expendableMap.size() > maxExpendableMapSize) {
                    maxExpendableMapSize = expendableMap.size();
                }
                return type;
            } else {
                if (demotionSystemActive) {
                    addedSinceLastDemote.add(key);
                }
                return tMap.put(key, type);
            }
        }

        public void report() {
            if (!memoryProfiling) {
                return;
            }
            checkq();
            w.getMessageHandler().handleMessage(MessageUtil.info("MEMORY: world expendable type map reached maximum size of #" + maxExpendableMapSize + " entries"));
            w.getMessageHandler().handleMessage(MessageUtil.info("MEMORY: types collected through garbage collection #" + collectedTypes + " entries"));
        }

        public void checkq() {
            if (!memoryProfiling) {
                return;
            }
            Reference<? extends ResolvedType> r = null;
            while ((r = rq.poll()) != null) {
                collectedTypes++;
            }
        }

        public ResolvedType get(String key) {
            checkq();
            ResolvedType ret = tMap.get(key);
            if (ret == null) {
                if (policy == USE_WEAK_REFS) {
                    WeakReference<ResolvedType> ref = (WeakReference<ResolvedType>) expendableMap.get(key);
                    if (ref != null) {
                        ret = ref.get();
                    }
                } else if (policy == USE_SOFT_REFS) {
                    SoftReference<ResolvedType> ref = (SoftReference<ResolvedType>) expendableMap.get(key);
                    if (ref != null) {
                        ret = ref.get();
                    }
                }
            }
            return ret;
        }

        public ResolvedType remove(String key) {
            ResolvedType ret = tMap.remove(key);
            if (ret == null) {
                if (policy == USE_WEAK_REFS) {
                    WeakReference<ResolvedType> wref = (WeakReference<ResolvedType>) expendableMap.remove(key);
                    if (wref != null) {
                        ret = wref.get();
                    }
                } else if (policy == USE_SOFT_REFS) {
                    SoftReference<ResolvedType> wref = (SoftReference<ResolvedType>) expendableMap.remove(key);
                    if (wref != null) {
                        ret = wref.get();
                    }
                }
            }
            return ret;
        }

        public void classWriteEvent(String classname) {
            if (demotionSystemActive) {
                writtenClasses.add(classname);
            }
            if (debugDemotion) {
                System.out.println("Class write event for " + classname);
            }
        }

        public void demote(ResolvedType type) {
            String key = type.getSignature();
            if (debugDemotion) {
                addedSinceLastDemote.remove(key);
            }
            tMap.remove(key);
            insertInExpendableMap(key, type);
        }
    }

    private static class AspectPrecedenceCalculator {

        private final World world;

        private final Map<PrecedenceCacheKey, Integer> cachedResults;

        public AspectPrecedenceCalculator(World forSomeWorld) {
            world = forSomeWorld;
            cachedResults = new HashMap<>();
        }

        public int compareByPrecedence(ResolvedType firstAspect, ResolvedType secondAspect) {
            PrecedenceCacheKey key = new PrecedenceCacheKey(firstAspect, secondAspect);
            if (cachedResults.containsKey(key)) {
                return cachedResults.get(key);
            } else {
                int order = 0;
                DeclarePrecedence orderer = null;
                for (Declare declare : world.getCrosscuttingMembersSet().getDeclareDominates()) {
                    DeclarePrecedence d = (DeclarePrecedence) declare;
                    int thisOrder = d.compare(firstAspect, secondAspect);
                    if (thisOrder != 0) {
                        if (orderer == null) {
                            orderer = d;
                        }
                        if (order != 0 && order != thisOrder) {
                            ISourceLocation[] isls = new ISourceLocation[2];
                            isls[0] = orderer.getSourceLocation();
                            isls[1] = d.getSourceLocation();
                            Message m = new Message("conflicting declare precedence orderings for aspects: " + firstAspect.getName() + " and " + secondAspect.getName(), null, true, isls);
                            world.getMessageHandler().handleMessage(m);
                        } else {
                            order = thisOrder;
                        }
                    }
                }
                cachedResults.put(key, order);
                return order;
            }
        }

        public Integer getPrecedenceIfAny(ResolvedType aspect1, ResolvedType aspect2) {
            return cachedResults.get(new PrecedenceCacheKey(aspect1, aspect2));
        }

        public int compareByPrecedenceAndHierarchy(ResolvedType firstAspect, ResolvedType secondAspect) {
            if (firstAspect.equals(secondAspect)) {
                return 0;
            }
            int ret = compareByPrecedence(firstAspect, secondAspect);
            if (ret != 0) {
                return ret;
            }
            if (firstAspect.isAssignableFrom(secondAspect)) {
                return -1;
            } else if (secondAspect.isAssignableFrom(firstAspect)) {
                return +1;
            }
            return 0;
        }

        private static class PrecedenceCacheKey {

            public ResolvedType aspect1;

            public ResolvedType aspect2;

            public PrecedenceCacheKey(ResolvedType a1, ResolvedType a2) {
                aspect1 = a1;
                aspect2 = a2;
            }

            @Override
            public boolean equals(Object obj) {
                if (!(obj instanceof PrecedenceCacheKey)) {
                    return false;
                }
                PrecedenceCacheKey other = (PrecedenceCacheKey) obj;
                return (aspect1 == other.aspect1 && aspect2 == other.aspect2);
            }

            @Override
            public int hashCode() {
                return aspect1.hashCode() + aspect2.hashCode();
            }
        }
    }

    public void validateType(UnresolvedType type) {
    }

    public boolean isDemotionActive() {
        return true;
    }

    private final Map<Class<?>, TypeVariable[]> workInProgress1 = new HashMap<>();

    public TypeVariable[] getTypeVariablesCurrentlyBeingProcessed(Class<?> baseClass) {
        return workInProgress1.get(baseClass);
    }

    public void recordTypeVariablesCurrentlyBeingProcessed(Class<?> baseClass, TypeVariable[] typeVariables) {
        workInProgress1.put(baseClass, typeVariables);
    }

    public void forgetTypeVariablesCurrentlyBeingProcessed(Class<?> baseClass) {
        workInProgress1.remove(baseClass);
    }

    public void setAddSerialVerUID(boolean b) {
        addSerialVerUID = b;
    }

    public boolean isAddSerialVerUID() {
        return addSerialVerUID;
    }

    public void flush() {
        typeMap.expendableMap.clear();
    }

    public void ensureAdvancedConfigurationProcessed() {
        if (!checkedAdvancedConfiguration) {
            Properties p = getExtraConfiguration();
            if (p != null) {
                String s = p.getProperty(xsetBCEL_REPOSITORY_CACHING, xsetBCEL_REPOSITORY_CACHING_DEFAULT);
                bcelRepositoryCaching = s.equalsIgnoreCase("true");
                if (!bcelRepositoryCaching) {
                    getMessageHandler().handleMessage(MessageUtil.info("[bcelRepositoryCaching=false] AspectJ will not use a bcel cache for class information"));
                }
                s = p.getProperty(xsetITD_VERSION, xsetITD_VERSION_DEFAULT);
                if (s.equals(xsetITD_VERSION_ORIGINAL)) {
                    itdVersion = 1;
                }
                s = p.getProperty(xsetAVOID_FINAL, "false");
                if (s.equalsIgnoreCase("true")) {
                    useFinal = false;
                }
                s = p.getProperty(xsetMINIMAL_MODEL, "true");
                if (s.equalsIgnoreCase("false")) {
                    minimalModel = false;
                }
                s = p.getProperty(xsetTARGETING_RUNTIME_1610, "false");
                if (s.equalsIgnoreCase("true")) {
                    targettingRuntime1_6_10 = true;
                }
                s = p.getProperty(xsetFAST_PACK_METHODS, "true");
                fastMethodPacking = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetPIPELINE_COMPILATION, xsetPIPELINE_COMPILATION_DEFAULT);
                shouldPipelineCompilation = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetGENERATE_STACKMAPS, "false");
                shouldGenerateStackMaps = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetCOMPLETE_BINARY_TYPES, xsetCOMPLETE_BINARY_TYPES_DEFAULT);
                completeBinaryTypes = s.equalsIgnoreCase("true");
                if (completeBinaryTypes) {
                    getMessageHandler().handleMessage(MessageUtil.info("[completeBinaryTypes=true] Completion of binary types activated"));
                }
                s = p.getProperty(xsetTYPE_DEMOTION);
                if (s != null) {
                    boolean b = typeMap.demotionSystemActive;
                    if (b && s.equalsIgnoreCase("false")) {
                        System.out.println("typeDemotion=false: type demotion switched OFF");
                        typeMap.demotionSystemActive = false;
                    } else if (!b && s.equalsIgnoreCase("true")) {
                        System.out.println("typeDemotion=true: type demotion switched ON");
                        typeMap.demotionSystemActive = true;
                    }
                }
                s = p.getProperty(xsetOVERWEAVING, "false");
                if (s.equalsIgnoreCase("true")) {
                    overWeaving = true;
                }
                s = p.getProperty(xsetTYPE_DEMOTION_DEBUG, "false");
                if (s.equalsIgnoreCase("true")) {
                    typeMap.debugDemotion = true;
                }
                s = p.getProperty(xsetTYPE_REFS, "true");
                if (s.equalsIgnoreCase("false")) {
                    typeMap.policy = TypeMap.USE_SOFT_REFS;
                }
                runMinimalMemorySet = p.getProperty(xsetRUN_MINIMAL_MEMORY) != null;
                s = p.getProperty(xsetRUN_MINIMAL_MEMORY, "false");
                runMinimalMemory = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetDEBUG_STRUCTURAL_CHANGES_CODE, "false");
                forDEBUG_structuralChangesCode = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetTRANSIENT_TJP_FIELDS, "false");
                transientTjpFields = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetDEBUG_BRIDGING, "false");
                forDEBUG_bridgingCode = s.equalsIgnoreCase("true");
                s = p.getProperty(xsetGENERATE_NEW_LVTS, "true");
                generateNewLvts = s.equalsIgnoreCase("true");
                if (!generateNewLvts) {
                    getMessageHandler().handleMessage(MessageUtil.info("[generateNewLvts=false] for methods without an incoming local variable table, do not generate one"));
                }
                s = p.getProperty(xsetOPTIMIZED_MATCHING, "true");
                optimizedMatching = s.equalsIgnoreCase("true");
                if (!optimizedMatching) {
                    getMessageHandler().handleMessage(MessageUtil.info("[optimizedMatching=false] optimized matching turned off"));
                }
                s = p.getProperty(xsetTIMERS_PER_JOINPOINT, "25000");
                try {
                    timersPerJoinpoint = Integer.parseInt(s);
                } catch (Exception e) {
                    getMessageHandler().handleMessage(MessageUtil.error("unable to process timersPerJoinpoint value of " + s));
                    timersPerJoinpoint = 25000;
                }
                s = p.getProperty(xsetTIMERS_PER_FASTMATCH_CALL, "250");
                try {
                    timersPerType = Integer.parseInt(s);
                } catch (Exception e) {
                    getMessageHandler().handleMessage(MessageUtil.error("unable to process timersPerType value of " + s));
                    timersPerType = 250;
                }
            }
            try {
                if (systemPropertyOverWeaving) {
                    overWeaving = true;
                }
                String value = null;
                value = System.getProperty("aspectj.typeDemotion", "false");
                if (value.equalsIgnoreCase("true")) {
                    System.out.println("ASPECTJ: aspectj.typeDemotion=true: type demotion switched ON");
                    typeMap.demotionSystemActive = true;
                }
                value = System.getProperty("aspectj.minimalModel", "false");
                if (value.equalsIgnoreCase("true")) {
                    System.out.println("ASPECTJ: aspectj.minimalModel=true: minimal model switched ON");
                    minimalModel = true;
                }
            } catch (Throwable t) {
                System.err.println("ASPECTJ: Unable to read system properties");
                t.printStackTrace();
            }
            checkedAdvancedConfiguration = true;
        }
    }

    public boolean isRunMinimalMemory() {
        ensureAdvancedConfigurationProcessed();
        return runMinimalMemory;
    }

    public boolean isTransientTjpFields() {
        ensureAdvancedConfigurationProcessed();
        return transientTjpFields;
    }

    public boolean isRunMinimalMemorySet() {
        ensureAdvancedConfigurationProcessed();
        return runMinimalMemorySet;
    }

    public boolean shouldFastPackMethods() {
        ensureAdvancedConfigurationProcessed();
        return fastMethodPacking;
    }

    public boolean shouldPipelineCompilation() {
        ensureAdvancedConfigurationProcessed();
        return shouldPipelineCompilation;
    }

    public boolean shouldGenerateStackMaps() {
        ensureAdvancedConfigurationProcessed();
        return shouldGenerateStackMaps;
    }

    public void setIncrementalCompileCouldFollow(boolean b) {
        incrementalCompileCouldFollow = b;
    }

    public boolean couldIncrementalCompileFollow() {
        return incrementalCompileCouldFollow;
    }

    public void setSynchronizationPointcutsInUse() {
        if (trace.isTraceEnabled()) {
            trace.enter("setSynchronizationPointcutsInUse", this);
        }
        synchronizationPointcutsInUse = true;
        if (trace.isTraceEnabled()) {
            trace.exit("setSynchronizationPointcutsInUse");
        }
    }

    public boolean areSynchronizationPointcutsInUse() {
        return synchronizationPointcutsInUse;
    }

    public void registerPointcutHandler(PointcutDesignatorHandler designatorHandler) {
        if (pointcutDesignators == null) {
            pointcutDesignators = new HashSet<>();
        }
        pointcutDesignators.add(designatorHandler);
    }

    public Set<PointcutDesignatorHandler> getRegisteredPointcutHandlers() {
        if (pointcutDesignators == null) {
            return Collections.emptySet();
        }
        return pointcutDesignators;
    }

    public void reportMatch(ShadowMunger munger, Shadow shadow) {
    }

    public boolean isOverWeaving() {
        return overWeaving;
    }

    public void reportCheckerMatch(Checker checker, Shadow shadow) {
    }

    public boolean isXmlConfigured() {
        return false;
    }

    public boolean isAspectIncluded(ResolvedType aspectType) {
        return true;
    }

    public boolean hasUnsatisfiedDependency(ResolvedType aspectType) {
        return false;
    }

    public TypePattern getAspectScope(ResolvedType declaringType) {
        return null;
    }

    public Map<String, ResolvedType> getFixed() {
        return typeMap.tMap;
    }

    public Map<String, Reference<ResolvedType>> getExpendable() {
        return typeMap.expendableMap;
    }

    public void demote() {
        typeMap.demote();
    }

    protected boolean isExpendable(ResolvedType type) {
        return !type.equals(UnresolvedType.OBJECT) && !type.isExposedToWeaver() && !type.isPrimitiveType() && !type.isPrimitiveArray();
    }

    private Map<ResolvedType, Set<ResolvedType>> exclusionMap = new HashMap<>();

    public Map<ResolvedType, Set<ResolvedType>> getExclusionMap() {
        return exclusionMap;
    }

    private TimeCollector timeCollector = null;

    public void record(Pointcut pointcut, long timetaken) {
        if (timeCollector == null) {
            ensureAdvancedConfigurationProcessed();
            timeCollector = new TimeCollector(this);
        }
        timeCollector.record(pointcut, timetaken);
    }

    public void recordFastMatch(Pointcut pointcut, long timetaken) {
        if (timeCollector == null) {
            ensureAdvancedConfigurationProcessed();
            timeCollector = new TimeCollector(this);
        }
        timeCollector.recordFastMatch(pointcut, timetaken);
    }

    public void reportTimers() {
        if (timeCollector != null && !timingPeriodically) {
            timeCollector.report();
            timeCollector = new TimeCollector(this);
        }
    }

    private static class TimeCollector {

        private World world;

        long joinpointCount;

        long typeCount;

        long perJoinpointCount;

        long perTypes;

        Map<String, Long> joinpointsPerPointcut = new HashMap<>();

        Map<String, Long> timePerPointcut = new HashMap<>();

        Map<String, Long> fastMatchTimesPerPointcut = new HashMap<>();

        Map<String, Long> fastMatchTypesPerPointcut = new HashMap<>();

        TimeCollector(World world) {
            this.perJoinpointCount = world.timersPerJoinpoint;
            this.perTypes = world.timersPerType;
            this.world = world;
            this.joinpointCount = 0;
            this.typeCount = 0;
            this.joinpointsPerPointcut = new HashMap<>();
            this.timePerPointcut = new HashMap<>();
        }

        public void report() {
            long totalTime = 0L;
            for (String p : joinpointsPerPointcut.keySet()) {
                totalTime += timePerPointcut.get(p);
            }
            world.getMessageHandler().handleMessage(MessageUtil.info("Pointcut matching cost (total=" + (totalTime / 1000000) + "ms for " + joinpointCount + " joinpoint match calls):"));
            for (String p : joinpointsPerPointcut.keySet()) {
                StringBuilder sb = new StringBuilder();
                sb.append("Time:" + (timePerPointcut.get(p) / 1000000) + "ms (jps:#" + joinpointsPerPointcut.get(p) + ") matching against " + p);
                world.getMessageHandler().handleMessage(MessageUtil.info(sb.toString()));
            }
            world.getMessageHandler().handleMessage(MessageUtil.info("---"));
            totalTime = 0L;
            for (String p : fastMatchTimesPerPointcut.keySet()) {
                totalTime += fastMatchTimesPerPointcut.get(p);
            }
            world.getMessageHandler().handleMessage(MessageUtil.info("Pointcut fast matching cost (total=" + (totalTime / 1000000) + "ms for " + typeCount + " fast match calls):"));
            for (String p : fastMatchTimesPerPointcut.keySet()) {
                StringBuilder sb = new StringBuilder();
                sb.append("Time:" + (fastMatchTimesPerPointcut.get(p) / 1000000) + "ms (types:#" + fastMatchTypesPerPointcut.get(p) + ") fast matching against " + p);
                world.getMessageHandler().handleMessage(MessageUtil.info(sb.toString()));
            }
            world.getMessageHandler().handleMessage(MessageUtil.info("---"));
        }

        void record(Pointcut pointcut, long timetakenInNs) {
            joinpointCount++;
            String pointcutText = pointcut.toString();
            Long jpcounter = joinpointsPerPointcut.get(pointcutText);
            if (jpcounter == null) {
                jpcounter = 1L;
            } else {
                jpcounter++;
            }
            joinpointsPerPointcut.put(pointcutText, jpcounter);
            Long time = timePerPointcut.get(pointcutText);
            if (time == null) {
                time = timetakenInNs;
            } else {
                time += timetakenInNs;
            }
            timePerPointcut.put(pointcutText, time);
            if (world.timingPeriodically) {
                if ((joinpointCount % perJoinpointCount) == 0) {
                    long totalTime = 0L;
                    for (String p : joinpointsPerPointcut.keySet()) {
                        totalTime += timePerPointcut.get(p);
                    }
                    world.getMessageHandler().handleMessage(MessageUtil.info("Pointcut matching cost (total=" + (totalTime / 1000000) + "ms for " + joinpointCount + " joinpoint match calls):"));
                    for (String p : joinpointsPerPointcut.keySet()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("Time:" + (timePerPointcut.get(p) / 1000000) + "ms (jps:#" + joinpointsPerPointcut.get(p) + ") matching against " + p);
                        world.getMessageHandler().handleMessage(MessageUtil.info(sb.toString()));
                    }
                    world.getMessageHandler().handleMessage(MessageUtil.info("---"));
                }
            }
        }

        void recordFastMatch(Pointcut pointcut, long timetakenInNs) {
            typeCount++;
            String pointcutText = pointcut.toString();
            Long typecounter = fastMatchTypesPerPointcut.get(pointcutText);
            if (typecounter == null) {
                typecounter = 1L;
            } else {
                typecounter++;
            }
            fastMatchTypesPerPointcut.put(pointcutText, typecounter);
            Long time = fastMatchTimesPerPointcut.get(pointcutText);
            if (time == null) {
                time = timetakenInNs;
            } else {
                time += timetakenInNs;
            }
            fastMatchTimesPerPointcut.put(pointcutText, time);
            if (world.timingPeriodically) {
                if ((typeCount % perTypes) == 0) {
                    long totalTime = 0L;
                    for (String p : fastMatchTimesPerPointcut.keySet()) {
                        totalTime += fastMatchTimesPerPointcut.get(p);
                    }
                    world.getMessageHandler().handleMessage(MessageUtil.info("Pointcut fast matching cost (total=" + (totalTime / 1000000) + "ms for " + typeCount + " fast match calls):"));
                    for (String p : fastMatchTimesPerPointcut.keySet()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("Time:" + (fastMatchTimesPerPointcut.get(p) / 1000000) + "ms (types:#" + fastMatchTypesPerPointcut.get(p) + ") fast matching against " + p);
                        world.getMessageHandler().handleMessage(MessageUtil.info(sb.toString()));
                    }
                    world.getMessageHandler().handleMessage(MessageUtil.info("---"));
                }
            }
        }
    }

    public TypeMap getTypeMap() {
        return typeMap;
    }

    public static void reset() {
    }

    public int getItdVersion() {
        return itdVersion;
    }

    public abstract boolean isLoadtimeWeaving();

    public void classWriteEvent(char[][] compoundName) {
    }
}
