package org.aspectj.weaver.bcel;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.Field;
import org.aspectj.apache.bcel.generic.ArrayType;
import org.aspectj.apache.bcel.generic.BranchHandle;
import org.aspectj.apache.bcel.generic.FieldInstruction;
import org.aspectj.apache.bcel.generic.INVOKEINTERFACE;
import org.aspectj.apache.bcel.generic.Instruction;
import org.aspectj.apache.bcel.generic.InstructionBranch;
import org.aspectj.apache.bcel.generic.InstructionConstants;
import org.aspectj.apache.bcel.generic.InstructionFactory;
import org.aspectj.apache.bcel.generic.InstructionHandle;
import org.aspectj.apache.bcel.generic.InstructionLV;
import org.aspectj.apache.bcel.generic.InstructionList;
import org.aspectj.apache.bcel.generic.InstructionTargeter;
import org.aspectj.apache.bcel.generic.InvokeInstruction;
import org.aspectj.apache.bcel.generic.LineNumberTag;
import org.aspectj.apache.bcel.generic.LocalVariableTag;
import org.aspectj.apache.bcel.generic.MULTIANEWARRAY;
import org.aspectj.apache.bcel.generic.ObjectType;
import org.aspectj.apache.bcel.generic.TargetLostException;
import org.aspectj.apache.bcel.generic.Type;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.weaver.Advice;
import org.aspectj.weaver.AdviceKind;
import org.aspectj.weaver.AjcMemberMaker;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.ConcreteTypeMunger;
import org.aspectj.weaver.IntMap;
import org.aspectj.weaver.Member;
import org.aspectj.weaver.MemberImpl;
import org.aspectj.weaver.NameMangler;
import org.aspectj.weaver.NewConstructorTypeMunger;
import org.aspectj.weaver.NewFieldTypeMunger;
import org.aspectj.weaver.NewMethodTypeMunger;
import org.aspectj.weaver.ResolvedMember;
import org.aspectj.weaver.ResolvedMemberImpl;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.ShadowMunger;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.World;
import org.aspectj.weaver.ast.Var;
import org.aspectj.weaver.patterns.AbstractPatternNodeVisitor;
import org.aspectj.weaver.patterns.AndPointcut;
import org.aspectj.weaver.patterns.NotPointcut;
import org.aspectj.weaver.patterns.OrPointcut;
import org.aspectj.weaver.patterns.ThisOrTargetPointcut;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class BcelShadow extends Shadow {

    private static final String[] NoDeclaredExceptions = new String[0];

    private ShadowRange range;

    private final BcelWorld world;

    private final LazyMethodGen enclosingMethod;

    public static boolean appliedLazyTjpOptimization;

    private String actualInstructionTargetType;

    public BcelShadow(BcelWorld world, Kind kind, Member signature, LazyMethodGen enclosingMethod, BcelShadow enclosingShadow) {
        super(kind, signature, enclosingShadow);
        this.world = world;
        this.enclosingMethod = enclosingMethod;
    }

    public BcelShadow copyInto(LazyMethodGen recipient, BcelShadow enclosing) {
        BcelShadow s = new BcelShadow(world, getKind(), getSignature(), recipient, enclosing);
        if (mungers.size() > 0) {
            List<ShadowMunger> src = mungers;
            if (s.mungers == Collections.EMPTY_LIST) {
                s.mungers = new ArrayList<>();
            }
            List<ShadowMunger> dest = s.mungers;
            for (ShadowMunger shadowMunger : src) {
                dest.add(shadowMunger);
            }
        }
        return s;
    }

    @Override
    public World getIWorld() {
        return world;
    }

    private boolean deleteNewAndDup() {
        final ConstantPool cpool = getEnclosingClass().getConstantPool();
        int depth = 1;
        InstructionHandle ih = range.getStart();
        while (ih != null) {
            Instruction inst = ih.getInstruction();
            if (inst.opcode == Constants.INVOKESPECIAL && ((InvokeInstruction) inst).getName(cpool).equals("<init>")) {
                depth++;
            } else if (inst.opcode == Constants.NEW) {
                depth--;
                if (depth == 0) {
                    break;
                }
            }
            ih = ih.getPrev();
        }
        if (ih == null) {
            return false;
        }
        InstructionHandle newHandle = ih;
        InstructionHandle endHandle = newHandle.getNext();
        InstructionHandle nextHandle;
        if (endHandle.getInstruction().opcode == Constants.DUP) {
            nextHandle = endHandle.getNext();
            retargetFrom(newHandle, nextHandle);
            retargetFrom(endHandle, nextHandle);
        } else if (endHandle.getInstruction().opcode == Constants.DUP_X1) {
            InstructionHandle dupHandle = endHandle;
            endHandle = endHandle.getNext();
            nextHandle = endHandle.getNext();
            boolean skipEndRepositioning = false;
            if (endHandle.getInstruction().opcode == Constants.SWAP) {
            } else if (endHandle.getInstruction().opcode == Constants.IMPDEP1) {
                skipEndRepositioning = true;
            } else {
                throw new RuntimeException("Unhandled kind of new " + endHandle);
            }
            retargetFrom(newHandle, nextHandle);
            retargetFrom(dupHandle, nextHandle);
            if (!skipEndRepositioning) {
                retargetFrom(endHandle, nextHandle);
            }
        } else {
            endHandle = newHandle;
            nextHandle = endHandle.getNext();
            retargetFrom(newHandle, nextHandle);
            getRange().insert(InstructionConstants.POP, Range.OutsideAfter);
        }
        try {
            range.getBody().delete(newHandle, endHandle);
        } catch (TargetLostException e) {
            throw new BCException("shouldn't happen");
        }
        return true;
    }

    private void retargetFrom(InstructionHandle old, InstructionHandle fresh) {
        for (InstructionTargeter targeter : old.getTargetersCopy()) {
            if (targeter instanceof ExceptionRange) {
                ExceptionRange it = (ExceptionRange) targeter;
                it.updateTarget(old, fresh, it.getBody());
            } else {
                targeter.updateTarget(old, fresh);
            }
        }
    }

    private List<BcelAdvice> badAdvice = null;

    public void addAdvicePreventingLazyTjp(BcelAdvice advice) {
        if (badAdvice == null) {
            badAdvice = new ArrayList<>();
        }
        badAdvice.add(advice);
    }

    @Override
    protected void prepareForMungers() {
        boolean deletedNewAndDup = true;
        if (getKind() == ConstructorCall) {
            if (!world.isJoinpointArrayConstructionEnabled() || !this.getSignature().getDeclaringType().isArray()) {
                deletedNewAndDup = deleteNewAndDup();
            }
            initializeArgVars();
        } else if (getKind() == PreInitialization) {
            ShadowRange range = getRange();
            range.insert(InstructionConstants.NOP, Range.InsideAfter);
        } else if (getKind() == ExceptionHandler) {
            ShadowRange range = getRange();
            InstructionList body = range.getBody();
            InstructionHandle start = range.getStart();
            argVars = new BcelVar[1];
            UnresolvedType tx = getArgType(0);
            argVars[0] = genTempVar(tx, "ajc$arg0");
            InstructionHandle insertedInstruction = range.insert(argVars[0].createStore(getFactory()), Range.OutsideBefore);
            for (InstructionTargeter t : start.getTargetersCopy()) {
                if (t instanceof ExceptionRange) {
                    ExceptionRange er = (ExceptionRange) t;
                    er.updateTarget(start, insertedInstruction, body);
                }
            }
        }
        isThisJoinPointLazy = true;
        badAdvice = null;
        for (ShadowMunger munger : mungers) {
            munger.specializeOn(this);
        }
        initializeThisJoinPoint();
        if (thisJoinPointVar != null && !isThisJoinPointLazy && badAdvice != null && badAdvice.size() > 1) {
            int valid = 0;
            for (BcelAdvice element : badAdvice) {
                ISourceLocation sLoc = element.getSourceLocation();
                if (sLoc != null && sLoc.getLine() > 0) {
                    valid++;
                }
            }
            if (valid != 0) {
                ISourceLocation[] badLocs = new ISourceLocation[valid];
                int i = 0;
                for (BcelAdvice element : badAdvice) {
                    ISourceLocation sLoc = element.getSourceLocation();
                    if (sLoc != null) {
                        badLocs[i++] = sLoc;
                    }
                }
                world.getLint().multipleAdviceStoppingLazyTjp.signal(new String[]{this.toString()}, getSourceLocation(), badLocs);
            }
        }
        badAdvice = null;
        InstructionFactory fact = getFactory();
        if (getKind().argsOnStack() && argVars != null) {
            if (getKind() == ExceptionHandler && range.getEnd().getNext().getInstruction().equals(InstructionConstants.POP)) {
                range.getEnd().getNext().setInstruction(InstructionConstants.NOP);
            } else {
                range.insert(BcelRenderer.renderExprs(fact, world, argVars), Range.InsideBefore);
                if (targetVar != null) {
                    range.insert(BcelRenderer.renderExpr(fact, world, targetVar), Range.InsideBefore);
                }
                if (getKind() == ConstructorCall) {
                    if (!world.isJoinpointArrayConstructionEnabled() || !this.getSignature().getDeclaringType().isArray()) {
                        if (deletedNewAndDup) {
                            range.insert(InstructionFactory.createDup(1), Range.InsideBefore);
                            range.insert(fact.createNew((ObjectType) BcelWorld.makeBcelType(getSignature().getDeclaringType())), Range.InsideBefore);
                        }
                    }
                }
            }
        }
    }

    public ShadowRange getRange() {
        return range;
    }

    public void setRange(ShadowRange range) {
        this.range = range;
    }

    private int sourceline = -1;

    public int getSourceLine() {
        if (sourceline != -1) {
            return sourceline;
        }
        Kind kind = getKind();
        if ((kind == MethodExecution) || (kind == ConstructorExecution) || (kind == AdviceExecution) || (kind == StaticInitialization) || (kind == PreInitialization) || (kind == Initialization)) {
            if (getEnclosingMethod().hasDeclaredLineNumberInfo()) {
                sourceline = getEnclosingMethod().getDeclarationLineNumber();
                return sourceline;
            }
        }
        if (range == null) {
            if (getEnclosingMethod().hasBody()) {
                sourceline = Utility.getSourceLine(getEnclosingMethod().getBody().getStart());
                return sourceline;
            } else {
                sourceline = 0;
                return sourceline;
            }
        }
        sourceline = Utility.getSourceLine(range.getStart());
        if (sourceline < 0) {
            sourceline = 0;
        }
        return sourceline;
    }

    @Override
    public ResolvedType getEnclosingType() {
        return getEnclosingClass().getType();
    }

    public LazyClassGen getEnclosingClass() {
        return enclosingMethod.getEnclosingClass();
    }

    public BcelWorld getWorld() {
        return world;
    }

    public static BcelShadow makeConstructorExecution(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle justBeforeStart) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, ConstructorExecution, world.makeJoinPointSignatureFromMethod(enclosingMethod, Member.CONSTRUCTOR), enclosingMethod, null);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, justBeforeStart.getNext()), Range.genEnd(body));
        return s;
    }

    public static BcelShadow makeStaticInitialization(BcelWorld world, LazyMethodGen enclosingMethod) {
        InstructionList body = enclosingMethod.getBody();
        InstructionHandle clinitStart = body.getStart();
        if (clinitStart.getInstruction() instanceof InvokeInstruction) {
            InvokeInstruction ii = (InvokeInstruction) clinitStart.getInstruction();
            if (ii.getName(enclosingMethod.getEnclosingClass().getConstantPool()).equals(NameMangler.AJC_PRE_CLINIT_NAME)) {
                clinitStart = clinitStart.getNext();
            }
        }
        InstructionHandle clinitEnd = body.getEnd();
        BcelShadow s = new BcelShadow(world, StaticInitialization, world.makeJoinPointSignatureFromMethod(enclosingMethod, Member.STATIC_INITIALIZATION), enclosingMethod, null);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, clinitStart), Range.genEnd(body, clinitEnd));
        return s;
    }

    public static BcelShadow makeExceptionHandler(BcelWorld world, ExceptionRange exceptionRange, LazyMethodGen enclosingMethod, InstructionHandle startOfHandler, BcelShadow enclosingShadow) {
        InstructionList body = enclosingMethod.getBody();
        UnresolvedType catchType = exceptionRange.getCatchType();
        UnresolvedType inType = enclosingMethod.getEnclosingClass().getType();
        ResolvedMemberImpl sig = MemberImpl.makeExceptionHandlerSignature(inType, catchType);
        sig.setParameterNames(new String[]{findHandlerParamName(startOfHandler)});
        BcelShadow s = new BcelShadow(world, ExceptionHandler, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        InstructionHandle start = Range.genStart(body, startOfHandler);
        InstructionHandle end = Range.genEnd(body, start);
        r.associateWithTargets(start, end);
        exceptionRange.updateTarget(startOfHandler, start, body);
        return s;
    }

    private static String findHandlerParamName(InstructionHandle startOfHandler) {
        if (startOfHandler.getInstruction().isStoreInstruction() && startOfHandler.getNext() != null) {
            int slot = startOfHandler.getInstruction().getIndex();
            for (InstructionTargeter targeter : startOfHandler.getNext().getTargeters()) {
                if (targeter instanceof LocalVariableTag) {
                    LocalVariableTag t = (LocalVariableTag) targeter;
                    if (t.getSlot() == slot) {
                        return t.getName();
                    }
                }
            }
        }
        return "<missing>";
    }

    public static BcelShadow makeIfaceInitialization(BcelWorld world, LazyMethodGen constructor, Member interfaceConstructorSignature) {
        constructor.getBody();
        BcelShadow s = new BcelShadow(world, Initialization, interfaceConstructorSignature, constructor, null);
        return s;
    }

    public void initIfaceInitializer(InstructionHandle end) {
        final InstructionList body = enclosingMethod.getBody();
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(this);
        InstructionHandle nop = body.insert(end, InstructionConstants.NOP);
        r.associateWithTargets(Range.genStart(body, nop), Range.genEnd(body, nop));
    }

    public static BcelShadow makeUnfinishedInitialization(BcelWorld world, LazyMethodGen constructor) {
        BcelShadow ret = new BcelShadow(world, Initialization, world.makeJoinPointSignatureFromMethod(constructor, Member.CONSTRUCTOR), constructor, null);
        if (constructor.getEffectiveSignature() != null) {
            ret.setMatchingSignature(constructor.getEffectiveSignature().getEffectiveSignature());
        }
        return ret;
    }

    public static BcelShadow makeUnfinishedPreinitialization(BcelWorld world, LazyMethodGen constructor) {
        BcelShadow ret = new BcelShadow(world, PreInitialization, world.makeJoinPointSignatureFromMethod(constructor, Member.CONSTRUCTOR), constructor, null);
        if (constructor.getEffectiveSignature() != null) {
            ret.setMatchingSignature(constructor.getEffectiveSignature().getEffectiveSignature());
        }
        return ret;
    }

    public static BcelShadow makeMethodExecution(BcelWorld world, LazyMethodGen enclosingMethod, boolean lazyInit) {
        if (!lazyInit) {
            return makeMethodExecution(world, enclosingMethod);
        }
        BcelShadow s = new BcelShadow(world, MethodExecution, enclosingMethod.getMemberView(), enclosingMethod, null);
        return s;
    }

    public void init() {
        if (range != null) {
            return;
        }
        final InstructionList body = enclosingMethod.getBody();
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(this);
        r.associateWithTargets(Range.genStart(body), Range.genEnd(body));
    }

    public static BcelShadow makeMethodExecution(BcelWorld world, LazyMethodGen enclosingMethod) {
        return makeShadowForMethod(world, enclosingMethod, MethodExecution, enclosingMethod.getMemberView());
    }

    public static BcelShadow makeShadowForMethod(BcelWorld world, LazyMethodGen enclosingMethod, Shadow.Kind kind, Member sig) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, kind, sig, enclosingMethod, null);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        // OPTIMIZE this occurs lots of times for all jp kinds...
        r.associateWithTargets(Range.genStart(body), Range.genEnd(body));
        return s;
    }

    public static BcelShadow makeAdviceExecution(BcelWorld world, LazyMethodGen enclosingMethod) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, AdviceExecution, world.makeJoinPointSignatureFromMethod(enclosingMethod, Member.ADVICE), enclosingMethod, null);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body), Range.genEnd(body));
        return s;
    }

    public static BcelShadow makeConstructorCall(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle callHandle, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        Member sig = world.makeJoinPointSignatureForMethodInvocation(enclosingMethod.getEnclosingClass(), (InvokeInstruction) callHandle.getInstruction());
        BcelShadow s = new BcelShadow(world, ConstructorCall, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, callHandle), Range.genEnd(body, callHandle));
        retargetAllBranches(callHandle, r.getStart());
        return s;
    }

    public static BcelShadow makeArrayConstructorCall(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle arrayInstruction, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        Member sig = world.makeJoinPointSignatureForArrayConstruction(enclosingMethod.getEnclosingClass(), arrayInstruction);
        BcelShadow s = new BcelShadow(world, ConstructorCall, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, arrayInstruction), Range.genEnd(body, arrayInstruction));
        retargetAllBranches(arrayInstruction, r.getStart());
        return s;
    }

    public static BcelShadow makeMonitorEnter(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle monitorInstruction, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        Member sig = world.makeJoinPointSignatureForMonitorEnter(enclosingMethod.getEnclosingClass(), monitorInstruction);
        BcelShadow s = new BcelShadow(world, SynchronizationLock, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, monitorInstruction), Range.genEnd(body, monitorInstruction));
        retargetAllBranches(monitorInstruction, r.getStart());
        return s;
    }

    public static BcelShadow makeMonitorExit(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle monitorInstruction, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        Member sig = world.makeJoinPointSignatureForMonitorExit(enclosingMethod.getEnclosingClass(), monitorInstruction);
        BcelShadow s = new BcelShadow(world, SynchronizationUnlock, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, monitorInstruction), Range.genEnd(body, monitorInstruction));
        retargetAllBranches(monitorInstruction, r.getStart());
        return s;
    }

    public static BcelShadow makeMethodCall(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle callHandle, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, MethodCall, world.makeJoinPointSignatureForMethodInvocation(enclosingMethod.getEnclosingClass(), (InvokeInstruction) callHandle.getInstruction()), enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, callHandle), Range.genEnd(body, callHandle));
        retargetAllBranches(callHandle, r.getStart());
        return s;
    }

    public static BcelShadow makeShadowForMethodCall(BcelWorld world, LazyMethodGen enclosingMethod, InstructionHandle callHandle, BcelShadow enclosingShadow, Kind kind, ResolvedMember sig) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, kind, sig, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, callHandle), Range.genEnd(body, callHandle));
        retargetAllBranches(callHandle, r.getStart());
        return s;
    }

    public static BcelShadow makeFieldGet(BcelWorld world, ResolvedMember field, LazyMethodGen enclosingMethod, InstructionHandle getHandle, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, FieldGet, field, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, getHandle), Range.genEnd(body, getHandle));
        retargetAllBranches(getHandle, r.getStart());
        return s;
    }

    public static BcelShadow makeFieldSet(BcelWorld world, ResolvedMember field, LazyMethodGen enclosingMethod, InstructionHandle setHandle, BcelShadow enclosingShadow) {
        final InstructionList body = enclosingMethod.getBody();
        BcelShadow s = new BcelShadow(world, FieldSet, field, enclosingMethod, enclosingShadow);
        ShadowRange r = new ShadowRange(body);
        r.associateWithShadow(s);
        r.associateWithTargets(Range.genStart(body, setHandle), Range.genEnd(body, setHandle));
        retargetAllBranches(setHandle, r.getStart());
        return s;
    }

    public static void retargetAllBranches(InstructionHandle from, InstructionHandle to) {
        for (InstructionTargeter source : from.getTargetersCopy()) {
            if (source instanceof InstructionBranch) {
                source.updateTarget(from, to);
            }
        }
    }

    public boolean terminatesWithReturn() {
        return getRange().getRealNext() == null;
    }

    public boolean arg0HoldsThis() {
        if (getKind().isEnclosingKind()) {
            return !Modifier.isStatic(getSignature().getModifiers());
        } else if (enclosingShadow == null) {
            return !enclosingMethod.isStatic();
        } else {
            return ((BcelShadow) enclosingShadow).arg0HoldsThis();
        }
    }

    private BcelVar thisVar = null;

    private BcelVar targetVar = null;

    private BcelVar[] argVars = null;

    private Map<ResolvedType, AnnotationAccessVar> kindedAnnotationVars = null;

    private Map<ResolvedType, TypeAnnotationAccessVar> thisAnnotationVars = null;

    private Map<ResolvedType, TypeAnnotationAccessVar> targetAnnotationVars = null;

    private Map<ResolvedType, AnnotationAccessVar> withinAnnotationVars = null;

    private Map<ResolvedType, AnnotationAccessVar> withincodeAnnotationVars = null;

    private boolean allArgVarsInitialized = false;

    boolean closureVarInitialized = false;

    @Override
    public Var getThisVar() {
        if (!hasThis()) {
            throw new IllegalStateException("no this");
        }
        initializeThisVar();
        return thisVar;
    }

    @Override
    public Var getThisAnnotationVar(UnresolvedType forAnnotationType) {
        if (!hasThis()) {
            throw new IllegalStateException("no this");
        }
        initializeThisAnnotationVars();
        Var v = thisAnnotationVars.get(forAnnotationType);
        if (v == null) {
            v = new TypeAnnotationAccessVar(forAnnotationType.resolve(world), (BcelVar) getThisVar());
        }
        return v;
    }

    @Override
    public Var getTargetVar() {
        if (!hasTarget()) {
            throw new IllegalStateException("no target");
        }
        initializeTargetVar();
        return targetVar;
    }

    @Override
    public Var getTargetAnnotationVar(UnresolvedType forAnnotationType) {
        if (!hasTarget()) {
            throw new IllegalStateException("no target");
        }
        initializeTargetAnnotationVars();
        Var v = targetAnnotationVars.get(forAnnotationType);
        if (v == null) {
            v = new TypeAnnotationAccessVar(forAnnotationType.resolve(world), (BcelVar) getTargetVar());
        }
        return v;
    }

    @Override
    public Var getArgVar(int i) {
        ensureInitializedArgVar(i);
        return argVars[i];
    }

    @Override
    public Var getArgAnnotationVar(int i, UnresolvedType forAnnotationType) {
        return new TypeAnnotationAccessVar(forAnnotationType.resolve(world), (BcelVar) getArgVar(i));
    }

    @Override
    public Var getKindedAnnotationVar(UnresolvedType forAnnotationType) {
        initializeKindedAnnotationVars();
        return kindedAnnotationVars.get(forAnnotationType);
    }

    @Override
    public Var getWithinAnnotationVar(UnresolvedType forAnnotationType) {
        initializeWithinAnnotationVars();
        return withinAnnotationVars.get(forAnnotationType);
    }

    @Override
    public Var getWithinCodeAnnotationVar(UnresolvedType forAnnotationType) {
        initializeWithinCodeAnnotationVars();
        return withincodeAnnotationVars.get(forAnnotationType);
    }

    private BcelVar thisJoinPointVar = null;

    private boolean isThisJoinPointLazy;

    private int lazyTjpConsumers = 0;

    private BcelVar thisJoinPointStaticPartVar = null;

    @Override
    public final Var getThisJoinPointStaticPartVar() {
        return getThisJoinPointStaticPartBcelVar();
    }

    @Override
    public final Var getThisEnclosingJoinPointStaticPartVar() {
        return getThisEnclosingJoinPointStaticPartBcelVar();
    }

    public void requireThisJoinPoint(boolean hasGuardTest, boolean isAround) {
        if (!isAround) {
            if (!hasGuardTest) {
                isThisJoinPointLazy = false;
            } else {
                lazyTjpConsumers++;
            }
        }
        if (thisJoinPointVar == null) {
            thisJoinPointVar = genTempVar(UnresolvedType.forName("org.aspectj.lang.JoinPoint"));
        }
    }

    @Override
    public Var getThisJoinPointVar() {
        requireThisJoinPoint(false, false);
        return thisJoinPointVar;
    }

    void initializeThisJoinPoint() {
        if (thisJoinPointVar == null) {
            return;
        }
        if (isThisJoinPointLazy) {
            isThisJoinPointLazy = checkLazyTjp();
        }
        if (isThisJoinPointLazy) {
            appliedLazyTjpOptimization = true;
            createThisJoinPoint();
            if (lazyTjpConsumers == 1) {
                return;
            }
            InstructionFactory fact = getFactory();
            InstructionList il = new InstructionList();
            il.append(InstructionConstants.ACONST_NULL);
            il.append(thisJoinPointVar.createStore(fact));
            range.insert(il, Range.OutsideBefore);
        } else {
            appliedLazyTjpOptimization = false;
            InstructionFactory fact = getFactory();
            InstructionList il = createThisJoinPoint();
            il.append(thisJoinPointVar.createStore(fact));
            range.insert(il, Range.OutsideBefore);
        }
    }

    private boolean checkLazyTjp() {
        for (ShadowMunger munger : mungers) {
            if (munger instanceof Advice) {
                if (((Advice) munger).getKind() == AdviceKind.Around) {
                    if (munger.getSourceLocation() != null) {
                        if (world.getLint().canNotImplementLazyTjp.isEnabled()) {
                            world.getLint().canNotImplementLazyTjp.signal(new String[]{toString()}, getSourceLocation(), new ISourceLocation[]{munger.getSourceLocation()});
                        }
                    }
                    return false;
                }
            }
        }
        return true;
    }

    InstructionList loadThisJoinPoint() {
        InstructionFactory fact = getFactory();
        InstructionList il = new InstructionList();
        if (isThisJoinPointLazy) {
            il.append(createThisJoinPoint());
            if (lazyTjpConsumers > 1) {
                il.append(thisJoinPointVar.createStore(fact));
                InstructionHandle end = il.append(thisJoinPointVar.createLoad(fact));
                il.insert(InstructionFactory.createBranchInstruction(Constants.IFNONNULL, end));
                il.insert(thisJoinPointVar.createLoad(fact));
            }
        } else {
            thisJoinPointVar.appendLoad(il, fact);
        }
        return il;
    }

    InstructionList createThisJoinPoint() {
        InstructionFactory fact = getFactory();
        InstructionList il = new InstructionList();
        BcelVar staticPart = getThisJoinPointStaticPartBcelVar();
        staticPart.appendLoad(il, fact);
        if (hasThis()) {
            ((BcelVar) getThisVar()).appendLoad(il, fact);
        } else {
            il.append(InstructionConstants.ACONST_NULL);
        }
        if (hasTarget()) {
            ((BcelVar) getTargetVar()).appendLoad(il, fact);
        } else {
            il.append(InstructionConstants.ACONST_NULL);
        }
        switch (getArgCount()) {
            case 0:
                il.append(fact.createInvoke("org.aspectj.runtime.reflect.Factory", "makeJP", LazyClassGen.tjpType, new Type[]{LazyClassGen.staticTjpType, Type.OBJECT, Type.OBJECT}, Constants.INVOKESTATIC));
                break;
            case 1:
                ((BcelVar) getArgVar(0)).appendLoadAndConvert(il, fact, world.getCoreType(ResolvedType.OBJECT));
                il.append(fact.createInvoke("org.aspectj.runtime.reflect.Factory", "makeJP", LazyClassGen.tjpType, new Type[]{LazyClassGen.staticTjpType, Type.OBJECT, Type.OBJECT, Type.OBJECT}, Constants.INVOKESTATIC));
                break;
            case 2:
                ((BcelVar) getArgVar(0)).appendLoadAndConvert(il, fact, world.getCoreType(ResolvedType.OBJECT));
                ((BcelVar) getArgVar(1)).appendLoadAndConvert(il, fact, world.getCoreType(ResolvedType.OBJECT));
                il.append(fact.createInvoke("org.aspectj.runtime.reflect.Factory", "makeJP", LazyClassGen.tjpType, new Type[]{LazyClassGen.staticTjpType, Type.OBJECT, Type.OBJECT, Type.OBJECT, Type.OBJECT}, Constants.INVOKESTATIC));
                break;
            default:
                il.append(makeArgsObjectArray());
                il.append(fact.createInvoke("org.aspectj.runtime.reflect.Factory", "makeJP", LazyClassGen.tjpType, new Type[]{LazyClassGen.staticTjpType, Type.OBJECT, Type.OBJECT, new ArrayType(Type.OBJECT, 1)}, Constants.INVOKESTATIC));
                break;
        }
        return il;
    }

    public BcelVar getThisJoinPointStaticPartBcelVar() {
        return getThisJoinPointStaticPartBcelVar(false);
    }

    @Override
    public BcelVar getThisAspectInstanceVar(ResolvedType aspectType) {
        return new AspectInstanceVar(aspectType);
    }

    public BcelVar getThisJoinPointStaticPartBcelVar(final boolean isEnclosingJp) {
        if (thisJoinPointStaticPartVar == null) {
            Field field = getEnclosingClass().getTjpField(this, isEnclosingJp);
            ResolvedType sjpType = null;
            if (world.isTargettingAspectJRuntime12()) {
                sjpType = world.getCoreType(UnresolvedType.JOINPOINT_STATICPART);
            } else {
                sjpType = isEnclosingJp ? world.getCoreType(UnresolvedType.JOINPOINT_ENCLOSINGSTATICPART) : world.getCoreType(UnresolvedType.JOINPOINT_STATICPART);
            }
            thisJoinPointStaticPartVar = new BcelFieldRef(sjpType, getEnclosingClass().getClassName(), field.getName());
        }
        return thisJoinPointStaticPartVar;
    }

    public BcelVar getThisEnclosingJoinPointStaticPartBcelVar() {
        if (enclosingShadow == null) {
            return getThisJoinPointStaticPartBcelVar(true);
        } else {
            return ((BcelShadow) enclosingShadow).getThisJoinPointStaticPartBcelVar(true);
        }
    }

    @Override
    public Member getEnclosingCodeSignature() {
        if (getKind().isEnclosingKind()) {
            return getSignature();
        } else if (getKind() == Shadow.PreInitialization) {
            return getSignature();
        } else if (enclosingShadow == null) {
            return getEnclosingMethod().getMemberView();
        } else {
            return enclosingShadow.getSignature();
        }
    }

    public Member getRealEnclosingCodeSignature() {
        return enclosingMethod.getMemberView();
    }

    private InstructionList makeArgsObjectArray() {
        InstructionFactory fact = getFactory();
        BcelVar arrayVar = genTempVar(UnresolvedType.OBJECTARRAY);
        final InstructionList il = new InstructionList();
        int alen = getArgCount();
        il.append(Utility.createConstant(fact, alen));
        il.append(fact.createNewArray(Type.OBJECT, (short) 1));
        arrayVar.appendStore(il, fact);
        int stateIndex = 0;
        for (int i = 0, len = getArgCount(); i < len; i++) {
            arrayVar.appendConvertableArrayStore(il, fact, stateIndex, (BcelVar) getArgVar(i));
            stateIndex++;
        }
        arrayVar.appendLoad(il, fact);
        return il;
    }

    private void initializeThisVar() {
        if (thisVar != null) {
            return;
        }
        thisVar = new BcelVar(getThisType().resolve(world), 0);
        thisVar.setPositionInAroundState(0);
    }

    public void initializeTargetVar() {
        InstructionFactory fact = getFactory();
        if (targetVar != null) {
            return;
        }
        if (getKind().isTargetSameAsThis()) {
            if (hasThis()) {
                initializeThisVar();
            }
            targetVar = thisVar;
        } else {
            initializeArgVars();
            UnresolvedType type = getTargetType();
            type = ensureTargetTypeIsCorrect(type);
            targetVar = genTempVar(type, "ajc$target");
            range.insert(targetVar.createStore(fact), Range.OutsideBefore);
            targetVar.setPositionInAroundState(hasThis() ? 1 : 0);
        }
    }

    public UnresolvedType ensureTargetTypeIsCorrect(UnresolvedType tx) {
        Member msig = getSignature();
        if (msig.getArity() == 0 && getKind() == MethodCall && msig.getName().charAt(0) == 'c' && tx.equals(ResolvedType.OBJECT) && msig.getReturnType().equals(ResolvedType.OBJECT) && msig.getName().equals("clone")) {
            InstructionHandle searchPtr = range.getStart().getPrev();
            while (Range.isRangeHandle(searchPtr) || searchPtr.getInstruction().isStoreInstruction()) {
                searchPtr = searchPtr.getPrev();
            }
            if (searchPtr.getInstruction().isLoadInstruction()) {
                LocalVariableTag lvt = LazyMethodGen.getLocalVariableTag(searchPtr, searchPtr.getInstruction().getIndex());
                if (lvt != null) {
                    return UnresolvedType.forSignature(lvt.getType());
                }
            }
            if (searchPtr.getInstruction() instanceof FieldInstruction) {
                FieldInstruction si = (FieldInstruction) searchPtr.getInstruction();
                Type t = si.getFieldType(getEnclosingClass().getConstantPool());
                return BcelWorld.fromBcel(t);
            }
            if (searchPtr.getInstruction().opcode == Constants.ANEWARRAY) {
                return BcelWorld.fromBcel(new ArrayType(Type.OBJECT, 1));
            }
            if (searchPtr.getInstruction() instanceof MULTIANEWARRAY) {
                MULTIANEWARRAY ana = (MULTIANEWARRAY) searchPtr.getInstruction();
                return BcelWorld.fromBcel(new ArrayType(Type.OBJECT, ana.getDimensions()));
            }
            throw new BCException("Can't determine real target of clone() when processing instruction " + searchPtr.getInstruction() + ".  Perhaps avoid selecting clone with your pointcut?");
        }
        return tx;
    }

    public void ensureInitializedArgVar(int argNumber) {
        if (allArgVarsInitialized || (argVars != null && argVars[argNumber] != null)) {
            return;
        }
        InstructionFactory fact = getFactory();
        int len = getArgCount();
        if (argVars == null) {
            argVars = new BcelVar[len];
        }
        int positionOffset = (hasTarget() ? 1 : 0) + ((hasThis() && !getKind().isTargetSameAsThis()) ? 1 : 0);
        if (getKind().argsOnStack()) {
            for (int i = len - 1; i >= 0; i--) {
                UnresolvedType type = getArgType(i);
                BcelVar tmp = genTempVar(type, "ajc$arg" + i);
                range.insert(tmp.createStore(getFactory()), Range.OutsideBefore);
                int position = i;
                position += positionOffset;
                tmp.setPositionInAroundState(position);
                argVars[i] = tmp;
            }
            allArgVarsInitialized = true;
        } else {
            int index = 0;
            if (arg0HoldsThis()) {
                index++;
            }
            boolean allInited = true;
            for (int i = 0; i < len; i++) {
                UnresolvedType type = getArgType(i);
                if (i == argNumber) {
                    argVars[argNumber] = genTempVar(type, "ajc$arg" + argNumber);
                    range.insert(argVars[argNumber].createCopyFrom(fact, index), Range.OutsideBefore);
                    argVars[argNumber].setPositionInAroundState(argNumber + positionOffset);
                }
                allInited = allInited && argVars[i] != null;
                index += type.getSize();
            }
            if (allInited && (argNumber + 1) == len) {
                allArgVarsInitialized = true;
            }
        }
    }

    public void initializeArgVars() {
        if (allArgVarsInitialized) {
            return;
        }
        InstructionFactory fact = getFactory();
        int len = getArgCount();
        if (argVars == null) {
            argVars = new BcelVar[len];
        }
        int positionOffset = (hasTarget() ? 1 : 0) + ((hasThis() && !getKind().isTargetSameAsThis()) ? 1 : 0);
        if (getKind().argsOnStack()) {
            for (int i = len - 1; i >= 0; i--) {
                UnresolvedType type = getArgType(i);
                BcelVar tmp = genTempVar(type, "ajc$arg" + i);
                range.insert(tmp.createStore(getFactory()), Range.OutsideBefore);
                int position = i;
                position += positionOffset;
                tmp.setPositionInAroundState(position);
                argVars[i] = tmp;
            }
        } else {
            int index = 0;
            if (arg0HoldsThis()) {
                index++;
            }
            for (int i = 0; i < len; i++) {
                UnresolvedType type = getArgType(i);
                if (argVars[i] == null) {
                    BcelVar tmp = genTempVar(type, "ajc$arg" + i);
                    range.insert(tmp.createCopyFrom(fact, index), Range.OutsideBefore);
                    argVars[i] = tmp;
                    tmp.setPositionInAroundState(i + positionOffset);
                }
                index += type.resolve(world).getSize();
            }
        }
        allArgVarsInitialized = true;
    }

    public void initializeForAroundClosure() {
        initializeArgVars();
        if (hasTarget()) {
            initializeTargetVar();
        }
        if (hasThis()) {
            initializeThisVar();
        }
    }

    public void initializeThisAnnotationVars() {
        if (thisAnnotationVars != null) {
            return;
        }
        thisAnnotationVars = new HashMap<>();
    }

    public void initializeTargetAnnotationVars() {
        if (targetAnnotationVars != null) {
            return;
        }
        if (getKind().isTargetSameAsThis()) {
            if (hasThis()) {
                initializeThisAnnotationVars();
            }
            targetAnnotationVars = thisAnnotationVars;
        } else {
            targetAnnotationVars = new HashMap<>();
            ResolvedType[] rtx = this.getTargetType().resolve(world).getAnnotationTypes();
            for (ResolvedType typeX : rtx) {
                targetAnnotationVars.put(typeX, new TypeAnnotationAccessVar(typeX, (BcelVar) getTargetVar()));
            }
        }
    }

    protected ResolvedMember getRelevantMember(ResolvedMember foundMember, Member relevantMember, ResolvedType relevantType) {
        if (foundMember != null) {
            return foundMember;
        }
        foundMember = getSignature().resolve(world);
        if (foundMember == null && relevantMember != null) {
            foundMember = relevantType.lookupMemberWithSupersAndITDs(relevantMember);
        }
        List<ConcreteTypeMunger> mungers = relevantType.resolve(world).getInterTypeMungers();
        for (ConcreteTypeMunger typeMunger : mungers) {
            if (typeMunger.getMunger() instanceof NewMethodTypeMunger || typeMunger.getMunger() instanceof NewConstructorTypeMunger) {
                ResolvedMember fakerm = typeMunger.getSignature();
                if (fakerm.getName().equals(getSignature().getName()) && fakerm.getParameterSignature().equals(getSignature().getParameterSignature())) {
                    if (foundMember.getKind() == ResolvedMember.CONSTRUCTOR) {
                        foundMember = AjcMemberMaker.interConstructor(relevantType, foundMember, typeMunger.getAspectType());
                    } else {
                        foundMember = AjcMemberMaker.interMethod(foundMember, typeMunger.getAspectType(), false);
                    }
                    return foundMember;
                }
            }
        }
        return foundMember;
    }

    protected ResolvedType[] getAnnotations(ResolvedMember foundMember, Member relevantMember, ResolvedType relevantType) {
        if (foundMember == null) {
            List<ConcreteTypeMunger> mungers = relevantType.resolve(world).getInterTypeMungers();
            for (ConcreteTypeMunger typeMunger : mungers) {
                if (typeMunger.getMunger() instanceof NewMethodTypeMunger || typeMunger.getMunger() instanceof NewConstructorTypeMunger) {
                    ResolvedMember fakerm = typeMunger.getSignature();
                    ResolvedMember ajcMethod = (getSignature().getKind() == ResolvedMember.CONSTRUCTOR ? AjcMemberMaker.postIntroducedConstructor(typeMunger.getAspectType(), fakerm.getDeclaringType(), fakerm.getParameterTypes()) : AjcMemberMaker.interMethodDispatcher(fakerm, typeMunger.getAspectType()));
                    ResolvedMember rmm = findMethod(typeMunger.getAspectType(), ajcMethod);
                    if (fakerm.getName().equals(getSignature().getName()) && fakerm.getParameterSignature().equals(getSignature().getParameterSignature())) {
                        relevantType = typeMunger.getAspectType();
                        foundMember = rmm;
                        return foundMember.getAnnotationTypes();
                    }
                }
            }
            foundMember = relevantType.lookupMemberWithSupersAndITDs(relevantMember);
            if (foundMember == null) {
                throw new IllegalStateException("Couldn't find member " + relevantMember + " for type " + relevantType);
            }
        }
        return foundMember.getAnnotationTypes();
    }

    public void initializeKindedAnnotationVars() {
        if (kindedAnnotationVars != null) {
            return;
        }
        kindedAnnotationVars = new HashMap<>();
        ResolvedType[] annotations = null;
        Member shadowSignature = getSignature();
        Member annotationHolder = getSignature();
        ResolvedType relevantType = shadowSignature.getDeclaringType().resolve(world);
        if (relevantType.isRawType() || relevantType.isParameterizedType()) {
            relevantType = relevantType.getGenericType();
        }
        if (getKind() == Shadow.StaticInitialization) {
            annotations = relevantType.resolve(world).getAnnotationTypes();
        } else if (getKind() == Shadow.MethodCall || getKind() == Shadow.ConstructorCall) {
            ResolvedMember foundMember = findMethod2(relevantType.resolve(world).getDeclaredMethods(), getSignature());
            annotations = getAnnotations(foundMember, shadowSignature, relevantType);
            annotationHolder = getRelevantMember(foundMember, shadowSignature, relevantType);
            relevantType = annotationHolder.getDeclaringType().resolve(world);
        } else if (getKind() == Shadow.FieldSet || getKind() == Shadow.FieldGet) {
            annotationHolder = findField(relevantType.getDeclaredFields(), getSignature());
            if (annotationHolder == null) {
                List<ConcreteTypeMunger> mungers = relevantType.resolve(world).getInterTypeMungers();
                for (ConcreteTypeMunger typeMunger : mungers) {
                    if (typeMunger.getMunger() instanceof NewFieldTypeMunger) {
                        ResolvedMember fakerm = typeMunger.getSignature();
                        ResolvedMember ajcMethod = AjcMemberMaker.interFieldInitializer(fakerm, typeMunger.getAspectType());
                        ResolvedMember rmm = findMethod(typeMunger.getAspectType(), ajcMethod);
                        if (fakerm.equals(getSignature())) {
                            relevantType = typeMunger.getAspectType();
                            annotationHolder = rmm;
                        }
                    }
                }
            }
            annotations = ((ResolvedMember) annotationHolder).getAnnotationTypes();
        } else if (getKind() == Shadow.MethodExecution || getKind() == Shadow.ConstructorExecution || getKind() == Shadow.AdviceExecution) {
            ResolvedMember foundMember = findMethod2(relevantType.getDeclaredMethods(), getSignature());
            annotations = getAnnotations(foundMember, shadowSignature, relevantType);
            annotationHolder = getRelevantMember(foundMember, annotationHolder, relevantType);
            UnresolvedType ut = annotationHolder.getDeclaringType();
            relevantType = ut.resolve(world);
        } else if (getKind() == Shadow.ExceptionHandler) {
            relevantType = getSignature().getParameterTypes()[0].resolve(world);
            annotations = relevantType.getAnnotationTypes();
        } else if (getKind() == Shadow.PreInitialization || getKind() == Shadow.Initialization) {
            ResolvedMember found = findMethod2(relevantType.getDeclaredMethods(), getSignature());
            annotations = found.getAnnotationTypes();
        }
        if (annotations == null) {
            throw new BCException("Could not discover annotations for shadow: " + getKind());
        }
        for (ResolvedType annotationType : annotations) {
            AnnotationAccessVar accessVar = new AnnotationAccessVar(this, getKind(), annotationType.resolve(world), relevantType, annotationHolder, false);
            kindedAnnotationVars.put(annotationType, accessVar);
        }
    }

    private ResolvedMember findMethod2(ResolvedMember[] members, Member sig) {
        String signatureName = sig.getName();
        String parameterSignature = sig.getParameterSignature();
        for (ResolvedMember member : members) {
            if (member.getName().equals(signatureName) && member.getParameterSignature().equals(parameterSignature)) {
                return member;
            }
        }
        return null;
    }

    private ResolvedMember findMethod(ResolvedType aspectType, ResolvedMember ajcMethod) {
        ResolvedMember[] decMethods = aspectType.getDeclaredMethods();
        for (ResolvedMember member : decMethods) {
            if (member.equals(ajcMethod)) {
                return member;
            }
        }
        return null;
    }

    private ResolvedMember findField(ResolvedMember[] members, Member lookingFor) {
        for (ResolvedMember member : members) {
            if (member.getName().equals(getSignature().getName()) && member.getType().equals(getSignature().getType())) {
                return member;
            }
        }
        return null;
    }

    public void initializeWithinAnnotationVars() {
        if (withinAnnotationVars != null) {
            return;
        }
        withinAnnotationVars = new HashMap<>();
        ResolvedType[] annotations = getEnclosingType().resolve(world).getAnnotationTypes();
        for (ResolvedType ann : annotations) {
            Kind k = Shadow.StaticInitialization;
            withinAnnotationVars.put(ann, new AnnotationAccessVar(this, k, ann, getEnclosingType(), null, true));
        }
    }

    public void initializeWithinCodeAnnotationVars() {
        if (withincodeAnnotationVars != null) {
            return;
        }
        withincodeAnnotationVars = new HashMap<>();
        ResolvedType[] annotations = getEnclosingMethod().getMemberView().getAnnotationTypes();
        for (ResolvedType ann : annotations) {
            Kind k = (getEnclosingMethod().getMemberView().getKind() == Member.CONSTRUCTOR ? Shadow.ConstructorExecution : Shadow.MethodExecution);
            withincodeAnnotationVars.put(ann, new AnnotationAccessVar(this, k, ann, getEnclosingType(), getEnclosingCodeSignature(), true));
        }
    }

    void weaveBefore(BcelAdvice munger) {
        range.insert(munger.getAdviceInstructions(this, null, range.getRealStart()), Range.InsideBefore);
    }

    public void weaveAfter(BcelAdvice munger) {
        weaveAfterThrowing(munger, UnresolvedType.THROWABLE);
        weaveAfterReturning(munger);
    }

    public void weaveAfterReturning(BcelAdvice munger) {
        List<InstructionHandle> returns = findReturnInstructions();
        boolean hasReturnInstructions = !returns.isEmpty();
        InstructionList retList = new InstructionList();
        BcelVar returnValueVar = null;
        if (hasReturnInstructions) {
            returnValueVar = generateReturnInstructions(returns, retList);
        } else {
            retList.append(InstructionConstants.NOP);
        }
        InstructionList advice = getAfterReturningAdviceDispatchInstructions(munger, retList.getStart());
        if (hasReturnInstructions) {
            InstructionHandle gotoTarget = advice.getStart();
            for (InstructionHandle ih : returns) {
                retargetReturnInstruction(munger.hasExtraParameter(), returnValueVar, gotoTarget, ih);
            }
        }
        range.append(advice);
        range.append(retList);
    }

    private List<InstructionHandle> findReturnInstructions() {
        List<InstructionHandle> returns = new ArrayList<>();
        for (InstructionHandle ih = range.getStart(); ih != range.getEnd(); ih = ih.getNext()) {
            if (ih.getInstruction().isReturnInstruction()) {
                returns.add(ih);
            }
        }
        return returns;
    }

    private BcelVar generateReturnInstructions(List<InstructionHandle> returns, InstructionList returnInstructions) {
        BcelVar returnValueVar = null;
        if (this.hasANonVoidReturnType()) {
            Instruction newReturnInstruction = null;
            int i = returns.size() - 1;
            while (newReturnInstruction == null && i >= 0) {
                InstructionHandle ih = returns.get(i);
                if (ih.getInstruction().opcode != Constants.RETURN) {
                    newReturnInstruction = Utility.copyInstruction(ih.getInstruction());
                }
                i--;
            }
            returnValueVar = genTempVar(this.getReturnType());
            returnValueVar.appendLoad(returnInstructions, getFactory());
            returnInstructions.append(newReturnInstruction);
        } else {
            InstructionHandle lastReturnHandle = returns.get(returns.size() - 1);
            Instruction newReturnInstruction = Utility.copyInstruction(lastReturnHandle.getInstruction());
            returnInstructions.append(newReturnInstruction);
        }
        return returnValueVar;
    }

    private boolean hasANonVoidReturnType() {
        return !this.getReturnType().equals(UnresolvedType.VOID);
    }

    private InstructionList getAfterReturningAdviceDispatchInstructions(BcelAdvice munger, InstructionHandle firstInstructionInReturnSequence) {
        InstructionList advice = new InstructionList();
        BcelVar tempVar = null;
        if (munger.hasExtraParameter()) {
            tempVar = insertAdviceInstructionsForBindingReturningParameter(advice);
        }
        advice.append(munger.getAdviceInstructions(this, tempVar, firstInstructionInReturnSequence));
        return advice;
    }

    private BcelVar insertAdviceInstructionsForBindingReturningParameter(InstructionList advice) {
        BcelVar tempVar;
        UnresolvedType tempVarType = getReturnType();
        if (tempVarType.equals(UnresolvedType.VOID)) {
            tempVar = genTempVar(UnresolvedType.OBJECT);
            advice.append(InstructionConstants.ACONST_NULL);
            tempVar.appendStore(advice, getFactory());
        } else {
            tempVar = genTempVar(tempVarType);
            advice.append(InstructionFactory.createDup(tempVarType.getSize()));
            tempVar.appendStore(advice, getFactory());
        }
        return tempVar;
    }

    private void retargetReturnInstruction(boolean hasReturningParameter, BcelVar returnValueVar, InstructionHandle gotoTarget, InstructionHandle returnHandle) {
        InstructionList newInstructions = new InstructionList();
        if (returnValueVar != null) {
            if (hasReturningParameter) {
                newInstructions.append(InstructionFactory.createDup(this.getReturnType().getSize()));
            }
            returnValueVar.appendStore(newInstructions, getFactory());
        }
        if (!isLastInstructionInRange(returnHandle, range)) {
            newInstructions.append(InstructionFactory.createBranchInstruction(Constants.GOTO, gotoTarget));
        }
        if (newInstructions.isEmpty()) {
            newInstructions.append(InstructionConstants.NOP);
        }
        Utility.replaceInstruction(returnHandle, newInstructions, enclosingMethod);
    }

    private boolean isLastInstructionInRange(InstructionHandle ih, ShadowRange aRange) {
        return ih.getNext() == aRange.getEnd();
    }

    public void weaveAfterThrowing(BcelAdvice munger, UnresolvedType catchType) {
        if (getRange().getStart().getNext() == getRange().getEnd()) {
            return;
        }
        InstructionFactory fact = getFactory();
        InstructionList handler = new InstructionList();
        BcelVar exceptionVar = genTempVar(catchType);
        exceptionVar.appendStore(handler, fact);
        if (this.getEnclosingMethod().getName().equals("<clinit>")) {
            ResolvedType eiieType = world.resolve("java.lang.ExceptionInInitializerError");
            ObjectType eiieBcelType = (ObjectType) BcelWorld.makeBcelType(eiieType);
            InstructionList ih = new InstructionList(InstructionConstants.NOP);
            handler.append(exceptionVar.createLoad(fact));
            handler.append(fact.createInstanceOf(eiieBcelType));
            InstructionBranch bi = InstructionFactory.createBranchInstruction(Constants.IFEQ, ih.getStart());
            handler.append(bi);
            handler.append(exceptionVar.createLoad(fact));
            handler.append(fact.createCheckCast(eiieBcelType));
            handler.append(InstructionConstants.ATHROW);
            handler.append(ih);
        }
        InstructionList endHandler = new InstructionList(exceptionVar.createLoad(fact));
        handler.append(munger.getAdviceInstructions(this, exceptionVar, endHandler.getStart()));
        handler.append(endHandler);
        handler.append(InstructionConstants.ATHROW);
        InstructionHandle handlerStart = handler.getStart();
        if (isFallsThrough()) {
            InstructionHandle jumpTarget = handler.append(InstructionConstants.NOP);
            handler.insert(InstructionFactory.createBranchInstruction(Constants.GOTO, jumpTarget));
        }
        InstructionHandle protectedEnd = handler.getStart();
        range.insert(handler, Range.InsideAfter);
        enclosingMethod.addExceptionHandler(range.getStart().getNext(), protectedEnd.getPrev(), handlerStart, (ObjectType) BcelWorld.makeBcelType(catchType), getKind().hasHighPriorityExceptions());
    }

    public void weaveSoftener(BcelAdvice munger, UnresolvedType catchType) {
        if (getRange().getStart().getNext() == getRange().getEnd()) {
            return;
        }
        InstructionFactory fact = getFactory();
        InstructionList handler = new InstructionList();
        InstructionList rtExHandler = new InstructionList();
        BcelVar exceptionVar = genTempVar(catchType);
        handler.append(fact.createNew(NameMangler.SOFT_EXCEPTION_TYPE));
        handler.append(InstructionFactory.createDup(1));
        handler.append(exceptionVar.createLoad(fact));
        handler.append(fact.createInvoke(NameMangler.SOFT_EXCEPTION_TYPE, "<init>", Type.VOID, new Type[]{Type.THROWABLE}, Constants.INVOKESPECIAL));
        handler.append(InstructionConstants.ATHROW);
        exceptionVar.appendStore(rtExHandler, fact);
        rtExHandler.append(exceptionVar.createLoad(fact));
        rtExHandler.append(fact.createInstanceOf(new ObjectType("java.lang.RuntimeException")));
        rtExHandler.append(InstructionFactory.createBranchInstruction(Constants.IFEQ, handler.getStart()));
        rtExHandler.append(exceptionVar.createLoad(fact));
        rtExHandler.append(InstructionFactory.ATHROW);
        InstructionHandle handlerStart = rtExHandler.getStart();
        if (isFallsThrough()) {
            InstructionHandle jumpTarget = range.getEnd();
            rtExHandler.insert(InstructionFactory.createBranchInstruction(Constants.GOTO, jumpTarget));
        }
        rtExHandler.append(handler);
        InstructionHandle protectedEnd = rtExHandler.getStart();
        range.insert(rtExHandler, Range.InsideAfter);
        enclosingMethod.addExceptionHandler(range.getStart().getNext(), protectedEnd.getPrev(), handlerStart, (ObjectType) BcelWorld.makeBcelType(catchType), getKind().hasHighPriorityExceptions());
    }

    public void weavePerObjectEntry(final BcelAdvice munger, final BcelVar onVar) {
        final InstructionFactory fact = getFactory();
        InstructionList entryInstructions = new InstructionList();
        InstructionList entrySuccessInstructions = new InstructionList();
        onVar.appendLoad(entrySuccessInstructions, fact);
        entrySuccessInstructions.append(Utility.createInvoke(fact, world, AjcMemberMaker.perObjectBind(munger.getConcreteAspect())));
        InstructionList testInstructions = munger.getTestInstructions(this, entrySuccessInstructions.getStart(), range.getRealStart(), entrySuccessInstructions.getStart());
        entryInstructions.append(testInstructions);
        entryInstructions.append(entrySuccessInstructions);
        range.insert(entryInstructions, Range.InsideBefore);
    }

    public void weavePerTypeWithinAspectInitialization(final BcelAdvice munger, UnresolvedType t) {
        ResolvedType tResolved = t.resolve(world);
        if (tResolved.isInterface()) {
            return;
        }
        ResolvedType aspectRT = munger.getConcreteAspect();
        BcelWorld.getBcelObjectType(aspectRT);
        if (!(tResolved.canBeSeenBy(aspectRT) || aspectRT.isPrivilegedAspect())) {
            return;
        }
        final InstructionFactory fact = getFactory();
        InstructionList entryInstructions = new InstructionList();
        InstructionList entrySuccessInstructions = new InstructionList();
        String aspectname = munger.getConcreteAspect().getName();
        String ptwField = NameMangler.perTypeWithinFieldForTarget(munger.getConcreteAspect());
        entrySuccessInstructions.append(InstructionFactory.PUSH(fact.getConstantPool(), t.getName()));
        entrySuccessInstructions.append(fact.createInvoke(aspectname, "ajc$createAspectInstance", new ObjectType(aspectname), new Type[]{new ObjectType("java.lang.String")}, Constants.INVOKESTATIC));
        entrySuccessInstructions.append(fact.createPutStatic(t.getName(), ptwField, new ObjectType(aspectname)));
        entryInstructions.append(entrySuccessInstructions);
        range.insert(entryInstructions, Range.InsideBefore);
    }

    public void weaveCflowEntry(final BcelAdvice munger, final Member cflowField) {
        final boolean isPer = munger.getKind() == AdviceKind.PerCflowBelowEntry || munger.getKind() == AdviceKind.PerCflowEntry;
        if (!isPer && getKind() == PreInitialization) {
            return;
        }
        final Type objectArrayType = new ArrayType(Type.OBJECT, 1);
        final InstructionFactory fact = getFactory();
        final BcelVar testResult = genTempVar(UnresolvedType.BOOLEAN);
        InstructionList entryInstructions = new InstructionList();
        {
            InstructionList entrySuccessInstructions = new InstructionList();
            if (munger.hasDynamicTests()) {
                entryInstructions.append(Utility.createConstant(fact, 0));
                testResult.appendStore(entryInstructions, fact);
                entrySuccessInstructions.append(Utility.createConstant(fact, 1));
                testResult.appendStore(entrySuccessInstructions, fact);
            }
            if (isPer) {
                entrySuccessInstructions.append(fact.createInvoke(munger.getConcreteAspect().getName(), NameMangler.PERCFLOW_PUSH_METHOD, Type.VOID, Type.NO_ARGS, Constants.INVOKESTATIC));
            } else {
                BcelVar[] cflowStateVars = munger.getExposedStateAsBcelVars(false);
                if (cflowStateVars.length == 0) {
                    if (!cflowField.getType().getName().endsWith("CFlowCounter")) {
                        throw new RuntimeException("Incorrectly attempting counter operation on stacked cflow");
                    }
                    entrySuccessInstructions.append(Utility.createGet(fact, cflowField));
                    entrySuccessInstructions.append(fact.createInvoke(NameMangler.CFLOW_COUNTER_TYPE, "inc", Type.VOID, Type.NO_ARGS, Constants.INVOKEVIRTUAL));
                } else {
                    BcelVar arrayVar = genTempVar(UnresolvedType.OBJECTARRAY);
                    int alen = cflowStateVars.length;
                    entrySuccessInstructions.append(Utility.createConstant(fact, alen));
                    entrySuccessInstructions.append(fact.createNewArray(Type.OBJECT, (short) 1));
                    arrayVar.appendStore(entrySuccessInstructions, fact);
                    for (int i = 0; i < alen; i++) {
                        arrayVar.appendConvertableArrayStore(entrySuccessInstructions, fact, i, cflowStateVars[i]);
                    }
                    entrySuccessInstructions.append(Utility.createGet(fact, cflowField));
                    arrayVar.appendLoad(entrySuccessInstructions, fact);
                    entrySuccessInstructions.append(fact.createInvoke(NameMangler.CFLOW_STACK_TYPE, "push", Type.VOID, new Type[]{objectArrayType}, Constants.INVOKEVIRTUAL));
                }
            }
            InstructionList testInstructions = munger.getTestInstructions(this, entrySuccessInstructions.getStart(), range.getRealStart(), entrySuccessInstructions.getStart());
            entryInstructions.append(testInstructions);
            entryInstructions.append(entrySuccessInstructions);
        }
        BcelAdvice exitAdvice = new BcelAdvice(null, null, null, 0, 0, 0, null, munger.getConcreteAspect()) {

            @Override
            public InstructionList getAdviceInstructions(BcelShadow s, BcelVar extraArgVar, InstructionHandle ifNoAdvice) {
                InstructionList exitInstructions = new InstructionList();
                if (munger.hasDynamicTests()) {
                    testResult.appendLoad(exitInstructions, fact);
                    exitInstructions.append(InstructionFactory.createBranchInstruction(Constants.IFEQ, ifNoAdvice));
                }
                exitInstructions.append(Utility.createGet(fact, cflowField));
                if (munger.getKind() != AdviceKind.PerCflowEntry && munger.getKind() != AdviceKind.PerCflowBelowEntry && munger.getExposedStateAsBcelVars(false).length == 0) {
                    exitInstructions.append(fact.createInvoke(NameMangler.CFLOW_COUNTER_TYPE, "dec", Type.VOID, Type.NO_ARGS, Constants.INVOKEVIRTUAL));
                } else {
                    exitInstructions.append(fact.createInvoke(NameMangler.CFLOW_STACK_TYPE, "pop", Type.VOID, Type.NO_ARGS, Constants.INVOKEVIRTUAL));
                }
                return exitInstructions;
            }
        };
        weaveAfter(exitAdvice);
        range.insert(entryInstructions, Range.InsideBefore);
    }

    public void weaveAroundInline(BcelAdvice munger, boolean hasDynamicTest) {
        Member mungerSig = munger.getSignature();
        if (mungerSig instanceof ResolvedMember) {
            ResolvedMember rm = (ResolvedMember) mungerSig;
            if (rm.hasBackingGenericMember()) {
                mungerSig = rm.getBackingGenericMember();
            }
        }
        ResolvedType declaringAspectType = world.resolve(mungerSig.getDeclaringType(), true);
        if (declaringAspectType.isMissing()) {
            world.getLint().cantFindType.signal(new String[]{WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE_DURING_AROUND_WEAVE, declaringAspectType.getClassName())}, getSourceLocation(), new ISourceLocation[]{munger.getSourceLocation()});
        }
        ResolvedType rt = (declaringAspectType.isParameterizedType() ? declaringAspectType.getGenericType() : declaringAspectType);
        BcelObjectType ot = BcelWorld.getBcelObjectType(rt);
        LazyMethodGen adviceMethod = ot.getLazyClassGen().getLazyMethodGen(mungerSig);
        if (!adviceMethod.getCanInline()) {
            weaveAroundClosure(munger, hasDynamicTest);
            return;
        }
        if (isAnnotationStylePassingProceedingJoinPointOutOfAdvice(munger, hasDynamicTest, adviceMethod)) {
            return;
        }
        enclosingMethod.setCanInline(false);
        LazyClassGen shadowClass = getEnclosingClass();
        String extractedShadowMethodName = NameMangler.aroundShadowMethodName(getSignature(), shadowClass.getNewGeneratedNameTag());
        List<String> parameterNames = new ArrayList<>();
        boolean shadowClassIsInterface = shadowClass.isInterface();
        LazyMethodGen extractedShadowMethod = extractShadowInstructionsIntoNewMethod(extractedShadowMethodName, shadowClassIsInterface ? Modifier.PUBLIC : Modifier.PRIVATE, munger.getSourceLocation(), parameterNames, shadowClassIsInterface);
        List<BcelVar> argsToCallLocalAdviceMethodWith = new ArrayList<>();
        List<BcelVar> proceedVarList = new ArrayList<>();
        int extraParamOffset = 0;
        if (thisVar != null) {
            argsToCallLocalAdviceMethodWith.add(thisVar);
            proceedVarList.add(new BcelVar(thisVar.getType(), extraParamOffset));
            extraParamOffset += thisVar.getType().getSize();
        }
        if (targetVar != null && targetVar != thisVar) {
            argsToCallLocalAdviceMethodWith.add(targetVar);
            proceedVarList.add(new BcelVar(targetVar.getType(), extraParamOffset));
            extraParamOffset += targetVar.getType().getSize();
        }
        for (int i = 0, len = getArgCount(); i < len; i++) {
            argsToCallLocalAdviceMethodWith.add(argVars[i]);
            proceedVarList.add(new BcelVar(argVars[i].getType(), extraParamOffset));
            extraParamOffset += argVars[i].getType().getSize();
        }
        if (thisJoinPointVar != null) {
            argsToCallLocalAdviceMethodWith.add(thisJoinPointVar);
            proceedVarList.add(new BcelVar(thisJoinPointVar.getType(), extraParamOffset));
            extraParamOffset += thisJoinPointVar.getType().getSize();
        }
        Type[] adviceParameterTypes = BcelWorld.makeBcelTypes(munger.getSignature().getParameterTypes());
        adviceMethod.getArgumentTypes();
        Type[] extractedMethodParameterTypes = extractedShadowMethod.getArgumentTypes();
        Type[] parameterTypes = new Type[extractedMethodParameterTypes.length + adviceParameterTypes.length + 1];
        int parameterIndex = 0;
        System.arraycopy(extractedMethodParameterTypes, 0, parameterTypes, parameterIndex, extractedMethodParameterTypes.length);
        parameterIndex += extractedMethodParameterTypes.length;
        parameterTypes[parameterIndex++] = BcelWorld.makeBcelType(adviceMethod.getEnclosingClass().getType());
        System.arraycopy(adviceParameterTypes, 0, parameterTypes, parameterIndex, adviceParameterTypes.length);
        String localAdviceMethodName = NameMangler.aroundAdviceMethodName(getSignature(), shadowClass.getNewGeneratedNameTag());
        int localAdviceMethodModifiers = Modifier.PRIVATE | (world.useFinal() & !shadowClassIsInterface ? Modifier.FINAL : 0) | Modifier.STATIC;
        LazyMethodGen localAdviceMethod = new LazyMethodGen(localAdviceMethodModifiers, BcelWorld.makeBcelType(mungerSig.getReturnType()), localAdviceMethodName, parameterTypes, NoDeclaredExceptions, shadowClass);
        shadowClass.addMethodGen(localAdviceMethod);
        int nVars = adviceMethod.getMaxLocals() + extraParamOffset;
        IntMap varMap = IntMap.idMap(nVars);
        for (int i = extraParamOffset; i < nVars; i++) {
            varMap.put(i - extraParamOffset, i);
        }
        final InstructionFactory fact = getFactory();
        localAdviceMethod.getBody().insert(BcelClassWeaver.genInlineInstructions(adviceMethod, localAdviceMethod, varMap, fact, true));
        localAdviceMethod.setMaxLocals(nVars);
        InstructionList advice = new InstructionList();
        {
            for (BcelVar var : argsToCallLocalAdviceMethodWith) {
                var.appendLoad(advice, fact);
            }
            boolean isAnnoStyleConcreteAspect = munger.getConcreteAspect().isAnnotationStyleAspect();
            boolean isAnnoStyleDeclaringAspect = munger.getDeclaringAspect() != null ? munger.getDeclaringAspect().resolve(world).isAnnotationStyleAspect() : false;
            InstructionList iList = null;
            if (isAnnoStyleConcreteAspect && isAnnoStyleDeclaringAspect) {
                iList = this.loadThisJoinPoint();
                iList.append(Utility.createConversion(getFactory(), LazyClassGen.tjpType, LazyClassGen.proceedingTjpType));
            } else {
                iList = new InstructionList(InstructionConstants.ACONST_NULL);
            }
            advice.append(munger.getAdviceArgSetup(this, null, iList));
            advice.append(Utility.createInvoke(fact, localAdviceMethod));
            advice.append(Utility.createConversion(getFactory(), BcelWorld.makeBcelType(mungerSig.getReturnType()), extractedShadowMethod.getReturnType(), world.isInJava5Mode()));
            if (!isFallsThrough()) {
                advice.append(InstructionFactory.createReturn(extractedShadowMethod.getReturnType()));
            }
        }
        if (!hasDynamicTest) {
            range.append(advice);
        } else {
            InstructionList afterThingie = new InstructionList(InstructionConstants.NOP);
            InstructionList callback = makeCallToCallback(extractedShadowMethod);
            if (terminatesWithReturn()) {
                callback.append(InstructionFactory.createReturn(extractedShadowMethod.getReturnType()));
            } else {
                advice.append(InstructionFactory.createBranchInstruction(Constants.GOTO, afterThingie.getStart()));
            }
            range.append(munger.getTestInstructions(this, advice.getStart(), callback.getStart(), advice.getStart()));
            range.append(advice);
            range.append(callback);
            range.append(afterThingie);
        }
        if (!munger.getDeclaringType().isAnnotationStyleAspect()) {
            String proceedName = NameMangler.proceedMethodName(munger.getSignature().getName());
            InstructionHandle curr = localAdviceMethod.getBody().getStart();
            InstructionHandle end = localAdviceMethod.getBody().getEnd();
            ConstantPool cpg = localAdviceMethod.getEnclosingClass().getConstantPool();
            while (curr != end) {
                InstructionHandle next = curr.getNext();
                Instruction inst = curr.getInstruction();
                if ((inst.opcode == Constants.INVOKESTATIC) && proceedName.equals(((InvokeInstruction) inst).getMethodName(cpg))) {
                    localAdviceMethod.getBody().append(curr, getRedoneProceedCall(fact, extractedShadowMethod, munger, localAdviceMethod, proceedVarList));
                    Utility.deleteInstruction(curr, localAdviceMethod);
                }
                curr = next;
            }
        } else {
            InstructionHandle curr = localAdviceMethod.getBody().getStart();
            InstructionHandle end = localAdviceMethod.getBody().getEnd();
            ConstantPool cpg = localAdviceMethod.getEnclosingClass().getConstantPool();
            while (curr != end) {
                InstructionHandle next = curr.getNext();
                Instruction inst = curr.getInstruction();
                if ((inst instanceof INVOKEINTERFACE) && "proceed".equals(((INVOKEINTERFACE) inst).getMethodName(cpg))) {
                    final boolean isProceedWithArgs;
                    if (((INVOKEINTERFACE) inst).getArgumentTypes(cpg).length == 1) {
                        isProceedWithArgs = true;
                    } else {
                        isProceedWithArgs = false;
                    }
                    InstructionList insteadProceedIl = getRedoneProceedCallForAnnotationStyle(fact, extractedShadowMethod, munger, localAdviceMethod, proceedVarList, isProceedWithArgs);
                    localAdviceMethod.getBody().append(curr, insteadProceedIl);
                    Utility.deleteInstruction(curr, localAdviceMethod);
                }
                curr = next;
            }
        }
        InstructionHandle start = localAdviceMethod.getBody().getStart();
        InstructionHandle end = localAdviceMethod.getBody().getEnd();
        while (start.getInstruction().opcode == Constants.IMPDEP1) {
            start = start.getNext();
        }
        while (end.getInstruction().opcode == Constants.IMPDEP1) {
            end = end.getPrev();
        }
        Type[] args = localAdviceMethod.getArgumentTypes();
        int argNumber = 0;
        for (int slot = 0; slot < extraParamOffset; argNumber++) {
            String argumentName = null;
            if (argNumber >= args.length || parameterNames.size() == 0 || argNumber >= parameterNames.size()) {
                argumentName = new StringBuilder("unknown").append(argNumber).toString();
            } else {
                argumentName = parameterNames.get(argNumber);
            }
            String argumentSignature = args[argNumber].getSignature();
            LocalVariableTag lvt = new LocalVariableTag(argumentSignature, argumentName, slot, 0);
            start.addTargeter(lvt);
            end.addTargeter(lvt);
            slot += args[argNumber].getSize();
        }
    }

    private boolean isAnnotationStylePassingProceedingJoinPointOutOfAdvice(BcelAdvice munger, boolean hasDynamicTest, LazyMethodGen adviceMethod) {
        if (munger.getConcreteAspect().isAnnotationStyleAspect()) {
            boolean canSeeProceedPassedToOther = false;
            InstructionHandle curr = adviceMethod.getBody().getStart();
            InstructionHandle end = adviceMethod.getBody().getEnd();
            ConstantPool cpg = adviceMethod.getEnclosingClass().getConstantPool();
            while (curr != end) {
                InstructionHandle next = curr.getNext();
                Instruction inst = curr.getInstruction();
                if ((inst instanceof InvokeInstruction) && ((InvokeInstruction) inst).getSignature(cpg).indexOf("Lorg/aspectj/lang/ProceedingJoinPoint;") > 0) {
                    canSeeProceedPassedToOther = true;
                    break;
                }
                curr = next;
            }
            if (canSeeProceedPassedToOther) {
                adviceMethod.setCanInline(false);
                weaveAroundClosure(munger, hasDynamicTest);
                return true;
            }
        }
        return false;
    }

    private InstructionList getRedoneProceedCall(InstructionFactory fact, LazyMethodGen callbackMethod, BcelAdvice munger, LazyMethodGen localAdviceMethod, List<BcelVar> argVarList) {
        InstructionList ret = new InstructionList();
        BcelVar[] adviceVars = munger.getExposedStateAsBcelVars(true);
        IntMap proceedMap = makeProceedArgumentMap(adviceVars);
        ResolvedType[] proceedParamTypes = world.resolve(munger.getSignature().getParameterTypes());
        if (munger.getBaseParameterCount() + 1 < proceedParamTypes.length) {
            int len = munger.getBaseParameterCount() + 1;
            ResolvedType[] newTypes = new ResolvedType[len];
            System.arraycopy(proceedParamTypes, 0, newTypes, 0, len);
            proceedParamTypes = newTypes;
        }
        BcelVar[] proceedVars = Utility.pushAndReturnArrayOfVars(proceedParamTypes, ret, fact, localAdviceMethod);
        Type[] stateTypes = callbackMethod.getArgumentTypes();
        for (int i = 0, len = stateTypes.length; i < len; i++) {
            Type stateType = stateTypes[i];
            ResolvedType stateTypeX = BcelWorld.fromBcel(stateType).resolve(world);
            if (proceedMap.hasKey(i)) {
                proceedVars[proceedMap.get(i)].appendLoadAndConvert(ret, fact, stateTypeX);
            } else {
                argVarList.get(i).appendLoad(ret, fact);
            }
        }
        ret.append(Utility.createInvoke(fact, callbackMethod));
        ret.append(Utility.createConversion(fact, callbackMethod.getReturnType(), BcelWorld.makeBcelType(munger.getSignature().getReturnType()), world.isInJava5Mode()));
        return ret;
    }

    private InstructionList getRedoneProceedCallForAnnotationStyle(InstructionFactory fact, LazyMethodGen callbackMethod, BcelAdvice munger, LazyMethodGen localAdviceMethod, List<BcelVar> argVarList, boolean isProceedWithArgs) {
        InstructionList ret = new InstructionList();
        if (isProceedWithArgs) {
            Type objectArrayType = Type.OBJECT_ARRAY;
            int theObjectArrayLocalNumber = localAdviceMethod.allocateLocal(objectArrayType);
            ret.append(InstructionFactory.createStore(objectArrayType, theObjectArrayLocalNumber));
            Type proceedingJpType = Type.getType("Lorg/aspectj/lang/ProceedingJoinPoint;");
            int pjpLocalNumber = localAdviceMethod.allocateLocal(proceedingJpType);
            ret.append(InstructionFactory.createStore(proceedingJpType, pjpLocalNumber));
            boolean pointcutBindsThis = bindsThis(munger);
            boolean pointcutBindsTarget = bindsTarget(munger);
            boolean targetIsSameAsThis = getKind().isTargetSameAsThis();
            int nextArgumentToProvideForCallback = 0;
            if (hasThis()) {
                if (!(pointcutBindsTarget && targetIsSameAsThis)) {
                    if (pointcutBindsThis) {
                        ret.append(InstructionFactory.createLoad(objectArrayType, theObjectArrayLocalNumber));
                        ret.append(Utility.createConstant(fact, 0));
                        ret.append(InstructionFactory.createArrayLoad(Type.OBJECT));
                        ret.append(Utility.createConversion(fact, Type.OBJECT, callbackMethod.getArgumentTypes()[0]));
                    } else {
                        ret.append(InstructionFactory.createALOAD(0));
                    }
                    nextArgumentToProvideForCallback++;
                }
            }
            if (hasTarget()) {
                if (pointcutBindsTarget) {
                    if (getKind().isTargetSameAsThis()) {
                        ret.append(InstructionFactory.createLoad(objectArrayType, theObjectArrayLocalNumber));
                        ret.append(Utility.createConstant(fact, pointcutBindsThis ? 1 : 0));
                        ret.append(InstructionFactory.createArrayLoad(Type.OBJECT));
                        ret.append(Utility.createConversion(fact, Type.OBJECT, callbackMethod.getArgumentTypes()[0]));
                    } else {
                        int position = (hasThis() && pointcutBindsThis) ? 1 : 0;
                        ret.append(InstructionFactory.createLoad(objectArrayType, theObjectArrayLocalNumber));
                        ret.append(Utility.createConstant(fact, position));
                        ret.append(InstructionFactory.createArrayLoad(Type.OBJECT));
                        ret.append(Utility.createConversion(fact, Type.OBJECT, callbackMethod.getArgumentTypes()[nextArgumentToProvideForCallback]));
                    }
                    nextArgumentToProvideForCallback++;
                } else {
                    if (getKind().isTargetSameAsThis()) {
                    } else {
                        ret.append(InstructionFactory.createLoad(localAdviceMethod.getArgumentTypes()[0], hasThis() ? 1 : 0));
                        nextArgumentToProvideForCallback++;
                    }
                }
            }
            int indexIntoObjectArrayForArguments = (pointcutBindsThis ? 1 : 0) + (pointcutBindsTarget ? 1 : 0);
            int len = callbackMethod.getArgumentTypes().length;
            for (int i = nextArgumentToProvideForCallback; i < len; i++) {
                Type stateType = callbackMethod.getArgumentTypes()[i];
                BcelWorld.fromBcel(stateType).resolve(world);
                if ("Lorg/aspectj/lang/JoinPoint;".equals(stateType.getSignature())) {
                    ret.append(new InstructionLV(Constants.ALOAD, pjpLocalNumber));
                } else {
                    ret.append(InstructionFactory.createLoad(objectArrayType, theObjectArrayLocalNumber));
                    ret.append(Utility.createConstant(fact, i - nextArgumentToProvideForCallback + indexIntoObjectArrayForArguments));
                    ret.append(InstructionFactory.createArrayLoad(Type.OBJECT));
                    ret.append(Utility.createConversion(fact, Type.OBJECT, stateType));
                }
            }
        } else {
            Type proceedingJpType = Type.getType("Lorg/aspectj/lang/ProceedingJoinPoint;");
            int localJp = localAdviceMethod.allocateLocal(proceedingJpType);
            ret.append(InstructionFactory.createStore(proceedingJpType, localJp));
            int idx = 0;
            for (int i = 0, len = callbackMethod.getArgumentTypes().length; i < len; i++) {
                Type stateType = callbackMethod.getArgumentTypes()[i];
                BcelWorld.fromBcel(stateType).resolve(world);
                if ("Lorg/aspectj/lang/JoinPoint;".equals(stateType.getSignature())) {
                    ret.append(InstructionFactory.createALOAD(localJp));
                    idx++;
                } else {
                    ret.append(InstructionFactory.createLoad(stateType, idx));
                    idx += stateType.getSize();
                }
            }
        }
        ret.append(Utility.createInvoke(fact, callbackMethod));
        if (!UnresolvedType.OBJECT.equals(munger.getSignature().getReturnType())) {
            ret.append(Utility.createConversion(fact, callbackMethod.getReturnType(), Type.OBJECT));
        }
        ret.append(Utility.createConversion(fact, callbackMethod.getReturnType(), BcelWorld.makeBcelType(munger.getSignature().getReturnType()), world.isInJava5Mode()));
        return ret;
    }

    private boolean bindsThis(BcelAdvice munger) {
        UsesThisVisitor utv = new UsesThisVisitor();
        munger.getPointcut().accept(utv, null);
        return utv.usesThis;
    }

    private boolean bindsTarget(BcelAdvice munger) {
        UsesTargetVisitor utv = new UsesTargetVisitor();
        munger.getPointcut().accept(utv, null);
        return utv.usesTarget;
    }

    private static class UsesThisVisitor extends AbstractPatternNodeVisitor {

        boolean usesThis = false;

        @Override
        public Object visit(ThisOrTargetPointcut node, Object data) {
            if (node.isThis() && node.isBinding()) {
                usesThis = true;
            }
            return node;
        }

        @Override
        public Object visit(AndPointcut node, Object data) {
            if (!usesThis) {
                node.getLeft().accept(this, data);
            }
            if (!usesThis) {
                node.getRight().accept(this, data);
            }
            return node;
        }

        @Override
        public Object visit(NotPointcut node, Object data) {
            if (!usesThis) {
                node.getNegatedPointcut().accept(this, data);
            }
            return node;
        }

        @Override
        public Object visit(OrPointcut node, Object data) {
            if (!usesThis) {
                node.getLeft().accept(this, data);
            }
            if (!usesThis) {
                node.getRight().accept(this, data);
            }
            return node;
        }
    }

    private static class UsesTargetVisitor extends AbstractPatternNodeVisitor {

        boolean usesTarget = false;

        @Override
        public Object visit(ThisOrTargetPointcut node, Object data) {
            if (!node.isThis() && node.isBinding()) {
                usesTarget = true;
            }
            return node;
        }

        @Override
        public Object visit(AndPointcut node, Object data) {
            if (!usesTarget) {
                node.getLeft().accept(this, data);
            }
            if (!usesTarget) {
                node.getRight().accept(this, data);
            }
            return node;
        }

        @Override
        public Object visit(NotPointcut node, Object data) {
            if (!usesTarget) {
                node.getNegatedPointcut().accept(this, data);
            }
            return node;
        }

        @Override
        public Object visit(OrPointcut node, Object data) {
            if (!usesTarget) {
                node.getLeft().accept(this, data);
            }
            if (!usesTarget) {
                node.getRight().accept(this, data);
            }
            return node;
        }
    }

    BcelVar aroundClosureInstance = null;

    public void weaveAroundClosure(BcelAdvice munger, boolean hasDynamicTest) {
        InstructionFactory fact = getFactory();
        enclosingMethod.setCanInline(false);
        int linenumber = getSourceLine();
        boolean shadowClassIsInterface = getEnclosingClass().isInterface();
        LazyMethodGen callbackMethod = extractShadowInstructionsIntoNewMethod(NameMangler.aroundShadowMethodName(getSignature(), getEnclosingClass().getNewGeneratedNameTag()), shadowClassIsInterface ? Modifier.PUBLIC : 0, munger.getSourceLocation(), new ArrayList<>(), shadowClassIsInterface);
        BcelVar[] adviceVars = munger.getExposedStateAsBcelVars(true);
        String closureClassName = NameMangler.makeClosureClassName(getEnclosingClass().getType(), getEnclosingClass().getNewGeneratedNameTag());
        Member constructorSig = new MemberImpl(Member.CONSTRUCTOR, UnresolvedType.forName(closureClassName), 0, "<init>", "([Ljava/lang/Object;)V");
        BcelVar closureHolder = null;
        if (getKind() == PreInitialization) {
            closureHolder = genTempVar(AjcMemberMaker.AROUND_CLOSURE_TYPE);
        }
        InstructionList closureInstantiation = makeClosureInstantiation(constructorSig, closureHolder);
        makeClosureClassAndReturnConstructor(closureClassName, callbackMethod, makeProceedArgumentMap(adviceVars));
        InstructionList returnConversionCode;
        if (getKind() == PreInitialization) {
            returnConversionCode = new InstructionList();
            BcelVar stateTempVar = genTempVar(UnresolvedType.OBJECTARRAY);
            closureHolder.appendLoad(returnConversionCode, fact);
            returnConversionCode.append(Utility.createInvoke(fact, world, AjcMemberMaker.aroundClosurePreInitializationGetter()));
            stateTempVar.appendStore(returnConversionCode, fact);
            Type[] stateTypes = getSuperConstructorParameterTypes();
            returnConversionCode.append(InstructionConstants.ALOAD_0);
            for (int i = 0, len = stateTypes.length; i < len; i++) {
                UnresolvedType bcelTX = BcelWorld.fromBcel(stateTypes[i]);
                ResolvedType stateRTX = world.resolve(bcelTX, true);
                if (stateRTX.isMissing()) {
                    world.getLint().cantFindType.signal(new String[]{WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE_DURING_AROUND_WEAVE_PREINIT, bcelTX.getClassName())}, getSourceLocation(), new ISourceLocation[]{munger.getSourceLocation()});
                }
                stateTempVar.appendConvertableArrayLoad(returnConversionCode, fact, i, stateRTX);
            }
        } else {
            Member mungerSignature = munger.getSignature();
            if (munger.getSignature() instanceof ResolvedMember) {
                if (((ResolvedMember) mungerSignature).hasBackingGenericMember()) {
                    mungerSignature = ((ResolvedMember) mungerSignature).getBackingGenericMember();
                }
            }
            UnresolvedType returnType = mungerSignature.getReturnType();
            returnConversionCode = Utility.createConversion(getFactory(), BcelWorld.makeBcelType(returnType), callbackMethod.getReturnType(), world.isInJava5Mode());
            if (!isFallsThrough()) {
                returnConversionCode.append(InstructionFactory.createReturn(callbackMethod.getReturnType()));
            }
        }
        int bitflags = 0x000000;
        if (getKind().isTargetSameAsThis()) {
            bitflags |= 0x010000;
        }
        if (hasThis()) {
            bitflags |= 0x001000;
        }
        if (bindsThis(munger)) {
            bitflags |= 0x000100;
        }
        if (hasTarget()) {
            bitflags |= 0x000010;
        }
        if (bindsTarget(munger)) {
            bitflags |= 0x000001;
        }
        closureVarInitialized = false;
        if (munger.getConcreteAspect() != null && munger.getConcreteAspect().isAnnotationStyleAspect() && munger.getDeclaringAspect() != null && munger.getDeclaringAspect().resolve(world).isAnnotationStyleAspect()) {
            aroundClosureInstance = genTempVar(AjcMemberMaker.AROUND_CLOSURE_TYPE);
            closureInstantiation.append(fact.createDup(1));
            aroundClosureInstance.appendStore(closureInstantiation, fact);
            closureInstantiation.append(fact.createConstant(bitflags));
            if (needAroundClosureStacking) {
                closureInstantiation.append(Utility.createInvoke(getFactory(), getWorld(), new MemberImpl(Member.METHOD, UnresolvedType.forName("org.aspectj.runtime.internal.AroundClosure"), Modifier.PUBLIC, "linkStackClosureAndJoinPoint", String.format("%s%s", "(I)", "Lorg/aspectj/lang/ProceedingJoinPoint;"))));
            } else {
                closureInstantiation.append(Utility.createInvoke(getFactory(), getWorld(), new MemberImpl(Member.METHOD, UnresolvedType.forName("org.aspectj.runtime.internal.AroundClosure"), Modifier.PUBLIC, "linkClosureAndJoinPoint", String.format("%s%s", "(I)", "Lorg/aspectj/lang/ProceedingJoinPoint;"))));
            }
        }
        InstructionList advice = new InstructionList();
        advice.append(munger.getAdviceArgSetup(this, null, closureInstantiation));
        InstructionHandle tryUnlinkPosition = advice.append(munger.getNonTestAdviceInstructions(this));
        if (needAroundClosureStacking) {
            if (munger.getConcreteAspect() != null && munger.getConcreteAspect().isAnnotationStyleAspect() && munger.getDeclaringAspect() != null && munger.getDeclaringAspect().resolve(world).isAnnotationStyleAspect() && closureVarInitialized) {
                aroundClosureInstance.appendLoad(advice, fact);
                InstructionHandle unlinkInsn = advice.append(Utility.createInvoke(getFactory(), getWorld(), new MemberImpl(Member.METHOD, UnresolvedType.forName("org.aspectj.runtime.internal.AroundClosure"), Modifier.PUBLIC, "unlink", "()V")));
                BranchHandle jumpOverHandler = advice.append(new InstructionBranch(Constants.GOTO, null));
                InstructionHandle handlerStart = advice.append(aroundClosureInstance.createLoad(fact));
                advice.append(Utility.createInvoke(getFactory(), getWorld(), new MemberImpl(Member.METHOD, UnresolvedType.forName("org.aspectj.runtime.internal.AroundClosure"), Modifier.PUBLIC, "unlink", "()V")));
                advice.append(InstructionConstants.ATHROW);
                InstructionHandle jumpTarget = advice.append(InstructionConstants.NOP);
                jumpOverHandler.setTarget(jumpTarget);
                enclosingMethod.addExceptionHandler(tryUnlinkPosition, unlinkInsn, handlerStart, null, false);
            }
        }
        advice.append(returnConversionCode);
        if (getKind() == Shadow.MethodExecution && linenumber > 0) {
            advice.getStart().addTargeter(new LineNumberTag(linenumber));
        }
        if (!hasDynamicTest) {
            range.append(advice);
        } else {
            InstructionList callback = makeCallToCallback(callbackMethod);
            InstructionList postCallback = new InstructionList();
            if (terminatesWithReturn()) {
                callback.append(InstructionFactory.createReturn(callbackMethod.getReturnType()));
            } else {
                advice.append(InstructionFactory.createBranchInstruction(Constants.GOTO, postCallback.append(InstructionConstants.NOP)));
            }
            range.append(munger.getTestInstructions(this, advice.getStart(), callback.getStart(), advice.getStart()));
            range.append(advice);
            range.append(callback);
            range.append(postCallback);
        }
    }

    InstructionList makeCallToCallback(LazyMethodGen callbackMethod) {
        InstructionFactory fact = getFactory();
        InstructionList callback = new InstructionList();
        if (thisVar != null) {
            callback.append(InstructionConstants.ALOAD_0);
        }
        if (targetVar != null && targetVar != thisVar) {
            callback.append(BcelRenderer.renderExpr(fact, world, targetVar));
        }
        callback.append(BcelRenderer.renderExprs(fact, world, argVars));
        if (thisJoinPointVar != null) {
            callback.append(BcelRenderer.renderExpr(fact, world, thisJoinPointVar));
        }
        callback.append(Utility.createInvoke(fact, callbackMethod));
        return callback;
    }

    private InstructionList makeClosureInstantiation(Member constructor, BcelVar holder) {
        InstructionFactory fact = getFactory();
        BcelVar arrayVar = genTempVar(UnresolvedType.OBJECTARRAY);
        final InstructionList il = new InstructionList();
        int alen = getArgCount() + (thisVar == null ? 0 : 1) + ((targetVar != null && targetVar != thisVar) ? 1 : 0) + (thisJoinPointVar == null ? 0 : 1);
        il.append(Utility.createConstant(fact, alen));
        il.append(fact.createNewArray(Type.OBJECT, (short) 1));
        arrayVar.appendStore(il, fact);
        int stateIndex = 0;
        if (thisVar != null) {
            arrayVar.appendConvertableArrayStore(il, fact, stateIndex, thisVar);
            thisVar.setPositionInAroundState(stateIndex);
            stateIndex++;
        }
        if (targetVar != null && targetVar != thisVar) {
            arrayVar.appendConvertableArrayStore(il, fact, stateIndex, targetVar);
            targetVar.setPositionInAroundState(stateIndex);
            stateIndex++;
        }
        for (int i = 0, len = getArgCount(); i < len; i++) {
            arrayVar.appendConvertableArrayStore(il, fact, stateIndex, argVars[i]);
            argVars[i].setPositionInAroundState(stateIndex);
            stateIndex++;
        }
        if (thisJoinPointVar != null) {
            arrayVar.appendConvertableArrayStore(il, fact, stateIndex, thisJoinPointVar);
            thisJoinPointVar.setPositionInAroundState(stateIndex);
            stateIndex++;
        }
        il.append(fact.createNew(new ObjectType(constructor.getDeclaringType().getName())));
        il.append(InstructionConstants.DUP);
        arrayVar.appendLoad(il, fact);
        il.append(Utility.createInvoke(fact, world, constructor));
        if (getKind() == PreInitialization) {
            il.append(InstructionConstants.DUP);
            holder.appendStore(il, fact);
        }
        return il;
    }

    private IntMap makeProceedArgumentMap(BcelVar[] adviceArgs) {
        IntMap ret = new IntMap();
        for (int i = 0, len = adviceArgs.length; i < len; i++) {
            BcelVar v = adviceArgs[i];
            if (v == null) {
                continue;
            }
            int pos = v.getPositionInAroundState();
            if (pos >= 0) {
                ret.put(pos, i);
            }
        }
        return ret;
    }

    private LazyMethodGen makeClosureClassAndReturnConstructor(String closureClassName, LazyMethodGen callbackMethod, IntMap proceedMap) {
        String superClassName = "org.aspectj.runtime.internal.AroundClosure";
        Type objectArrayType = new ArrayType(Type.OBJECT, 1);
        LazyClassGen closureClass = new LazyClassGen(closureClassName, superClassName, getEnclosingClass().getFileName(), Modifier.PUBLIC, new String[]{}, getWorld());
        closureClass.setMajorMinor(getEnclosingClass().getMajor(), getEnclosingClass().getMinor());
        InstructionFactory fact = new InstructionFactory(closureClass.getConstantPool());
        LazyMethodGen constructor = new LazyMethodGen(Modifier.PUBLIC, Type.VOID, "<init>", new Type[]{objectArrayType}, new String[]{}, closureClass);
        InstructionList cbody = constructor.getBody();
        cbody.append(InstructionFactory.createLoad(Type.OBJECT, 0));
        cbody.append(InstructionFactory.createLoad(objectArrayType, 1));
        cbody.append(fact.createInvoke(superClassName, "<init>", Type.VOID, new Type[]{objectArrayType}, Constants.INVOKESPECIAL));
        cbody.append(InstructionFactory.createReturn(Type.VOID));
        closureClass.addMethodGen(constructor);
        LazyMethodGen runMethod = new LazyMethodGen(Modifier.PUBLIC, Type.OBJECT, "run", new Type[]{objectArrayType}, new String[]{}, closureClass);
        InstructionList mbody = runMethod.getBody();
        BcelVar proceedVar = new BcelVar(UnresolvedType.OBJECTARRAY.resolve(world), 1);
        BcelVar stateVar = new BcelVar(UnresolvedType.OBJECTARRAY.resolve(world), runMethod.allocateLocal(1));
        mbody.append(InstructionFactory.createThis());
        mbody.append(fact.createGetField(superClassName, "state", objectArrayType));
        mbody.append(stateVar.createStore(fact));
        Type[] stateTypes = callbackMethod.getArgumentTypes();
        for (int i = 0, len = stateTypes.length; i < len; i++) {
            ResolvedType resolvedStateType = BcelWorld.fromBcel(stateTypes[i]).resolve(world);
            if (proceedMap.hasKey(i)) {
                mbody.append(proceedVar.createConvertableArrayLoad(fact, proceedMap.get(i), resolvedStateType));
            } else {
                mbody.append(stateVar.createConvertableArrayLoad(fact, i, resolvedStateType));
            }
        }
        mbody.append(Utility.createInvoke(fact, callbackMethod));
        if (getKind() == PreInitialization) {
            mbody.append(Utility.createSet(fact, AjcMemberMaker.aroundClosurePreInitializationField()));
            mbody.append(InstructionConstants.ACONST_NULL);
        } else {
            mbody.append(Utility.createConversion(fact, callbackMethod.getReturnType(), Type.OBJECT));
        }
        mbody.append(InstructionFactory.createReturn(Type.OBJECT));
        closureClass.addMethodGen(runMethod);
        getEnclosingClass().addGeneratedInner(closureClass);
        return constructor;
    }

    LazyMethodGen extractShadowInstructionsIntoNewMethod(String extractedMethodName, int extractedMethodVisibilityModifier, ISourceLocation adviceSourceLocation, List<String> parameterNames, boolean beingPlacedInInterface) {
        if (!getKind().allowsExtraction()) {
            throw new BCException("Attempt to extract method from a shadow kind (" + getKind() + ") that does not support this operation");
        }
        LazyMethodGen newMethod = createShadowMethodGen(extractedMethodName, extractedMethodVisibilityModifier, parameterNames, beingPlacedInInterface);
        IntMap remapper = makeRemap();
        range.extractInstructionsInto(newMethod, remapper, (getKind() != PreInitialization) && isFallsThrough());
        if (getKind() == PreInitialization) {
            addPreInitializationReturnCode(newMethod, getSuperConstructorParameterTypes());
        }
        getEnclosingClass().addMethodGen(newMethod, adviceSourceLocation);
        return newMethod;
    }

    private void addPreInitializationReturnCode(LazyMethodGen extractedMethod, Type[] superConstructorTypes) {
        InstructionList body = extractedMethod.getBody();
        final InstructionFactory fact = getFactory();
        BcelVar arrayVar = new BcelVar(world.getCoreType(UnresolvedType.OBJECTARRAY), extractedMethod.allocateLocal(1));
        int len = superConstructorTypes.length;
        body.append(Utility.createConstant(fact, len));
        body.append(fact.createNewArray(Type.OBJECT, (short) 1));
        arrayVar.appendStore(body, fact);
        for (int i = len - 1; i >= 0; i++) {
            body.append(Utility.createConversion(fact, superConstructorTypes[i], Type.OBJECT));
            arrayVar.appendLoad(body, fact);
            body.append(InstructionConstants.SWAP);
            body.append(Utility.createConstant(fact, i));
            body.append(InstructionConstants.SWAP);
            body.append(InstructionFactory.createArrayStore(Type.OBJECT));
        }
        arrayVar.appendLoad(body, fact);
        body.append(InstructionConstants.ARETURN);
    }

    private Type[] getSuperConstructorParameterTypes() {
        InstructionHandle superCallHandle = getRange().getEnd().getNext();
        InvokeInstruction superCallInstruction = (InvokeInstruction) superCallHandle.getInstruction();
        return superCallInstruction.getArgumentTypes(getEnclosingClass().getConstantPool());
    }

    private IntMap makeRemap() {
        IntMap ret = new IntMap(5);
        int reti = 0;
        if (thisVar != null) {
            ret.put(0, reti++);
        }
        if (targetVar != null && targetVar != thisVar) {
            ret.put(targetVar.getSlot(), reti++);
        }
        for (BcelVar argVar : argVars) {
            ret.put(argVar.getSlot(), reti);
            reti += argVar.getType().getSize();
        }
        if (thisJoinPointVar != null) {
            ret.put(thisJoinPointVar.getSlot(), reti++);
        }
        if (!getKind().argsOnStack()) {
            int oldi = 0;
            int newi = 0;
            if (arg0HoldsThis()) {
                ret.put(0, 0);
                oldi++;
                newi += 1;
            }
            for (int i = 0; i < getArgCount(); i++) {
                UnresolvedType type = getArgType(i);
                ret.put(oldi, newi);
                oldi += type.getSize();
                newi += type.getSize();
            }
        }
        return ret;
    }

    private LazyMethodGen createShadowMethodGen(String newMethodName, int visibilityModifier, List<String> parameterNames, boolean beingPlacedInInterface) {
        Type[] shadowParameterTypes = BcelWorld.makeBcelTypes(getArgTypes());
        int modifiers = (world.useFinal() && !beingPlacedInInterface ? Modifier.FINAL : 0) | Modifier.STATIC | visibilityModifier;
        if (targetVar != null && targetVar != thisVar) {
            UnresolvedType targetType = getTargetType();
            targetType = ensureTargetTypeIsCorrect(targetType);
            if ((getKind() == FieldGet || getKind() == FieldSet) && getActualTargetType() != null && !getActualTargetType().equals(targetType.getName())) {
                targetType = UnresolvedType.forName(getActualTargetType()).resolve(world);
            }
            ResolvedMember resolvedMember = getSignature().resolve(world);
            if (resolvedMember != null && Modifier.isProtected(resolvedMember.getModifiers()) && !samePackage(resolvedMember.getDeclaringType().getPackageName(), getEnclosingType().getPackageName()) && !resolvedMember.getName().equals("clone")) {
                if (!hasThis()) {
                    if (Modifier.isStatic(enclosingMethod.getAccessFlags()) && enclosingMethod.getName().startsWith("access$")) {
                        targetType = BcelWorld.fromBcel(enclosingMethod.getArgumentTypes()[0]);
                    }
                } else {
                    if (!targetType.resolve(world).isAssignableFrom(getThisType().resolve(world))) {
                        throw new BCException("bad bytecode");
                    }
                    targetType = getThisType();
                }
            }
            parameterNames.add("target");
            shadowParameterTypes = addTypeToFront(BcelWorld.makeBcelType(targetType), shadowParameterTypes);
        }
        if (thisVar != null) {
            UnresolvedType thisType = getThisType();
            parameterNames.add(0, "ajc$this");
            shadowParameterTypes = addTypeToFront(BcelWorld.makeBcelType(thisType), shadowParameterTypes);
        }
        if (this.getKind() == Shadow.FieldSet || this.getKind() == Shadow.FieldGet) {
            parameterNames.add(getSignature().getName());
        } else {
            String[] pnames = getSignature().getParameterNames(world);
            if (pnames != null) {
                for (int i = 0; i < pnames.length; i++) {
                    if (i == 0 && pnames[i].equals("this")) {
                        parameterNames.add("ajc$this");
                    } else {
                        parameterNames.add(pnames[i]);
                    }
                }
            }
        }
        if (thisJoinPointVar != null) {
            parameterNames.add("thisJoinPoint");
            shadowParameterTypes = addTypeToEnd(LazyClassGen.tjpType, shadowParameterTypes);
        }
        UnresolvedType returnType;
        if (getKind() == PreInitialization) {
            returnType = UnresolvedType.OBJECTARRAY;
        } else {
            if (getKind() == ConstructorCall) {
                returnType = getSignature().getDeclaringType();
            } else if (getKind() == FieldSet) {
                returnType = UnresolvedType.VOID;
            } else {
                returnType = getSignature().getReturnType().resolve(world);
            }
        }
        return new LazyMethodGen(modifiers, BcelWorld.makeBcelType(returnType), newMethodName, shadowParameterTypes, NoDeclaredExceptions, getEnclosingClass());
    }

    private boolean samePackage(String p1, String p2) {
        if (p1 == null) {
            return p2 == null;
        }
        if (p2 == null) {
            return false;
        }
        return p1.equals(p2);
    }

    private Type[] addTypeToFront(Type type, Type[] types) {
        int len = types.length;
        Type[] ret = new Type[len + 1];
        ret[0] = type;
        System.arraycopy(types, 0, ret, 1, len);
        return ret;
    }

    private Type[] addTypeToEnd(Type type, Type[] types) {
        int len = types.length;
        Type[] ret = new Type[len + 1];
        ret[len] = type;
        System.arraycopy(types, 0, ret, 0, len);
        return ret;
    }

    public BcelVar genTempVar(UnresolvedType utype) {
        ResolvedType rtype = utype.resolve(world);
        return new BcelVar(rtype, genTempVarIndex(rtype.getSize()));
    }

    public BcelVar genTempVar(UnresolvedType typeX, String localName) {
        BcelVar tv = genTempVar(typeX);
        return tv;
    }

    private int genTempVarIndex(int size) {
        return enclosingMethod.allocateLocal(size);
    }

    public InstructionFactory getFactory() {
        return getEnclosingClass().getFactory();
    }

    @Override
    public ISourceLocation getSourceLocation() {
        int sourceLine = getSourceLine();
        if (sourceLine == 0 || sourceLine == -1) {
            return getEnclosingClass().getType().getSourceLocation();
        } else {
            if (getKind() == Shadow.StaticInitialization && getEnclosingClass().getType().getSourceLocation().getOffset() != 0) {
                return getEnclosingClass().getType().getSourceLocation();
            } else {
                int offset = 0;
                Kind kind = getKind();
                if ((kind == MethodExecution) || (kind == ConstructorExecution) || (kind == AdviceExecution) || (kind == StaticInitialization) || (kind == PreInitialization) || (kind == Initialization)) {
                    if (getEnclosingMethod().hasDeclaredLineNumberInfo()) {
                        offset = getEnclosingMethod().getDeclarationOffset();
                    }
                }
                return getEnclosingClass().getType().getSourceContext().makeSourceLocation(sourceLine, offset);
            }
        }
    }

    public Shadow getEnclosingShadow() {
        return enclosingShadow;
    }

    public LazyMethodGen getEnclosingMethod() {
        return enclosingMethod;
    }

    public boolean isFallsThrough() {
        return !terminatesWithReturn();
    }

    public void setActualTargetType(String className) {
        this.actualInstructionTargetType = className;
    }

    public String getActualTargetType() {
        return actualInstructionTargetType;
    }
}
