package org.aspectj.weaver.bcel;

import org.aspectj.apache.bcel.classfile.ClassParser;
import org.aspectj.apache.bcel.classfile.JavaClass;
import org.aspectj.asm.AsmManager;
import org.aspectj.asm.IProgramElement;
import org.aspectj.asm.internal.AspectJElementHierarchy;
import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.Message;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.bridge.SourceLocation;
import org.aspectj.bridge.WeaveMessage;
import org.aspectj.bridge.context.CompilationAndWeavingContext;
import org.aspectj.bridge.context.ContextToken;
import org.aspectj.util.FileUtil;
import org.aspectj.util.FuzzyBoolean;
import org.aspectj.weaver.Advice;
import org.aspectj.weaver.AdviceKind;
import org.aspectj.weaver.AnnotationAJ;
import org.aspectj.weaver.AnnotationOnTypeMunger;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.CompressingDataOutputStream;
import org.aspectj.weaver.ConcreteTypeMunger;
import org.aspectj.weaver.CrosscuttingMembersSet;
import org.aspectj.weaver.CustomMungerFactory;
import org.aspectj.weaver.IClassFileProvider;
import org.aspectj.weaver.IUnwovenClassFile;
import org.aspectj.weaver.IWeaveRequestor;
import org.aspectj.weaver.NewParentTypeMunger;
import org.aspectj.weaver.ReferenceType;
import org.aspectj.weaver.ReferenceTypeDelegate;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.ResolvedTypeMunger;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.ShadowMunger;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.WeaverStateInfo;
import org.aspectj.weaver.World;
import org.aspectj.weaver.model.AsmRelationshipProvider;
import org.aspectj.weaver.patterns.AndPointcut;
import org.aspectj.weaver.patterns.BindingPattern;
import org.aspectj.weaver.patterns.BindingTypePattern;
import org.aspectj.weaver.patterns.ConcreteCflowPointcut;
import org.aspectj.weaver.patterns.DeclareAnnotation;
import org.aspectj.weaver.patterns.DeclareParents;
import org.aspectj.weaver.patterns.DeclareTypeErrorOrWarning;
import org.aspectj.weaver.patterns.FastMatchInfo;
import org.aspectj.weaver.patterns.IfPointcut;
import org.aspectj.weaver.patterns.KindedPointcut;
import org.aspectj.weaver.patterns.NameBindingPointcut;
import org.aspectj.weaver.patterns.NotPointcut;
import org.aspectj.weaver.patterns.OrPointcut;
import org.aspectj.weaver.patterns.Pointcut;
import org.aspectj.weaver.patterns.PointcutRewriter;
import org.aspectj.weaver.patterns.WithinPointcut;
import org.aspectj.weaver.tools.Trace;
import org.aspectj.weaver.tools.TraceFactory;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.Attributes.Name;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

public class BcelWeaver {

    public static final String CLOSURE_CLASS_PREFIX = "$Ajc";

    public static final String SYNTHETIC_CLASS_POSTFIX = "$ajc";

    private static Trace trace = TraceFactory.getTraceFactory().getTrace(BcelWeaver.class);

    private transient final BcelWorld world;

    private final CrosscuttingMembersSet xcutSet;

    private boolean inReweavableMode = false;

    private transient List<UnwovenClassFile> addedClasses = new ArrayList<>();

    private transient List<String> deletedTypenames = new ArrayList<>();

    private transient List<ShadowMunger> shadowMungerList = null;

    private transient List<ConcreteTypeMunger> typeMungerList = null;

    private transient List<ConcreteTypeMunger> lateTypeMungerList = null;

    private transient List<DeclareParents> declareParentsList = null;

    private Manifest manifest = null;

    private boolean needToReweaveWorld = false;

    private boolean isBatchWeave = true;

    private ZipOutputStream zipOutputStream;

    private CustomMungerFactory customMungerFactory;

    public BcelWeaver(BcelWorld world) {
        super();
        if (trace.isTraceEnabled()) {
            trace.enter("<init>", this, world);
        }
        this.world = world;
        this.xcutSet = world.getCrosscuttingMembersSet();
        if (trace.isTraceEnabled()) {
            trace.exit("<init>");
        }
    }

    public ResolvedType addLibraryAspect(String aspectName) {
        if (trace.isTraceEnabled()) {
            trace.enter("addLibraryAspect", this, aspectName);
        }
        UnresolvedType unresolvedT = UnresolvedType.forName(aspectName);
        unresolvedT.setNeedsModifiableDelegate(true);
        ResolvedType type = world.resolve(unresolvedT, true);
        if (type.isMissing()) {
            String fixedName = aspectName;
            int hasDot = fixedName.lastIndexOf('.');
            while (hasDot > 0) {
                char[] fixedNameChars = fixedName.toCharArray();
                fixedNameChars[hasDot] = '$';
                fixedName = new String(fixedNameChars);
                hasDot = fixedName.lastIndexOf('.');
                UnresolvedType ut = UnresolvedType.forName(fixedName);
                ut.setNeedsModifiableDelegate(true);
                type = world.resolve(ut, true);
                if (!type.isMissing()) {
                    break;
                }
            }
        }
        if (type.isAspect()) {
            WeaverStateInfo wsi = type.getWeaverState();
            if (wsi != null && wsi.isReweavable()) {
                BcelObjectType classType = getClassType(type.getName());
                JavaClass wovenJavaClass = classType.getJavaClass();
                byte[] bytes = wsi.getUnwovenClassFileData(wovenJavaClass.getBytes());
                JavaClass unwovenJavaClass = Utility.makeJavaClass(wovenJavaClass.getFileName(), bytes);
                world.storeClass(unwovenJavaClass);
                classType.setJavaClass(unwovenJavaClass, true);
            }
            xcutSet.addOrReplaceAspect(type);
            if (trace.isTraceEnabled()) {
                trace.exit("addLibraryAspect", type);
            }
            if (type.getSuperclass().isAspect()) {
                addLibraryAspect(type.getSuperclass().getName());
            }
            return type;
        } else {
            if (type.isMissing()) {
                IMessage message = new Message("The specified aspect '" + aspectName + "' cannot be found", null, true);
                world.getMessageHandler().handleMessage(message);
            } else {
                IMessage message = new Message("Cannot register '" + aspectName + "' because the type found with that name is not an aspect", null, true);
                world.getMessageHandler().handleMessage(message);
            }
            return null;
        }
    }

    public void addLibraryJarFile(File inFile) throws IOException {
        List<ResolvedType> addedAspects = null;
        if (inFile.isDirectory()) {
            addedAspects = addAspectsFromDirectory(inFile);
        } else {
            addedAspects = addAspectsFromJarFile(inFile);
        }
        for (ResolvedType addedAspect : addedAspects) {
            xcutSet.addOrReplaceAspect(addedAspect);
        }
    }

    private List<ResolvedType> addAspectsFromJarFile(File inFile) throws FileNotFoundException, IOException {
        ZipInputStream inStream = new ZipInputStream(new FileInputStream(inFile));
        List<ResolvedType> addedAspects = new ArrayList<>();
        try {
            while (true) {
                ZipEntry entry = inStream.getNextEntry();
                if (entry == null) {
                    break;
                }
                if (entry.isDirectory() || !entry.getName().endsWith(".class")) {
                    continue;
                }
                int size = (int) entry.getSize();
                ClassParser parser = new ClassParser(new ByteArrayInputStream(FileUtil.readAsByteArray(inStream)), entry.getName());
                JavaClass jc = parser.parse();
                inStream.closeEntry();
                ResolvedType type = world.addSourceObjectType(jc, false).getResolvedTypeX();
                type.setBinaryPath(inFile.getAbsolutePath());
                if (type.isAspect()) {
                    addedAspects.add(type);
                } else {
                    world.demote(type);
                }
            }
        } finally {
            inStream.close();
        }
        return addedAspects;
    }

    private List<ResolvedType> addAspectsFromDirectory(File directory) throws FileNotFoundException, IOException {
        List<ResolvedType> addedAspects = new ArrayList<>();
        File[] classFiles = FileUtil.listFiles(directory, new FileFilter() {

            public boolean accept(File pathname) {
                return pathname.getName().endsWith(".class");
            }
        });
        for (File classFile : classFiles) {
            FileInputStream fis = new FileInputStream(classFile);
            byte[] classBytes = FileUtil.readAsByteArray(fis);
            ResolvedType aspectType = isAspect(classBytes, classFile.getAbsolutePath(), directory);
            if (aspectType != null) {
                addedAspects.add(aspectType);
            }
            fis.close();
        }
        return addedAspects;
    }

    private ResolvedType isAspect(byte[] classbytes, String name, File dir) throws IOException {
        ClassParser parser = new ClassParser(new ByteArrayInputStream(classbytes), name);
        JavaClass jc = parser.parse();
        ResolvedType type = world.addSourceObjectType(jc, false).getResolvedTypeX();
        String typeName = type.getName().replace('.', File.separatorChar);
        int end = name.lastIndexOf(typeName + ".class");
        String binaryPath = null;
        if (end == -1) {
            binaryPath = dir.getAbsolutePath();
        } else {
            binaryPath = name.substring(0, end - 1);
        }
        type.setBinaryPath(binaryPath);
        if (type.isAspect()) {
            return type;
        } else {
            world.demote(type);
            return null;
        }
    }

    public List<UnwovenClassFile> addDirectoryContents(File inFile, File outDir) throws IOException {
        List<UnwovenClassFile> addedClassFiles = new ArrayList<>();
        File[] files = FileUtil.listFiles(inFile, new FileFilter() {

            public boolean accept(File f) {
                boolean accept = !f.isDirectory();
                return accept;
            }
        });
        for (File file : files) {
            addedClassFiles.add(addClassFile(file, inFile, outDir));
        }
        return addedClassFiles;
    }

    public List<UnwovenClassFile> addJarFile(File inFile, File outDir, boolean canBeDirectory) {
        List<UnwovenClassFile> addedClassFiles = new ArrayList<>();
        needToReweaveWorld = true;
        JarFile inJar = null;
        try {
            if (inFile.isDirectory() && canBeDirectory) {
                addedClassFiles.addAll(addDirectoryContents(inFile, outDir));
            } else {
                inJar = new JarFile(inFile);
                try {
                    addManifest(inJar.getManifest());
                    Enumeration<JarEntry> entries = inJar.entries();
                    while (entries.hasMoreElements()) {
                        JarEntry entry = (JarEntry) entries.nextElement();
                        InputStream inStream = inJar.getInputStream(entry);
                        byte[] bytes = FileUtil.readAsByteArray(inStream);
                        String filename = entry.getName();
                        UnwovenClassFile classFile = new UnwovenClassFile(new File(outDir, filename).getAbsolutePath(), bytes);
                        if (filename.endsWith(".class")) {
                            ReferenceType type = this.addClassFile(classFile, false);
                            StringBuilder sb = new StringBuilder();
                            sb.append(inFile.getAbsolutePath());
                            sb.append("!");
                            sb.append(entry.getName());
                            type.setBinaryPath(sb.toString());
                            addedClassFiles.add(classFile);
                        }
                        inStream.close();
                    }
                } finally {
                    inJar.close();
                }
                inJar.close();
            }
        } catch (FileNotFoundException ex) {
            IMessage message = new Message("Could not find input jar file " + inFile.getPath() + ", ignoring", new SourceLocation(inFile, 0), false);
            world.getMessageHandler().handleMessage(message);
        } catch (IOException ex) {
            IMessage message = new Message("Could not read input jar file " + inFile.getPath() + "(" + ex.getMessage() + ")", new SourceLocation(inFile, 0), true);
            world.getMessageHandler().handleMessage(message);
        } finally {
            if (inJar != null) {
                try {
                    inJar.close();
                } catch (IOException ex) {
                    IMessage message = new Message("Could not close input jar file " + inFile.getPath() + "(" + ex.getMessage() + ")", new SourceLocation(inFile, 0), true);
                    world.getMessageHandler().handleMessage(message);
                }
            }
        }
        return addedClassFiles;
    }

    public boolean needToReweaveWorld() {
        return needToReweaveWorld;
    }

    public ReferenceType addClassFile(UnwovenClassFile classFile, boolean fromInpath) {
        addedClasses.add(classFile);
        ReferenceType type = world.addSourceObjectType(classFile.getJavaClass(), false).getResolvedTypeX();
        if (fromInpath) {
            type.setBinaryPath(classFile.getFilename());
        }
        return type;
    }

    public UnwovenClassFile addClassFile(File classFile, File inPathDir, File outDir) throws IOException {
        FileInputStream fis = new FileInputStream(classFile);
        byte[] bytes = FileUtil.readAsByteArray(fis);
        String filename = classFile.getAbsolutePath().substring(inPathDir.getAbsolutePath().length() + 1);
        UnwovenClassFile ucf = new UnwovenClassFile(new File(outDir, filename).getAbsolutePath(), bytes);
        if (filename.endsWith(".class")) {
            StringBuilder sb = new StringBuilder();
            sb.append(inPathDir.getAbsolutePath());
            sb.append("!");
            sb.append(filename);
            ReferenceType type = this.addClassFile(ucf, false);
            type.setBinaryPath(sb.toString());
        }
        fis.close();
        return ucf;
    }

    public void deleteClassFile(String typename) {
        deletedTypenames.add(typename);
        world.deleteSourceObjectType(UnresolvedType.forName(typename));
    }

    public void setIsBatchWeave(boolean b) {
        isBatchWeave = b;
    }

    public void prepareForWeave() {
        if (trace.isTraceEnabled()) {
            trace.enter("prepareForWeave", this);
        }
        needToReweaveWorld = xcutSet.hasChangedSinceLastReset();
        for (UnwovenClassFile jc : addedClasses) {
            String name = jc.getClassName();
            ResolvedType type = world.resolve(name);
            if (type.isAspect()) {
                needToReweaveWorld |= xcutSet.addOrReplaceAspect(type);
            }
        }
        for (String name : deletedTypenames) {
            if (xcutSet.deleteAspect(UnresolvedType.forName(name))) {
                needToReweaveWorld = true;
            }
        }
        shadowMungerList = xcutSet.getShadowMungers();
        rewritePointcuts(shadowMungerList);
        typeMungerList = xcutSet.getTypeMungers();
        lateTypeMungerList = xcutSet.getLateTypeMungers();
        declareParentsList = xcutSet.getDeclareParents();
        addCustomMungers();
        shadowMungerList.sort(new Comparator<ShadowMunger>() {

            public int compare(ShadowMunger sm1, ShadowMunger sm2) {
                if (sm1.getSourceLocation() == null) {
                    return (sm2.getSourceLocation() == null ? 0 : 1);
                }
                if (sm2.getSourceLocation() == null) {
                    return -1;
                }
                return (sm2.getSourceLocation().getOffset() - sm1.getSourceLocation().getOffset());
            }
        });
        if (inReweavableMode) {
            world.showMessage(IMessage.INFO, WeaverMessages.format(WeaverMessages.REWEAVABLE_MODE), null, null);
        }
        if (trace.isTraceEnabled()) {
            trace.exit("prepareForWeave");
        }
    }

    private void addCustomMungers() {
        if (customMungerFactory != null) {
            for (UnwovenClassFile jc : addedClasses) {
                String name = jc.getClassName();
                ResolvedType type = world.resolve(name);
                if (type.isAspect()) {
                    Collection<ShadowMunger> shadowMungers = customMungerFactory.createCustomShadowMungers(type);
                    if (shadowMungers != null) {
                        shadowMungerList.addAll(shadowMungers);
                    }
                    Collection<ConcreteTypeMunger> typeMungers = customMungerFactory.createCustomTypeMungers(type);
                    if (typeMungers != null) {
                        typeMungerList.addAll(typeMungers);
                    }
                }
            }
        }
    }

    public void setCustomMungerFactory(CustomMungerFactory factory) {
        customMungerFactory = factory;
    }

    private void rewritePointcuts(List<ShadowMunger> shadowMungers) {
        PointcutRewriter rewriter = new PointcutRewriter();
        for (ShadowMunger munger : shadowMungers) {
            Pointcut p = munger.getPointcut();
            Pointcut newP = rewriter.rewrite(p);
            if (munger instanceof Advice) {
                Advice advice = (Advice) munger;
                if (advice.getSignature() != null) {
                    final int numFormals;
                    final String[] names;
                    if ((advice.getConcreteAspect().isAnnotationStyleAspect() && advice.getDeclaringAspect() != null && advice.getDeclaringAspect().resolve(world).isAnnotationStyleAspect()) || advice.isAnnotationStyle()) {
                        numFormals = advice.getBaseParameterCount();
                        int numArgs = advice.getSignature().getParameterTypes().length;
                        if (numFormals > 0) {
                            names = advice.getSignature().getParameterNames(world);
                            validateBindings(newP, p, numArgs, names);
                        }
                    } else {
                        numFormals = advice.getBaseParameterCount();
                        if (numFormals > 0) {
                            names = advice.getBaseParameterNames(world);
                            validateBindings(newP, p, numFormals, names);
                        }
                    }
                }
            }
            newP.m_ignoreUnboundBindingForNames = p.m_ignoreUnboundBindingForNames;
            munger.setPointcut(newP);
        }
        Map<Pointcut, Pointcut> pcMap = new HashMap<>();
        for (ShadowMunger munger : shadowMungers) {
            Pointcut p = munger.getPointcut();
            Pointcut newP = shareEntriesFromMap(p, pcMap);
            newP.m_ignoreUnboundBindingForNames = p.m_ignoreUnboundBindingForNames;
            munger.setPointcut(newP);
        }
    }

    private Pointcut shareEntriesFromMap(Pointcut p, Map<Pointcut, Pointcut> pcMap) {
        if (p instanceof NameBindingPointcut) {
            return p;
        }
        if (p instanceof IfPointcut) {
            return p;
        }
        if (p instanceof ConcreteCflowPointcut) {
            return p;
        }
        if (p instanceof AndPointcut) {
            AndPointcut apc = (AndPointcut) p;
            Pointcut left = shareEntriesFromMap(apc.getLeft(), pcMap);
            Pointcut right = shareEntriesFromMap(apc.getRight(), pcMap);
            return new AndPointcut(left, right);
        } else if (p instanceof OrPointcut) {
            OrPointcut opc = (OrPointcut) p;
            Pointcut left = shareEntriesFromMap(opc.getLeft(), pcMap);
            Pointcut right = shareEntriesFromMap(opc.getRight(), pcMap);
            return new OrPointcut(left, right);
        } else if (p instanceof NotPointcut) {
            NotPointcut npc = (NotPointcut) p;
            Pointcut not = shareEntriesFromMap(npc.getNegatedPointcut(), pcMap);
            return new NotPointcut(not);
        } else {
            if (pcMap.containsKey(p)) {
                return pcMap.get(p);
            } else {
                pcMap.put(p, p);
                return p;
            }
        }
    }

    private void validateBindings(Pointcut dnfPointcut, Pointcut userPointcut, int numFormals, String[] names) {
        if (numFormals == 0) {
            return;
        }
        if (dnfPointcut.couldMatchKinds() == Shadow.NO_SHADOW_KINDS_BITS) {
            return;
        }
        if (dnfPointcut instanceof OrPointcut) {
            OrPointcut orBasedDNFPointcut = (OrPointcut) dnfPointcut;
            Pointcut[] leftBindings = new Pointcut[numFormals];
            Pointcut[] rightBindings = new Pointcut[numFormals];
            validateOrBranch(orBasedDNFPointcut, userPointcut, numFormals, names, leftBindings, rightBindings);
        } else {
            Pointcut[] bindings = new Pointcut[numFormals];
            validateSingleBranch(dnfPointcut, userPointcut, numFormals, names, bindings);
        }
    }

    private void validateOrBranch(OrPointcut pc, Pointcut userPointcut, int numFormals, String[] names, Pointcut[] leftBindings, Pointcut[] rightBindings) {
        Pointcut left = pc.getLeft();
        Pointcut right = pc.getRight();
        if (left instanceof OrPointcut) {
            Pointcut[] newRightBindings = new Pointcut[numFormals];
            validateOrBranch((OrPointcut) left, userPointcut, numFormals, names, leftBindings, newRightBindings);
        } else {
            if (left.couldMatchKinds() != Shadow.NO_SHADOW_KINDS_BITS) {
                validateSingleBranch(left, userPointcut, numFormals, names, leftBindings);
            }
        }
        if (right instanceof OrPointcut) {
            Pointcut[] newLeftBindings = new Pointcut[numFormals];
            validateOrBranch((OrPointcut) right, userPointcut, numFormals, names, newLeftBindings, rightBindings);
        } else {
            if (right.couldMatchKinds() != Shadow.NO_SHADOW_KINDS_BITS) {
                validateSingleBranch(right, userPointcut, numFormals, names, rightBindings);
            }
        }
        int kindsInCommon = left.couldMatchKinds() & right.couldMatchKinds();
        if (kindsInCommon != Shadow.NO_SHADOW_KINDS_BITS && couldEverMatchSameJoinPoints(left, right)) {
            List<String> ambiguousNames = new ArrayList<>();
            for (int i = 0; i < numFormals; i++) {
                if (leftBindings[i] == null) {
                    if (rightBindings[i] != null) {
                        ambiguousNames.add(names[i]);
                    }
                } else if (!leftBindings[i].equals(rightBindings[i])) {
                    ambiguousNames.add(names[i]);
                }
            }
            if (!ambiguousNames.isEmpty()) {
                raiseAmbiguityInDisjunctionError(userPointcut, ambiguousNames);
            }
        }
    }

    private void validateSingleBranch(Pointcut pc, Pointcut userPointcut, int numFormals, String[] names, Pointcut[] bindings) {
        boolean[] foundFormals = new boolean[numFormals];
        for (int i = 0; i < foundFormals.length; i++) {
            foundFormals[i] = false;
        }
        validateSingleBranchRecursion(pc, userPointcut, foundFormals, names, bindings);
        for (int i = 0; i < foundFormals.length; i++) {
            if (!foundFormals[i]) {
                boolean ignore = false;
                for (int j = 0; j < userPointcut.m_ignoreUnboundBindingForNames.length; j++) {
                    if (names[i] != null && names[i].equals(userPointcut.m_ignoreUnboundBindingForNames[j])) {
                        ignore = true;
                        break;
                    }
                }
                if (!ignore) {
                    raiseUnboundFormalError(names[i], userPointcut);
                }
            }
        }
    }

    private void validateSingleBranchRecursion(Pointcut pc, Pointcut userPointcut, boolean[] foundFormals, String[] names, Pointcut[] bindings) {
        if (pc instanceof NotPointcut) {
            NotPointcut not = (NotPointcut) pc;
            if (not.getNegatedPointcut() instanceof NameBindingPointcut) {
                NameBindingPointcut nnbp = (NameBindingPointcut) not.getNegatedPointcut();
                if (!nnbp.getBindingAnnotationTypePatterns().isEmpty() && !nnbp.getBindingTypePatterns().isEmpty()) {
                    raiseNegationBindingError(userPointcut);
                }
            }
        } else if (pc instanceof AndPointcut) {
            AndPointcut and = (AndPointcut) pc;
            validateSingleBranchRecursion(and.getLeft(), userPointcut, foundFormals, names, bindings);
            validateSingleBranchRecursion(and.getRight(), userPointcut, foundFormals, names, bindings);
        } else if (pc instanceof NameBindingPointcut) {
            List<BindingTypePattern> bindingTypePatterns = ((NameBindingPointcut) pc).getBindingTypePatterns();
            for (BindingTypePattern bindingTypePattern : bindingTypePatterns) {
                int index = bindingTypePattern.getFormalIndex();
                bindings[index] = pc;
                if (foundFormals[index]) {
                    raiseAmbiguousBindingError(names[index], userPointcut);
                } else {
                    foundFormals[index] = true;
                }
            }
            List<BindingPattern> bindingAnnotationTypePatterns = ((NameBindingPointcut) pc).getBindingAnnotationTypePatterns();
            for (BindingPattern bindingAnnotationTypePattern : bindingAnnotationTypePatterns) {
                int index = bindingAnnotationTypePattern.getFormalIndex();
                bindings[index] = pc;
                if (foundFormals[index]) {
                    raiseAmbiguousBindingError(names[index], userPointcut);
                } else {
                    foundFormals[index] = true;
                }
            }
        } else if (pc instanceof ConcreteCflowPointcut) {
            ConcreteCflowPointcut cfp = (ConcreteCflowPointcut) pc;
            int[] slots = cfp.getUsedFormalSlots();
            for (int slot : slots) {
                bindings[slot] = cfp;
                if (foundFormals[slot]) {
                    raiseAmbiguousBindingError(names[slot], userPointcut);
                } else {
                    foundFormals[slot] = true;
                }
            }
        }
    }

    private boolean couldEverMatchSameJoinPoints(Pointcut left, Pointcut right) {
        if (left instanceof OrPointcut) {
            OrPointcut leftOrPointcut = (OrPointcut) left;
            if (couldEverMatchSameJoinPoints(leftOrPointcut.getLeft(), right)) {
                return true;
            }
            if (couldEverMatchSameJoinPoints(leftOrPointcut.getRight(), right)) {
                return true;
            }
            return false;
        }
        if (right instanceof OrPointcut) {
            OrPointcut rightOrPointcut = (OrPointcut) right;
            if (couldEverMatchSameJoinPoints(left, rightOrPointcut.getLeft())) {
                return true;
            }
            if (couldEverMatchSameJoinPoints(left, rightOrPointcut.getRight())) {
                return true;
            }
            return false;
        }
        WithinPointcut leftWithin = (WithinPointcut) findFirstPointcutIn(left, WithinPointcut.class);
        WithinPointcut rightWithin = (WithinPointcut) findFirstPointcutIn(right, WithinPointcut.class);
        if ((leftWithin != null) && (rightWithin != null)) {
            if (!leftWithin.couldEverMatchSameJoinPointsAs(rightWithin)) {
                return false;
            }
        }
        KindedPointcut leftKind = (KindedPointcut) findFirstPointcutIn(left, KindedPointcut.class);
        KindedPointcut rightKind = (KindedPointcut) findFirstPointcutIn(right, KindedPointcut.class);
        if ((leftKind != null) && (rightKind != null)) {
            if (!leftKind.couldEverMatchSameJoinPointsAs(rightKind)) {
                return false;
            }
        }
        return true;
    }

    private Pointcut findFirstPointcutIn(Pointcut toSearch, Class<?> toLookFor) {
        if (toSearch instanceof NotPointcut) {
            return null;
        }
        if (toLookFor.isInstance(toSearch)) {
            return toSearch;
        }
        if (toSearch instanceof AndPointcut) {
            AndPointcut apc = (AndPointcut) toSearch;
            Pointcut left = findFirstPointcutIn(apc.getLeft(), toLookFor);
            if (left != null) {
                return left;
            }
            return findFirstPointcutIn(apc.getRight(), toLookFor);
        }
        return null;
    }

    private void raiseNegationBindingError(Pointcut userPointcut) {
        world.showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.NEGATION_DOESNT_ALLOW_BINDING), userPointcut.getSourceContext().makeSourceLocation(userPointcut), null);
    }

    private void raiseAmbiguousBindingError(String name, Pointcut pointcut) {
        world.showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.AMBIGUOUS_BINDING, name), pointcut.getSourceContext().makeSourceLocation(pointcut), null);
    }

    private void raiseAmbiguityInDisjunctionError(Pointcut userPointcut, List<String> names) {
        StringBuffer formalNames = new StringBuffer(names.get(0).toString());
        for (int i = 1; i < names.size(); i++) {
            formalNames.append(", ");
            formalNames.append(names.get(i));
        }
        world.showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.AMBIGUOUS_BINDING_IN_OR, formalNames), userPointcut.getSourceContext().makeSourceLocation(userPointcut), null);
    }

    private void raiseUnboundFormalError(String name, Pointcut userPointcut) {
        world.showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.UNBOUND_FORMAL, name), userPointcut.getSourceLocation(), null);
    }

    public void addManifest(Manifest newManifest) {
        if (manifest == null) {
            manifest = newManifest;
        }
    }

    public Manifest getManifest(boolean shouldCreate) {
        if (manifest == null && shouldCreate) {
            String WEAVER_MANIFEST_VERSION = "1.0";
            Attributes.Name CREATED_BY = new Name("Created-By");
            String WEAVER_CREATED_BY = "AspectJ Compiler";
            manifest = new Manifest();
            Attributes attributes = manifest.getMainAttributes();
            attributes.put(Name.MANIFEST_VERSION, WEAVER_MANIFEST_VERSION);
            attributes.put(CREATED_BY, WEAVER_CREATED_BY);
        }
        return manifest;
    }

    public Collection<String> weave(File file) throws IOException {
        OutputStream os = FileUtil.makeOutputStream(file);
        this.zipOutputStream = new ZipOutputStream(os);
        prepareForWeave();
        Collection<String> c = weave(new IClassFileProvider() {

            public boolean isApplyAtAspectJMungersOnly() {
                return false;
            }

            public Iterator<UnwovenClassFile> getClassFileIterator() {
                return addedClasses.iterator();
            }

            public IWeaveRequestor getRequestor() {
                return new IWeaveRequestor() {

                    public void acceptResult(IUnwovenClassFile result) {
                        try {
                            writeZipEntry(result.getFilename(), result.getBytes());
                        } catch (IOException ex) {
                        }
                    }

                    public void processingReweavableState() {
                    }

                    public void addingTypeMungers() {
                    }

                    public void weavingAspects() {
                    }

                    public void weavingClasses() {
                    }

                    public void weaveCompleted() {
                    }
                };
            }
        });
        zipOutputStream.close();
        return c;
    }

    private Set<IProgramElement> candidatesForRemoval = null;

    public Collection<String> weave(IClassFileProvider input) throws IOException {
        if (trace.isTraceEnabled()) {
            trace.enter("weave", this, input);
        }
        ContextToken weaveToken = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.WEAVING, "");
        Collection<String> wovenClassNames = new ArrayList<>();
        IWeaveRequestor requestor = input.getRequestor();
        if (world.getModel() != null && world.isMinimalModel()) {
            candidatesForRemoval = new HashSet<>();
        }
        if (world.getModel() != null && !isBatchWeave) {
            AsmManager manager = world.getModelAsAsmManager();
            for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
                UnwovenClassFile classFile = i.next();
                manager.removeRelationshipsTargettingThisType(classFile.getClassName());
            }
        }
        for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
            UnwovenClassFile classFile = i.next();
            if (classFile.shouldBeWoven()) {
                String className = classFile.getClassName();
                ResolvedType theType = world.resolve(className);
                if (theType != null) {
                    theType.ensureConsistent();
                }
            }
        }
        if (input.isApplyAtAspectJMungersOnly()) {
            ContextToken atAspectJMungersOnly = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_ATASPECTJTYPE_MUNGERS_ONLY, "");
            requestor.weavingAspects();
            CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.WEAVING_ASPECTS, "");
            for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
                UnwovenClassFile classFile = i.next();
                if (classFile.shouldBeWoven()) {
                    String className = classFile.getClassName();
                    ResolvedType theType = world.resolve(className);
                    if (theType.isAnnotationStyleAspect()) {
                        BcelObjectType classType = BcelWorld.getBcelObjectType(theType);
                        if (classType == null) {
                            throw new BCException("Can't find bcel delegate for " + className + " type=" + theType.getClass());
                        }
                        LazyClassGen clazz = classType.getLazyClassGen();
                        BcelPerClauseAspectAdder selfMunger = new BcelPerClauseAspectAdder(theType, theType.getPerClause().getKind());
                        selfMunger.forceMunge(clazz, true);
                        classType.finishedWith();
                        UnwovenClassFile[] newClasses = getClassFilesFor(clazz);
                        for (UnwovenClassFile newClass : newClasses) {
                            requestor.acceptResult(newClass);
                        }
                        wovenClassNames.add(classFile.getClassName());
                    }
                }
            }
            requestor.weaveCompleted();
            CompilationAndWeavingContext.leavingPhase(atAspectJMungersOnly);
            return wovenClassNames;
        }
        requestor.processingReweavableState();
        ContextToken reweaveToken = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_REWEAVABLE_STATE, "");
        prepareToProcessReweavableState();
        for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
            UnwovenClassFile classFile = i.next();
            if (classFile.shouldBeWoven()) {
                String className = classFile.getClassName();
                BcelObjectType classType = getClassType(className);
                if (classType != null) {
                    ContextToken tok = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_REWEAVABLE_STATE, className);
                    processReweavableStateIfPresent(className, classType);
                    CompilationAndWeavingContext.leavingPhase(tok);
                }
            }
        }
        CompilationAndWeavingContext.leavingPhase(reweaveToken);
        ContextToken typeMungingToken = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_TYPE_MUNGERS, "");
        requestor.addingTypeMungers();
        List<String> typesToProcess = new ArrayList<>();
        for (Iterator<UnwovenClassFile> iter = input.getClassFileIterator(); iter.hasNext(); ) {
            UnwovenClassFile clf = iter.next();
            if (clf.shouldBeWoven()) {
                typesToProcess.add(clf.getClassName());
            }
        }
        while (typesToProcess.size() > 0) {
            weaveParentsFor(typesToProcess, typesToProcess.get(0), null);
        }
        for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
            UnwovenClassFile classFile = i.next();
            if (classFile.shouldBeWoven()) {
                String className = classFile.getClassName();
                addNormalTypeMungers(className);
            }
        }
        CompilationAndWeavingContext.leavingPhase(typeMungingToken);
        requestor.weavingAspects();
        ContextToken aspectToken = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.WEAVING_ASPECTS, "");
        for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
            UnwovenClassFile classFile = i.next();
            if (classFile.shouldBeWoven()) {
                String className = classFile.getClassName();
                ResolvedType theType = world.resolve(className);
                if (theType.isAspect()) {
                    BcelObjectType classType = BcelWorld.getBcelObjectType(theType);
                    if (classType == null) {
                        ReferenceTypeDelegate theDelegate = ((ReferenceType) theType).getDelegate();
                        if (theDelegate.getClass().getName().endsWith("EclipseSourceType")) {
                            continue;
                        }
                        throw new BCException("Can't find bcel delegate for " + className + " type=" + theType.getClass());
                    }
                    weaveAndNotify(classFile, classType, requestor);
                    wovenClassNames.add(className);
                }
            }
        }
        CompilationAndWeavingContext.leavingPhase(aspectToken);
        requestor.weavingClasses();
        ContextToken classToken = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.WEAVING_CLASSES, "");
        for (Iterator<UnwovenClassFile> i = input.getClassFileIterator(); i.hasNext(); ) {
            UnwovenClassFile classFile = i.next();
            if (classFile.shouldBeWoven()) {
                String className = classFile.getClassName();
                ResolvedType theType = world.resolve(className);
                if (!theType.isAspect()) {
                    BcelObjectType classType = BcelWorld.getBcelObjectType(theType);
                    if (classType == null) {
                        ReferenceTypeDelegate theDelegate = ((ReferenceType) theType).getDelegate();
                        if (theDelegate.getClass().getName().endsWith("EclipseSourceType")) {
                            continue;
                        }
                        throw new BCException("Can't find bcel delegate for " + className + " type=" + theType.getClass());
                    }
                    weaveAndNotify(classFile, classType, requestor);
                    wovenClassNames.add(className);
                }
            }
        }
        CompilationAndWeavingContext.leavingPhase(classToken);
        addedClasses.clear();
        deletedTypenames.clear();
        requestor.weaveCompleted();
        CompilationAndWeavingContext.leavingPhase(weaveToken);
        if (trace.isTraceEnabled()) {
            trace.exit("weave", wovenClassNames);
        }
        if (world.getModel() != null && world.isMinimalModel()) {
            candidatesForRemoval.clear();
        }
        return wovenClassNames;
    }

    public void allWeavingComplete() {
        warnOnUnmatchedAdvice();
    }

    private void warnOnUnmatchedAdvice() {
        class AdviceLocation {

            private final int lineNo;

            private final UnresolvedType inAspect;

            public AdviceLocation(BcelAdvice advice) {
                this.lineNo = advice.getSourceLocation().getLine();
                this.inAspect = advice.getDeclaringAspect();
            }

            @Override
            public boolean equals(Object obj) {
                if (!(obj instanceof AdviceLocation)) {
                    return false;
                }
                AdviceLocation other = (AdviceLocation) obj;
                if (this.lineNo != other.lineNo) {
                    return false;
                }
                if (!this.inAspect.equals(other.inAspect)) {
                    return false;
                }
                return true;
            }

            @Override
            public int hashCode() {
                return 37 + 17 * lineNo + 17 * inAspect.hashCode();
            }
        }
        if (world.isInJava5Mode() && world.getLint().adviceDidNotMatch.isEnabled()) {
            List<ShadowMunger> l = world.getCrosscuttingMembersSet().getShadowMungers();
            Set<AdviceLocation> alreadyWarnedLocations = new HashSet<>();
            for (ShadowMunger element : l) {
                if (element instanceof BcelAdvice) {
                    BcelAdvice ba = (BcelAdvice) element;
                    if (ba.getKind() == AdviceKind.CflowEntry || ba.getKind() == AdviceKind.CflowBelowEntry) {
                        continue;
                    }
                    if (!ba.hasMatchedSomething()) {
                        if (ba.getSignature() != null) {
                            AdviceLocation loc = new AdviceLocation(ba);
                            if (alreadyWarnedLocations.contains(loc)) {
                                continue;
                            } else {
                                alreadyWarnedLocations.add(loc);
                            }
                            if (!(ba.getSignature() instanceof BcelMethod) || !Utility.isSuppressing(ba.getSignature(), "adviceDidNotMatch")) {
                                world.getLint().adviceDidNotMatch.signal(ba.getDeclaringAspect().toString(), new SourceLocation(element.getSourceLocation().getSourceFile(), element.getSourceLocation().getLine()));
                            }
                        }
                    }
                }
            }
        }
    }

    private void weaveParentsFor(List<String> typesForWeaving, String typeToWeave, ResolvedType resolvedTypeToWeave) {
        if (resolvedTypeToWeave == null) {
            resolvedTypeToWeave = world.resolve(typeToWeave);
        }
        ResolvedType superclassType = resolvedTypeToWeave.getSuperclass();
        String superclassTypename = (superclassType == null ? null : superclassType.getName());
        if (superclassType != null && !superclassType.isTypeHierarchyComplete() && superclassType.isExposedToWeaver() && typesForWeaving.contains(superclassTypename)) {
            weaveParentsFor(typesForWeaving, superclassTypename, superclassType);
        }
        ResolvedType[] interfaceTypes = resolvedTypeToWeave.getDeclaredInterfaces();
        for (ResolvedType resolvedSuperInterface : interfaceTypes) {
            if (!resolvedSuperInterface.isTypeHierarchyComplete()) {
                String interfaceTypename = resolvedSuperInterface.getName();
                if (resolvedSuperInterface.isExposedToWeaver()) {
                    weaveParentsFor(typesForWeaving, interfaceTypename, resolvedSuperInterface);
                }
            }
        }
        ContextToken tok = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_DECLARE_PARENTS, resolvedTypeToWeave.getName());
        if (!resolvedTypeToWeave.isTypeHierarchyComplete()) {
            weaveParentTypeMungers(resolvedTypeToWeave);
        }
        CompilationAndWeavingContext.leavingPhase(tok);
        typesForWeaving.remove(typeToWeave);
        resolvedTypeToWeave.tagAsTypeHierarchyComplete();
    }

    public void prepareToProcessReweavableState() {
    }

    public void processReweavableStateIfPresent(String className, BcelObjectType classType) {
        WeaverStateInfo wsi = classType.getWeaverState();
        if (wsi != null && wsi.isReweavable()) {
            world.showMessage(IMessage.INFO, WeaverMessages.format(WeaverMessages.PROCESSING_REWEAVABLE, className, classType.getSourceLocation().getSourceFile()), null, null);
            Set<String> aspectsPreviouslyInWorld = wsi.getAspectsAffectingType();
            Set<String> alreadyConfirmedReweavableState = new HashSet<>();
            for (String requiredTypeSignature : aspectsPreviouslyInWorld) {
                if (!alreadyConfirmedReweavableState.contains(requiredTypeSignature)) {
                    ResolvedType rtx = world.resolve(UnresolvedType.forSignature(requiredTypeSignature), true);
                    boolean exists = !rtx.isMissing();
                    if (!world.isOverWeaving()) {
                        if (!exists) {
                            world.getLint().missingAspectForReweaving.signal(new String[]{rtx.getName(), className}, classType.getSourceLocation(), null);
                        } else {
                            if (!xcutSet.containsAspect(rtx)) {
                                world.showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.REWEAVABLE_ASPECT_NOT_REGISTERED, rtx.getName(), className), null, null);
                            } else if (!world.getMessageHandler().isIgnoring(IMessage.INFO)) {
                                world.showMessage(IMessage.INFO, WeaverMessages.format(WeaverMessages.VERIFIED_REWEAVABLE_TYPE, rtx.getName(), rtx.getSourceLocation().getSourceFile()), null, null);
                            }
                            alreadyConfirmedReweavableState.add(requiredTypeSignature);
                        }
                    }
                }
            }
            if (!world.isOverWeaving()) {
                byte[] ucfd = wsi.getUnwovenClassFileData();
                if (ucfd.length == 0) {
                    world.getMessageHandler().handleMessage(MessageUtil.error(WeaverMessages.format(WeaverMessages.MUST_KEEP_OVERWEAVING_ONCE_START, className)));
                } else {
                    byte[] bytes = wsi.getUnwovenClassFileData(classType.getJavaClass().getBytes());
                    JavaClass newJavaClass = Utility.makeJavaClass(classType.getJavaClass().getFileName(), bytes);
                    classType.setJavaClass(newJavaClass, true);
                    classType.getResolvedTypeX().ensureConsistent();
                }
            }
        }
    }

    private void weaveAndNotify(UnwovenClassFile classFile, BcelObjectType classType, IWeaveRequestor requestor) throws IOException {
        trace.enter("weaveAndNotify", this, new Object[]{classFile, classType, requestor});
        ContextToken tok = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.WEAVING_TYPE, classType.getResolvedTypeX().getName());
        LazyClassGen clazz = weaveWithoutDump(classFile, classType);
        classType.finishedWith();
        if (clazz != null) {
            UnwovenClassFile[] newClasses = getClassFilesFor(clazz);
            if (newClasses[0].getClassName().equals(classFile.getClassName())) {
                newClasses[0].setClassNameAsChars(classFile.getClassNameAsChars());
            }
            for (UnwovenClassFile newClass : newClasses) {
                requestor.acceptResult(newClass);
            }
        } else {
            requestor.acceptResult(classFile);
        }
        classType.weavingCompleted();
        CompilationAndWeavingContext.leavingPhase(tok);
        trace.exit("weaveAndNotify");
    }

    public BcelObjectType getClassType(String forClass) {
        return BcelWorld.getBcelObjectType(world.resolve(forClass));
    }

    public void addParentTypeMungers(String typeName) {
        weaveParentTypeMungers(world.resolve(typeName));
    }

    public void addNormalTypeMungers(String typeName) {
        weaveNormalTypeMungers(world.resolve(typeName));
    }

    public UnwovenClassFile[] getClassFilesFor(LazyClassGen clazz) {
        List<UnwovenClassFile.ChildClass> childClasses = clazz.getChildClasses(world);
        UnwovenClassFile[] ret = new UnwovenClassFile[1 + childClasses.size()];
        ret[0] = new UnwovenClassFile(clazz.getFileName(), clazz.getClassName(), clazz.getJavaClassBytesIncludingReweavable(world));
        int index = 1;
        for (UnwovenClassFile.ChildClass element : childClasses) {
            UnwovenClassFile childClass = new UnwovenClassFile(clazz.getFileName() + "$" + element.name, element.bytes);
            ret[index++] = childClass;
        }
        return ret;
    }

    public void weaveParentTypeMungers(ResolvedType onType) {
        if (onType.isRawType() || onType.isParameterizedType()) {
            onType = onType.getGenericType();
        }
        onType.clearInterTypeMungers();
        List<DeclareParents> decpToRepeat = new ArrayList<>();
        boolean aParentChangeOccurred = false;
        boolean anAnnotationChangeOccurred = false;
        for (DeclareParents decp : declareParentsList) {
            boolean typeChanged = applyDeclareParents(decp, onType);
            if (typeChanged) {
                aParentChangeOccurred = true;
            } else {
                decpToRepeat.add(decp);
            }
        }
        for (DeclareAnnotation decA : xcutSet.getDeclareAnnotationOnTypes()) {
            boolean typeChanged = applyDeclareAtType(decA, onType, true);
            if (typeChanged) {
                anAnnotationChangeOccurred = true;
            }
        }
        while ((aParentChangeOccurred || anAnnotationChangeOccurred) && !decpToRepeat.isEmpty()) {
            anAnnotationChangeOccurred = aParentChangeOccurred = false;
            List<DeclareParents> decpToRepeatNextTime = new ArrayList<>();
            for (DeclareParents decp : decpToRepeat) {
                boolean typeChanged = applyDeclareParents(decp, onType);
                if (typeChanged) {
                    aParentChangeOccurred = true;
                } else {
                    decpToRepeatNextTime.add(decp);
                }
            }
            for (DeclareAnnotation decA : xcutSet.getDeclareAnnotationOnTypes()) {
                boolean typeChanged = applyDeclareAtType(decA, onType, false);
                if (typeChanged) {
                    anAnnotationChangeOccurred = true;
                }
            }
            decpToRepeat = decpToRepeatNextTime;
        }
    }

    private boolean applyDeclareAtType(DeclareAnnotation decA, ResolvedType onType, boolean reportProblems) {
        boolean didSomething = false;
        if (decA.matches(onType)) {
            AnnotationAJ theAnnotation = decA.getAnnotation();
            if (theAnnotation == null) {
                return false;
            }
            if (onType.hasAnnotation(theAnnotation.getType())) {
                return false;
            }
            AnnotationAJ annoX = decA.getAnnotation();
            boolean problemReported = verifyTargetIsOK(decA, onType, annoX, reportProblems);
            if (!problemReported) {
                AsmRelationshipProvider.addDeclareAnnotationRelationship(world.getModelAsAsmManager(), decA.getSourceLocation(), onType.getSourceLocation(), false);
                if (!getWorld().getMessageHandler().isIgnoring(IMessage.WEAVEINFO)) {
                    getWorld().getMessageHandler().handleMessage(WeaveMessage.constructWeavingMessage(WeaveMessage.WEAVEMESSAGE_ANNOTATES, new String[]{onType.toString(), Utility.beautifyLocation(onType.getSourceLocation()), decA.getAnnotationString(), "type", decA.getAspect().toString(), Utility.beautifyLocation(decA.getSourceLocation())}));
                }
                didSomething = true;
                ResolvedTypeMunger newAnnotationTM = new AnnotationOnTypeMunger(annoX);
                newAnnotationTM.setSourceLocation(decA.getSourceLocation());
                onType.addInterTypeMunger(new BcelTypeMunger(newAnnotationTM, decA.getAspect().resolve(world)), false);
                decA.copyAnnotationTo(onType);
            }
        }
        return didSomething;
    }

    private boolean verifyTargetIsOK(DeclareAnnotation decA, ResolvedType onType, AnnotationAJ annoX, boolean outputProblems) {
        boolean problemReported = false;
        if (annoX.specifiesTarget()) {
            if ((onType.isAnnotation() && !annoX.allowedOnAnnotationType()) || (!annoX.allowedOnRegularType())) {
                if (outputProblems) {
                    if (decA.isExactPattern()) {
                        world.getMessageHandler().handleMessage(MessageUtil.error(WeaverMessages.format(WeaverMessages.INCORRECT_TARGET_FOR_DECLARE_ANNOTATION, onType.getName(), annoX.getTypeName(), annoX.getValidTargets()), decA.getSourceLocation()));
                    } else {
                        if (world.getLint().invalidTargetForAnnotation.isEnabled()) {
                            world.getLint().invalidTargetForAnnotation.signal(new String[]{onType.getName(), annoX.getTypeName(), annoX.getValidTargets()}, decA.getSourceLocation(), new ISourceLocation[]{onType.getSourceLocation()});
                        }
                    }
                }
                problemReported = true;
            }
        }
        return problemReported;
    }

    private boolean applyDeclareParents(DeclareParents p, ResolvedType onType) {
        boolean didSomething = false;
        List<ResolvedType> newParents = p.findMatchingNewParents(onType, true);
        if (!newParents.isEmpty()) {
            didSomething = true;
            BcelWorld.getBcelObjectType(onType);
            for (ResolvedType newParent : newParents) {
                onType.addParent(newParent);
                NewParentTypeMunger newParentMunger = new NewParentTypeMunger(newParent, p.getDeclaringType());
                if (p.isMixin()) {
                    newParentMunger.setIsMixin(true);
                }
                newParentMunger.setSourceLocation(p.getSourceLocation());
                onType.addInterTypeMunger(new BcelTypeMunger(newParentMunger, xcutSet.findAspectDeclaringParents(p)), false);
            }
        }
        return didSomething;
    }

    public void weaveNormalTypeMungers(ResolvedType onType) {
        ContextToken tok = CompilationAndWeavingContext.enteringPhase(CompilationAndWeavingContext.PROCESSING_TYPE_MUNGERS, onType.getName());
        if (onType.isRawType() || onType.isParameterizedType()) {
            onType = onType.getGenericType();
        }
        for (ConcreteTypeMunger m : typeMungerList) {
            if (!m.isLateMunger() && m.matches(onType)) {
                onType.addInterTypeMunger(m, false);
            }
        }
        CompilationAndWeavingContext.leavingPhase(tok);
    }

    public LazyClassGen weaveWithoutDump(UnwovenClassFile classFile, BcelObjectType classType) throws IOException {
        return weave(classFile, classType, false);
    }

    LazyClassGen weave(UnwovenClassFile classFile, BcelObjectType classType) throws IOException {
        LazyClassGen ret = weave(classFile, classType, true);
        return ret;
    }

    private LazyClassGen weave(UnwovenClassFile classFile, BcelObjectType classType, boolean dump) throws IOException {
        try {
            if (classType.isSynthetic()) {
                if (dump) {
                    dumpUnchanged(classFile);
                }
                return null;
            }
            ReferenceType resolvedClassType = classType.getResolvedTypeX();
            if (world.isXmlConfigured() && world.getXmlConfiguration().excludesType(resolvedClassType)) {
                if (!world.getMessageHandler().isIgnoring(IMessage.INFO)) {
                    world.getMessageHandler().handleMessage(MessageUtil.info("Type '" + resolvedClassType.getName() + "' not woven due to exclusion via XML weaver exclude section"));
                }
                if (dump) {
                    dumpUnchanged(classFile);
                }
                return null;
            }
            List<ShadowMunger> shadowMungers = fastMatch(shadowMungerList, resolvedClassType);
            List<ConcreteTypeMunger> typeMungers = classType.getResolvedTypeX().getInterTypeMungers();
            resolvedClassType.checkInterTypeMungers();
            boolean mightNeedToWeave = shadowMungers.size() > 0 || typeMungers.size() > 0 || classType.isAspect() || world.getDeclareAnnotationOnMethods().size() > 0 || world.getDeclareAnnotationOnFields().size() > 0;
            boolean mightNeedBridgeMethods = world.isInJava5Mode() && !classType.isInterface() && resolvedClassType.getInterTypeMungersIncludingSupers().size() > 0;
            LazyClassGen clazz = null;
            if (mightNeedToWeave || mightNeedBridgeMethods) {
                clazz = classType.getLazyClassGen();
                try {
                    boolean isChanged = false;
                    if (mightNeedToWeave) {
                        isChanged = BcelClassWeaver.weave(world, clazz, shadowMungers, typeMungers, lateTypeMungerList, inReweavableMode);
                    }
                    checkDeclareTypeErrorOrWarning(world, classType);
                    if (mightNeedBridgeMethods) {
                        isChanged = BcelClassWeaver.calculateAnyRequiredBridgeMethods(world, clazz) || isChanged;
                    }
                    if (isChanged) {
                        if (dump) {
                            dump(classFile, clazz);
                        }
                        return clazz;
                    }
                } catch (RuntimeException re) {
                    String classDebugInfo = null;
                    try {
                        classDebugInfo = clazz.toLongString();
                    } catch (Throwable e) {
                        new RuntimeException("Crashed whilst crashing with this exception: " + e, e).printStackTrace();
                        classDebugInfo = clazz.getClassName();
                    }
                    String messageText = "trouble in: \n" + classDebugInfo;
                    getWorld().getMessageHandler().handleMessage(new Message(messageText, IMessage.ABORT, re, null));
                } catch (Error re) {
                    String classDebugInfo = null;
                    try {
                        classDebugInfo = clazz.toLongString();
                    } catch (OutOfMemoryError oome) {
                        System.err.println("Ran out of memory creating debug info for an error");
                        re.printStackTrace(System.err);
                        classDebugInfo = clazz.getClassName();
                    } catch (Throwable e) {
                        classDebugInfo = clazz.getClassName();
                    }
                    String messageText = "trouble in: \n" + classDebugInfo;
                    getWorld().getMessageHandler().handleMessage(new Message(messageText, IMessage.ABORT, re, null));
                }
            } else {
                checkDeclareTypeErrorOrWarning(world, classType);
            }
            AsmManager model = world.getModelAsAsmManager();
            if (world.isMinimalModel() && model != null && !classType.isAspect()) {
                AspectJElementHierarchy hierarchy = (AspectJElementHierarchy) model.getHierarchy();
                String pkgname = classType.getResolvedTypeX().getPackageName();
                String tname = classType.getResolvedTypeX().getSimpleBaseName();
                IProgramElement typeElement = hierarchy.findElementForType(pkgname, tname);
                if (typeElement != null && hasInnerType(typeElement)) {
                    candidatesForRemoval.add(typeElement);
                }
                if (typeElement != null && !hasInnerType(typeElement)) {
                    IProgramElement parent = typeElement.getParent();
                    if (parent != null) {
                        parent.removeChild(typeElement);
                        if (parent.getKind().isSourceFile()) {
                            removeSourceFileIfNoMoreTypeDeclarationsInside(hierarchy, typeElement, parent);
                        } else {
                            hierarchy.forget(null, typeElement);
                            walkUpRemovingEmptyTypesAndPossiblyEmptySourceFile(hierarchy, tname, parent);
                        }
                    }
                }
            }
            if (dump) {
                dumpUnchanged(classFile);
                return clazz;
            } else {
                if (clazz != null && !clazz.getChildClasses(world).isEmpty()) {
                    return clazz;
                }
                return null;
            }
        } finally {
            world.demote();
        }
    }

    private void walkUpRemovingEmptyTypesAndPossiblyEmptySourceFile(AspectJElementHierarchy hierarchy, String tname, IProgramElement typeThatHasChildRemoved) {
        while (typeThatHasChildRemoved != null && !(typeThatHasChildRemoved.getKind().isType() || typeThatHasChildRemoved.getKind().isSourceFile())) {
            typeThatHasChildRemoved = typeThatHasChildRemoved.getParent();
        }
        if (candidatesForRemoval.contains(typeThatHasChildRemoved) && !hasInnerType(typeThatHasChildRemoved)) {
            IProgramElement parent = typeThatHasChildRemoved.getParent();
            if (parent != null) {
                parent.removeChild(typeThatHasChildRemoved);
                candidatesForRemoval.remove(typeThatHasChildRemoved);
                if (parent.getKind().isSourceFile()) {
                    removeSourceFileIfNoMoreTypeDeclarationsInside(hierarchy, typeThatHasChildRemoved, parent);
                } else {
                    walkUpRemovingEmptyTypesAndPossiblyEmptySourceFile(hierarchy, tname, parent);
                }
            }
        }
    }

    private void removeSourceFileIfNoMoreTypeDeclarationsInside(AspectJElementHierarchy hierarchy, IProgramElement typeElement, IProgramElement sourceFileNode) {
        IProgramElement compilationUnit = sourceFileNode;
        boolean anyOtherTypeDeclarations = false;
        for (IProgramElement child : compilationUnit.getChildren()) {
            IProgramElement.Kind k = child.getKind();
            if (k.isType()) {
                anyOtherTypeDeclarations = true;
                break;
            }
        }
        if (!anyOtherTypeDeclarations) {
            IProgramElement cuParent = compilationUnit.getParent();
            if (cuParent != null) {
                compilationUnit.setParent(null);
                cuParent.removeChild(compilationUnit);
            }
            hierarchy.forget(sourceFileNode, typeElement);
        } else {
            hierarchy.forget(null, typeElement);
        }
    }

    private boolean hasInnerType(IProgramElement typeNode) {
        for (IProgramElement child : typeNode.getChildren()) {
            IProgramElement.Kind kind = child.getKind();
            if (kind.isType()) {
                return true;
            }
            if (kind.isType() || kind == IProgramElement.Kind.METHOD || kind == IProgramElement.Kind.CONSTRUCTOR) {
                boolean b = hasInnerType(child);
                if (b) {
                    return b;
                }
            }
        }
        return false;
    }

    private void checkDeclareTypeErrorOrWarning(BcelWorld world2, BcelObjectType classType) {
        List<DeclareTypeErrorOrWarning> dteows = world.getDeclareTypeEows();
        for (DeclareTypeErrorOrWarning dteow : dteows) {
            if (dteow.getTypePattern().matchesStatically(classType.getResolvedTypeX())) {
                if (dteow.isError()) {
                    world.getMessageHandler().handleMessage(MessageUtil.error(dteow.getMessage(), classType.getResolvedTypeX().getSourceLocation()));
                } else {
                    world.getMessageHandler().handleMessage(MessageUtil.warn(dteow.getMessage(), classType.getResolvedTypeX().getSourceLocation()));
                }
            }
        }
    }

    private void dumpUnchanged(UnwovenClassFile classFile) throws IOException {
        if (zipOutputStream != null) {
            writeZipEntry(getEntryName(classFile.getJavaClass().getClassName()), classFile.getBytes());
        } else {
            classFile.writeUnchangedBytes();
        }
    }

    private String getEntryName(String className) {
        return className.replace('.', '/') + ".class";
    }

    private void dump(UnwovenClassFile classFile, LazyClassGen clazz) throws IOException {
        if (zipOutputStream != null) {
            String mainClassName = classFile.getJavaClass().getClassName();
            writeZipEntry(getEntryName(mainClassName), clazz.getJavaClass(world).getBytes());
            List<UnwovenClassFile.ChildClass> childClasses = clazz.getChildClasses(world);
            if (!childClasses.isEmpty()) {
                for (UnwovenClassFile.ChildClass c : childClasses) {
                    writeZipEntry(getEntryName(mainClassName + "$" + c.name), c.bytes);
                }
            }
        } else {
            classFile.writeWovenBytes(clazz.getJavaClass(world).getBytes(), clazz.getChildClasses(world));
        }
    }

    private void writeZipEntry(String name, byte[] bytes) throws IOException {
        ZipEntry newEntry = new ZipEntry(name);
        zipOutputStream.putNextEntry(newEntry);
        zipOutputStream.write(bytes);
        zipOutputStream.closeEntry();
    }

    private List<ShadowMunger> fastMatch(List<ShadowMunger> list, ResolvedType type) {
        if (list == null) {
            return Collections.emptyList();
        }
        boolean isOverweaving = world.isOverWeaving();
        WeaverStateInfo typeWeaverState = (isOverweaving ? type.getWeaverState() : null);
        FastMatchInfo info = new FastMatchInfo(type, null, world);
        List<ShadowMunger> result = new ArrayList<>();
        if (world.areInfoMessagesEnabled() && world.isTimingEnabled()) {
            for (ShadowMunger munger : list) {
                if (typeWeaverState != null) {
                    ResolvedType declaringAspect = munger.getDeclaringType();
                    if (typeWeaverState.isAspectAlreadyApplied(declaringAspect)) {
                        continue;
                    }
                }
                Pointcut pointcut = munger.getPointcut();
                long starttime = System.nanoTime();
                FuzzyBoolean fb = pointcut.fastMatch(info);
                long endtime = System.nanoTime();
                world.recordFastMatch(pointcut, endtime - starttime);
                if (fb.maybeTrue()) {
                    result.add(munger);
                }
            }
        } else {
            for (ShadowMunger munger : list) {
                if (typeWeaverState != null) {
                    ResolvedType declaringAspect = munger.getConcreteAspect();
                    if (typeWeaverState.isAspectAlreadyApplied(declaringAspect)) {
                        continue;
                    }
                }
                Pointcut pointcut = munger.getPointcut();
                FuzzyBoolean fb = pointcut.fastMatch(info);
                if (fb.maybeTrue()) {
                    result.add(munger);
                }
            }
        }
        return result;
    }

    public void setReweavableMode(boolean xNotReweavable) {
        inReweavableMode = !xNotReweavable;
        WeaverStateInfo.setReweavableModeDefaults(!xNotReweavable, false, true);
    }

    public boolean isReweavable() {
        return inReweavableMode;
    }

    public World getWorld() {
        return world;
    }

    public void tidyUp() {
        if (trace.isTraceEnabled()) {
            trace.enter("tidyUp", this);
        }
        shadowMungerList = null;
        typeMungerList = null;
        lateTypeMungerList = null;
        declareParentsList = null;
        if (trace.isTraceEnabled()) {
            trace.exit("tidyUp");
        }
    }

    public void write(CompressingDataOutputStream dos) throws IOException {
        xcutSet.write(dos);
    }

    public void setShadowMungers(List<ShadowMunger> shadowMungers) {
        shadowMungerList = shadowMungers;
    }
}
