package org.aspectj.weaver.bcel;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.Attribute;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.Method;
import org.aspectj.apache.bcel.classfile.Synthetic;
import org.aspectj.apache.bcel.classfile.annotation.AnnotationGen;
import org.aspectj.apache.bcel.generic.BranchHandle;
import org.aspectj.apache.bcel.generic.ClassGenException;
import org.aspectj.apache.bcel.generic.CodeExceptionGen;
import org.aspectj.apache.bcel.generic.Instruction;
import org.aspectj.apache.bcel.generic.InstructionBranch;
import org.aspectj.apache.bcel.generic.InstructionHandle;
import org.aspectj.apache.bcel.generic.InstructionList;
import org.aspectj.apache.bcel.generic.InstructionSelect;
import org.aspectj.apache.bcel.generic.InstructionTargeter;
import org.aspectj.apache.bcel.generic.LineNumberTag;
import org.aspectj.apache.bcel.generic.LocalVariableTag;
import org.aspectj.apache.bcel.generic.MethodGen;
import org.aspectj.apache.bcel.generic.ObjectType;
import org.aspectj.apache.bcel.generic.Tag;
import org.aspectj.apache.bcel.generic.TargetLostException;
import org.aspectj.apache.bcel.generic.Type;
import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.weaver.AjAttribute;
import org.aspectj.weaver.AjAttribute.WeaverVersionInfo;
import org.aspectj.weaver.AnnotationAJ;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.ISourceContext;
import org.aspectj.weaver.MemberImpl;
import org.aspectj.weaver.NameMangler;
import org.aspectj.weaver.ResolvedMember;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.World;
import org.aspectj.weaver.tools.Traceable;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

public final class LazyMethodGen implements Traceable {

    private int modifiers;

    private Type returnType;

    private final String name;

    private Type[] argumentTypes;

    private String[] declaredExceptions;

    private InstructionList body;

    private List<Attribute> attributes;

    private List<AnnotationAJ> newAnnotations;

    private List<ResolvedType> annotationsForRemoval;

    private AnnotationAJ[][] newParameterAnnotations;

    private final LazyClassGen enclosingClass;

    private BcelMethod memberView;

    private AjAttribute.EffectiveSignatureAttribute effectiveSignature;

    int highestLineNumber = 0;

    boolean wasPackedOptimally = false;

    private Method savedMethod = null;

    private final boolean originalMethodHasLocalVariableTable;

    String fromFilename = null;

    private int maxLocals;

    private boolean canInline = true;

    private boolean isSynthetic = false;

    List<BcelShadow> matchedShadows;

    public ResolvedType definingType = null;

    static class LightweightBcelMethod extends BcelMethod {

        LightweightBcelMethod(BcelObjectType declaringType, Method method) {
            super(declaringType, method);
        }
    }

    public LazyMethodGen(int modifiers, Type returnType, String name, Type[] paramTypes, String[] declaredExceptions, LazyClassGen enclosingClass) {
        this.memberView = null;
        this.modifiers = modifiers;
        this.returnType = returnType;
        this.name = name;
        this.argumentTypes = paramTypes;
        this.declaredExceptions = declaredExceptions;
        if (!Modifier.isAbstract(modifiers)) {
            body = new InstructionList();
            setMaxLocals(calculateMaxLocals());
        } else {
            body = null;
        }
        this.attributes = new ArrayList<>();
        this.enclosingClass = enclosingClass;
        assertGoodBody();
        this.originalMethodHasLocalVariableTable = true;
        if (memberView != null && isAdviceMethod()) {
            if (enclosingClass.getType().isAnnotationStyleAspect()) {
                this.canInline = false;
            }
        }
    }

    private int calculateMaxLocals() {
        int ret = Modifier.isStatic(modifiers) ? 0 : 1;
        for (Type type : argumentTypes) {
            ret += type.getSize();
        }
        return ret;
    }

    public LazyMethodGen(Method m, LazyClassGen enclosingClass) {
        savedMethod = m;
        this.enclosingClass = enclosingClass;
        if (!(m.isAbstract() || m.isNative()) && m.getCode() == null) {
            throw new RuntimeException("bad non-abstract method with no code: " + m + " on " + enclosingClass);
        }
        if ((m.isAbstract() || m.isNative()) && m.getCode() != null) {
            throw new RuntimeException("bad abstract method with code: " + m + " on " + enclosingClass);
        }
        this.memberView = new BcelMethod(enclosingClass.getBcelObjectType(), m);
        this.originalMethodHasLocalVariableTable = savedMethod.getLocalVariableTable() != null;
        this.modifiers = m.getModifiers();
        this.name = m.getName();
        if (memberView != null && isAdviceMethod()) {
            if (enclosingClass.getType().isAnnotationStyleAspect()) {
                this.canInline = false;
            }
        }
    }

    private boolean isAbstractOrNative(int modifiers) {
        return Modifier.isAbstract(modifiers) || Modifier.isNative(modifiers);
    }

    public LazyMethodGen(BcelMethod m, LazyClassGen enclosingClass) {
        savedMethod = m.getMethod();
        this.enclosingClass = enclosingClass;
        if (!isAbstractOrNative(m.getModifiers()) && savedMethod.getCode() == null) {
            throw new RuntimeException("bad non-abstract method with no code: " + m + " on " + enclosingClass);
        }
        if (isAbstractOrNative(m.getModifiers()) && savedMethod.getCode() != null) {
            throw new RuntimeException("bad abstract method with code: " + m + " on " + enclosingClass);
        }
        this.memberView = m;
        this.modifiers = savedMethod.getModifiers();
        this.name = m.getName();
        this.originalMethodHasLocalVariableTable = savedMethod.getLocalVariableTable() != null;
        if (memberView != null && isAdviceMethod()) {
            if (enclosingClass.getType().isAnnotationStyleAspect()) {
                this.canInline = false;
            }
        }
    }

    public boolean hasDeclaredLineNumberInfo() {
        return (memberView != null && memberView.hasDeclarationLineNumberInfo());
    }

    public int getDeclarationLineNumber() {
        if (hasDeclaredLineNumberInfo()) {
            return memberView.getDeclarationLineNumber();
        } else {
            return -1;
        }
    }

    public int getDeclarationOffset() {
        if (hasDeclaredLineNumberInfo()) {
            return memberView.getDeclarationOffset();
        } else {
            return 0;
        }
    }

    public void addAnnotation(AnnotationAJ ax) {
        initialize();
        if (memberView == null) {
            if (newAnnotations == null) {
                newAnnotations = new ArrayList<>();
            }
            newAnnotations.add(ax);
        } else {
            memberView.addAnnotation(ax);
        }
    }

    public void removeAnnotation(ResolvedType annotationType) {
        initialize();
        if (memberView == null) {
            if (annotationsForRemoval == null) {
                annotationsForRemoval = new ArrayList<>();
            }
            annotationsForRemoval.add(annotationType);
        } else {
            memberView.removeAnnotation(annotationType);
        }
    }

    public void addParameterAnnotation(int parameterNumber, AnnotationAJ anno) {
        initialize();
        if (memberView == null) {
            if (newParameterAnnotations == null) {
                int pcount = getArgumentTypes().length;
                newParameterAnnotations = new AnnotationAJ[pcount][];
                for (int i = 0; i < pcount; i++) {
                    if (i == parameterNumber) {
                        newParameterAnnotations[i] = new AnnotationAJ[1];
                        newParameterAnnotations[i][0] = anno;
                    } else {
                        newParameterAnnotations[i] = AnnotationAJ.EMPTY_ARRAY;
                    }
                }
            } else {
                AnnotationAJ[] currentAnnoArray = newParameterAnnotations[parameterNumber];
                AnnotationAJ[] newAnnoArray = new AnnotationAJ[currentAnnoArray.length + 1];
                System.arraycopy(currentAnnoArray, 0, newAnnoArray, 0, currentAnnoArray.length);
                newAnnoArray[currentAnnoArray.length] = anno;
                newParameterAnnotations[parameterNumber] = newAnnoArray;
            }
        } else {
            memberView.addParameterAnnotation(parameterNumber, anno);
        }
    }

    public ResolvedType[] getAnnotationTypes() {
        initialize();
        if (memberView == null && newAnnotations != null && newAnnotations.size() != 0) {
            ResolvedType[] annotationTypes = new ResolvedType[newAnnotations.size()];
            for (int a = 0, len = newAnnotations.size(); a < len; a++) {
                annotationTypes[a] = newAnnotations.get(a).getType();
            }
            return annotationTypes;
        }
        return null;
    }

    public AnnotationAJ[] getAnnotations() {
        initialize();
        if (memberView == null && newAnnotations != null && newAnnotations.size() != 0) {
            return newAnnotations.toArray(AnnotationAJ.EMPTY_ARRAY);
        }
        return null;
    }

    public boolean hasAnnotation(UnresolvedType annotationType) {
        initialize();
        if (memberView == null) {
            if (annotationsForRemoval != null) {
                for (ResolvedType at : annotationsForRemoval) {
                    if (at.equals(annotationType)) {
                        return false;
                    }
                }
            }
            if (newAnnotations != null) {
                for (AnnotationAJ annotation : newAnnotations) {
                    if (annotation.getTypeSignature().equals(annotationType.getSignature())) {
                        return true;
                    }
                }
            }
            memberView = new BcelMethod(getEnclosingClass().getBcelObjectType(), getMethod());
            return memberView.hasAnnotation(annotationType);
        }
        return memberView.hasAnnotation(annotationType);
    }

    private void initialize() {
        if (returnType != null) {
            return;
        }
        MethodGen gen = new MethodGen(savedMethod, enclosingClass.getName(), enclosingClass.getConstantPool(), true);
        this.returnType = gen.getReturnType();
        this.argumentTypes = gen.getArgumentTypes();
        this.declaredExceptions = gen.getExceptions();
        this.attributes = gen.getAttributes();
        this.maxLocals = gen.getMaxLocals();
        if (gen.isAbstract() || gen.isNative()) {
            body = null;
        } else {
            body = gen.getInstructionList();
            unpackHandlers(gen);
            ensureAllLineNumberSetup();
            highestLineNumber = gen.getHighestlinenumber();
        }
        assertGoodBody();
    }

    private void unpackHandlers(MethodGen gen) {
        CodeExceptionGen[] exns = gen.getExceptionHandlers();
        if (exns != null) {
            int len = exns.length;
            int priority = len - 1;
            for (int i = 0; i < len; i++, priority--) {
                CodeExceptionGen exn = exns[i];
                InstructionHandle start = Range.genStart(body, getOutermostExceptionStart(exn.getStartPC()));
                InstructionHandle end = Range.genEnd(body, getOutermostExceptionEnd(exn.getEndPC()));
                ExceptionRange er = new ExceptionRange(body, exn.getCatchType() == null ? null : BcelWorld.fromBcel(exn.getCatchType()), priority);
                er.associateWithTargets(start, end, exn.getHandlerPC());
                exn.setStartPC(null);
                exn.setEndPC(null);
                exn.setHandlerPC(null);
            }
            gen.removeExceptionHandlers();
        }
    }

    private InstructionHandle getOutermostExceptionStart(InstructionHandle ih) {
        while (true) {
            if (ExceptionRange.isExceptionStart(ih.getPrev())) {
                ih = ih.getPrev();
            } else {
                return ih;
            }
        }
    }

    private InstructionHandle getOutermostExceptionEnd(InstructionHandle ih) {
        while (true) {
            if (ExceptionRange.isExceptionEnd(ih.getNext())) {
                ih = ih.getNext();
            } else {
                return ih;
            }
        }
    }

    public void ensureAllLineNumberSetup() {
        LineNumberTag lastKnownLineNumberTag = null;
        boolean skip = false;
        for (InstructionHandle ih = body.getStart(); ih != null; ih = ih.getNext()) {
            skip = false;
            for (InstructionTargeter targeter : ih.getTargeters()) {
                if (targeter instanceof LineNumberTag) {
                    lastKnownLineNumberTag = (LineNumberTag) targeter;
                    skip = true;
                }
            }
            if (lastKnownLineNumberTag != null && !skip) {
                ih.addTargeter(lastKnownLineNumberTag);
            }
        }
    }

    public int allocateLocal(Type type) {
        return allocateLocal(type.getSize());
    }

    public int allocateLocal(int slots) {
        int max = getMaxLocals();
        setMaxLocals(max + slots);
        return max;
    }

    public Method getMethod() {
        if (savedMethod != null) {
            return savedMethod;
        }
        try {
            MethodGen gen = pack();
            savedMethod = gen.getMethod();
            return savedMethod;
        } catch (ClassGenException e) {
            enclosingClass.getBcelObjectType().getResolvedTypeX().getWorld().showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.PROBLEM_GENERATING_METHOD, this.getClassName(), this.getName(), e.getMessage()), this.getMemberView() == null ? null : this.getMemberView().getSourceLocation(), null);
            body = null;
            MethodGen gen = pack();
            return gen.getMethod();
        } catch (RuntimeException re) {
            if (re.getCause() instanceof ClassGenException) {
                enclosingClass.getBcelObjectType().getResolvedTypeX().getWorld().showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.PROBLEM_GENERATING_METHOD, this.getClassName(), this.getName(), re.getCause().getMessage()), this.getMemberView() == null ? null : this.getMemberView().getSourceLocation(), null);
                body = null;
                MethodGen gen = pack();
                return gen.getMethod();
            }
            throw re;
        }
    }

    public void markAsChanged() {
        if (wasPackedOptimally) {
            throw new RuntimeException("Already packed method is being re-modified: " + getClassName() + " " + toShortString());
        }
        initialize();
        savedMethod = null;
    }

    @Override
    public String toString() {
        BcelObjectType bot = enclosingClass.getBcelObjectType();
        WeaverVersionInfo weaverVersion = (bot == null ? WeaverVersionInfo.CURRENT : bot.getWeaverVersionAttribute());
        return toLongString(weaverVersion);
    }

    public String toShortString() {
        String access = org.aspectj.apache.bcel.classfile.Utility.accessToString(getAccessFlags());
        StringBuilder buf = new StringBuilder();
        if (!access.equals("")) {
            buf.append(access);
            buf.append(" ");
        }
        buf.append(org.aspectj.apache.bcel.classfile.Utility.signatureToString(getReturnType().getSignature(), true));
        buf.append(" ");
        buf.append(getName());
        buf.append("(");
        {
            int len = argumentTypes.length;
            if (len > 0) {
                buf.append(org.aspectj.apache.bcel.classfile.Utility.signatureToString(argumentTypes[0].getSignature(), true));
                for (int i = 1; i < argumentTypes.length; i++) {
                    buf.append(", ");
                    buf.append(org.aspectj.apache.bcel.classfile.Utility.signatureToString(argumentTypes[i].getSignature(), true));
                }
            }
        }
        buf.append(")");
        {
            int len = declaredExceptions != null ? declaredExceptions.length : 0;
            if (len > 0) {
                buf.append(" throws ");
                buf.append(declaredExceptions[0]);
                for (int i = 1; i < declaredExceptions.length; i++) {
                    buf.append(", ");
                    buf.append(declaredExceptions[i]);
                }
            }
        }
        return buf.toString();
    }

    public String toLongString(WeaverVersionInfo weaverVersion) {
        ByteArrayOutputStream s = new ByteArrayOutputStream();
        print(new PrintStream(s), weaverVersion);
        return new String(s.toByteArray());
    }

    public void print(WeaverVersionInfo weaverVersion) {
        print(System.out, weaverVersion);
    }

    public void print(PrintStream out, WeaverVersionInfo weaverVersion) {
        out.print("  " + toShortString());
        printAspectAttributes(out, weaverVersion);
        InstructionList body = getBody();
        if (body == null) {
            out.println(";");
            return;
        }
        out.println(":");
        new BodyPrinter(out).run();
        out.println("  end " + toShortString());
    }

    private void printAspectAttributes(PrintStream out, WeaverVersionInfo weaverVersion) {
        ISourceContext context = null;
        if (enclosingClass != null && enclosingClass.getType() != null) {
            context = enclosingClass.getType().getSourceContext();
        }
        List<AjAttribute> as = Utility.readAjAttributes(getClassName(), attributes.toArray(Attribute.NoAttributes), context, null, weaverVersion, new BcelConstantPoolReader(this.enclosingClass.getConstantPool()));
        if (!as.isEmpty()) {
            out.println("    " + as.get(0));
        }
    }

    private class BodyPrinter {

        Map<InstructionHandle, String> labelMap = new HashMap<>();

        InstructionList body;

        PrintStream out;

        ConstantPool pool;

        BodyPrinter(PrintStream out) {
            this.pool = enclosingClass.getConstantPool();
            this.body = getBodyForPrint();
            this.out = out;
        }

        BodyPrinter(PrintStream out, InstructionList il) {
            this.pool = enclosingClass.getConstantPool();
            this.body = il;
            this.out = out;
        }

        void run() {
            assignLabels();
            print();
        }

        void assignLabels() {
            LinkedList<ExceptionRange> exnTable = new LinkedList<>();
            String pendingLabel = null;
            int lcounter = 0;
            for (InstructionHandle ih = body.getStart(); ih != null; ih = ih.getNext()) {
                for (InstructionTargeter t : ih.getTargeters()) {
                    if (t instanceof ExceptionRange) {
                        ExceptionRange r = (ExceptionRange) t;
                        if (r.getStart() == ih) {
                            insertHandler(r, exnTable);
                        }
                    } else if (t instanceof InstructionBranch) {
                        if (pendingLabel == null) {
                            pendingLabel = "L" + lcounter++;
                        }
                    } else {
                    }
                }
                if (pendingLabel != null) {
                    labelMap.put(ih, pendingLabel);
                    if (!Range.isRangeHandle(ih)) {
                        pendingLabel = null;
                    }
                }
            }
            int ecounter = 0;
            for (ExceptionRange er : exnTable) {
                String exceptionLabel = "E" + ecounter++;
                labelMap.put(Range.getRealStart(er.getHandler()), exceptionLabel);
                labelMap.put(er.getHandler(), exceptionLabel);
            }
        }

        void print() {
            int depth = 0;
            int currLine = -1;
            bodyPrint:
            for (InstructionHandle ih = body.getStart(); ih != null; ih = ih.getNext()) {
                if (Range.isRangeHandle(ih)) {
                    Range r = Range.getRange(ih);
                    for (InstructionHandle xx = r.getStart(); Range.isRangeHandle(xx); xx = xx.getNext()) {
                        if (xx == r.getEnd()) {
                            continue bodyPrint;
                        }
                    }
                    if (r.getStart() == ih) {
                        printRangeString(r, depth++);
                    } else {
                        if (r.getEnd() != ih) {
                            throw new RuntimeException("bad");
                        }
                        printRangeString(r, --depth);
                    }
                } else {
                    printInstruction(ih, depth);
                    int line = getLineNumber(ih, currLine);
                    if (line != currLine) {
                        currLine = line;
                        out.println("   (line " + line + ")");
                    } else {
                        out.println();
                    }
                }
            }
        }

        void printRangeString(Range r, int depth) {
            printDepth(depth);
            out.println(getRangeString(r, labelMap));
        }

        String getRangeString(Range r, Map<InstructionHandle, String> labelMap) {
            if (r instanceof ExceptionRange) {
                ExceptionRange er = (ExceptionRange) r;
                return er.toString() + " -> " + labelMap.get(er.getHandler());
            } else {
                return r.toString();
            }
        }

        void printDepth(int depth) {
            pad(BODY_INDENT);
            while (depth > 0) {
                out.print("| ");
                depth--;
            }
        }

        void printLabel(String s, int depth) {
            int space = Math.max(CODE_INDENT - depth * 2, 0);
            if (s == null) {
                pad(space);
            } else {
                space = Math.max(space - (s.length() + 2), 0);
                pad(space);
                out.print(s);
                out.print(": ");
            }
        }

        void printInstruction(InstructionHandle h, int depth) {
            printDepth(depth);
            printLabel(labelMap.get(h), depth);
            Instruction inst = h.getInstruction();
            if (inst.isConstantPoolInstruction()) {
                out.print(Constants.OPCODE_NAMES[inst.opcode].toUpperCase());
                out.print(" ");
                out.print(pool.constantToString(pool.getConstant(inst.getIndex())));
            } else if (inst instanceof InstructionSelect) {
                InstructionSelect sinst = (InstructionSelect) inst;
                out.println(Constants.OPCODE_NAMES[sinst.opcode].toUpperCase());
                int[] matches = sinst.getMatchs();
                InstructionHandle[] targets = sinst.getTargets();
                InstructionHandle defaultTarget = sinst.getTarget();
                for (int i = 0, len = matches.length; i < len; i++) {
                    printDepth(depth);
                    printLabel(null, depth);
                    out.print("  ");
                    out.print(matches[i]);
                    out.print(": \t");
                    out.println(labelMap.get(targets[i]));
                }
                printDepth(depth);
                printLabel(null, depth);
                out.print("  ");
                out.print("default: \t");
                out.print(labelMap.get(defaultTarget));
            } else if (inst instanceof InstructionBranch) {
                InstructionBranch brinst = (InstructionBranch) inst;
                out.print(Constants.OPCODE_NAMES[brinst.getOpcode()].toUpperCase());
                out.print(" ");
                out.print(labelMap.get(brinst.getTarget()));
            } else if (inst.isLocalVariableInstruction()) {
                out.print(inst.toString(false).toUpperCase());
                int index = inst.getIndex();
                LocalVariableTag tag = getLocalVariableTag(h, index);
                if (tag != null) {
                    out.print("     // ");
                    out.print(tag.getType());
                    out.print(" ");
                    out.print(tag.getName());
                }
            } else {
                out.print(inst.toString(false).toUpperCase());
            }
        }

        static final int BODY_INDENT = 4;

        static final int CODE_INDENT = 16;

        void pad(int size) {
            for (int i = 0; i < size; i++) {
                out.print(" ");
            }
        }
    }

    static LocalVariableTag getLocalVariableTag(InstructionHandle ih, int index) {
        for (InstructionTargeter t : ih.getTargeters()) {
            if (t instanceof LocalVariableTag) {
                LocalVariableTag lvt = (LocalVariableTag) t;
                if (lvt.getSlot() == index) {
                    return lvt;
                }
            }
        }
        return null;
    }

    static int getLineNumber(InstructionHandle ih, int prevLine) {
        for (InstructionTargeter t : ih.getTargeters()) {
            if (t instanceof LineNumberTag) {
                return ((LineNumberTag) t).getLineNumber();
            }
        }
        return prevLine;
    }

    public boolean isStatic() {
        return Modifier.isStatic(getAccessFlags());
    }

    public boolean isAbstract() {
        return Modifier.isAbstract(getAccessFlags());
    }

    public boolean isBridgeMethod() {
        return (getAccessFlags() & Constants.ACC_BRIDGE) != 0;
    }

    public void addExceptionHandler(InstructionHandle start, InstructionHandle end, InstructionHandle handlerStart, ObjectType catchType, boolean highPriority) {
        InstructionHandle start1 = Range.genStart(body, start);
        InstructionHandle end1 = Range.genEnd(body, end);
        ExceptionRange er = new ExceptionRange(body, (catchType == null ? null : BcelWorld.fromBcel(catchType)), highPriority);
        er.associateWithTargets(start1, end1, handlerStart);
    }

    public int getAccessFlags() {
        return modifiers;
    }

    public int getAccessFlagsWithoutSynchronized() {
        if (isSynchronized()) {
            return modifiers - Modifier.SYNCHRONIZED;
        }
        return modifiers;
    }

    public boolean isSynchronized() {
        return (modifiers & Modifier.SYNCHRONIZED) != 0;
    }

    public void setAccessFlags(int newFlags) {
        this.modifiers = newFlags;
    }

    public Type[] getArgumentTypes() {
        initialize();
        return argumentTypes;
    }

    public LazyClassGen getEnclosingClass() {
        return enclosingClass;
    }

    public int getMaxLocals() {
        return maxLocals;
    }

    public String getName() {
        return name;
    }

    public String getGenericReturnTypeSignature() {
        if (memberView == null) {
            return getReturnType().getSignature();
        } else {
            return memberView.getGenericReturnType().getSignature();
        }
    }

    public Type getReturnType() {
        initialize();
        return returnType;
    }

    public void setMaxLocals(int maxLocals) {
        this.maxLocals = maxLocals;
    }

    public InstructionList getBody() {
        markAsChanged();
        return body;
    }

    public InstructionList getBodyForPrint() {
        return body;
    }

    public boolean hasBody() {
        if (savedMethod != null) {
            return savedMethod.getCode() != null;
        }
        return body != null;
    }

    public List<Attribute> getAttributes() {
        return attributes;
    }

    public String[] getDeclaredExceptions() {
        return declaredExceptions;
    }

    public String getClassName() {
        return enclosingClass.getName();
    }

    public MethodGen pack() {
        forceSyntheticForAjcMagicMembers();
        int flags = getAccessFlags();
        if (enclosingClass.getWorld().isJoinpointSynchronizationEnabled() && enclosingClass.getWorld().areSynchronizationPointcutsInUse()) {
            flags = getAccessFlagsWithoutSynchronized();
        }
        MethodGen gen = new // getArgumentNames(),
                MethodGen(// getArgumentNames(),
                flags, // getArgumentNames(),
                getReturnType(), // getArgumentNames(),
                getArgumentTypes(), null, getName(), getEnclosingClass().getName(), new InstructionList(), getEnclosingClass().getConstantPool());
        for (String declaredException : declaredExceptions) {
            gen.addException(declaredException);
        }
        for (Attribute attr : attributes) {
            gen.addAttribute(attr);
        }
        if (newAnnotations != null) {
            for (AnnotationAJ element : newAnnotations) {
                gen.addAnnotation(new AnnotationGen(((BcelAnnotation) element).getBcelAnnotation(), gen.getConstantPool(), true));
            }
        }
        if (newParameterAnnotations != null) {
            for (int i = 0; i < newParameterAnnotations.length; i++) {
                AnnotationAJ[] annos = newParameterAnnotations[i];
                for (AnnotationAJ anno : annos) {
                    gen.addParameterAnnotation(i, new AnnotationGen(((BcelAnnotation) anno).getBcelAnnotation(), gen.getConstantPool(), true));
                }
            }
        }
        if (memberView != null && memberView.getAnnotations() != null && memberView.getAnnotations().length != 0) {
            AnnotationAJ[] ans = memberView.getAnnotations();
            for (AnnotationAJ an : ans) {
                AnnotationGen a = ((BcelAnnotation) an).getBcelAnnotation();
                gen.addAnnotation(new AnnotationGen(a, gen.getConstantPool(), true));
            }
        }
        if (isSynthetic) {
            if (enclosingClass.getWorld().isInJava5Mode()) {
                gen.setModifiers(gen.getModifiers() | Constants.ACC_SYNTHETIC);
            }
            if (!hasAttribute("Synthetic")) {
                ConstantPool cpg = gen.getConstantPool();
                int index = cpg.addUtf8("Synthetic");
                gen.addAttribute(new Synthetic(index, 0, new byte[0], cpg));
            }
        }
        if (hasBody()) {
            if (this.enclosingClass.getWorld().shouldFastPackMethods()) {
                if (isAdviceMethod() || getName().equals("<clinit>")) {
                    packBody(gen);
                } else {
                    optimizedPackBody(gen);
                }
            } else {
                packBody(gen);
            }
            gen.setMaxLocals(true);
            gen.setMaxStack();
        } else {
            gen.setInstructionList(null);
        }
        return gen;
    }

    private boolean hasAttribute(String attributeName) {
        for (Attribute attr : attributes) {
            if (attr.getName().equals(attributeName)) {
                return true;
            }
        }
        return false;
    }

    private void forceSyntheticForAjcMagicMembers() {
        if (NameMangler.isSyntheticMethod(getName(), inAspect())) {
            makeSynthetic();
        }
    }

    private boolean inAspect() {
        BcelObjectType objectType = enclosingClass.getBcelObjectType();
        return (objectType == null ? false : objectType.isAspect());
    }

    public void makeSynthetic() {
        isSynthetic = true;
    }

    private static class LVPosition {

        InstructionHandle start = null;

        InstructionHandle end = null;
    }

    public void packBody(MethodGen gen) {
        InstructionList fresh = gen.getInstructionList();
        Map<InstructionHandle, InstructionHandle> map = copyAllInstructionsExceptRangeInstructionsInto(fresh);
        InstructionHandle oldInstructionHandle = getBody().getStart();
        InstructionHandle newInstructionHandle = fresh.getStart();
        LinkedList<ExceptionRange> exceptionList = new LinkedList<>();
        Map<LocalVariableTag, LVPosition> localVariables = new HashMap<>();
        int currLine = -1;
        int lineNumberOffset = (fromFilename == null) ? 0 : getEnclosingClass().getSourceDebugExtensionOffset(fromFilename);
        while (oldInstructionHandle != null) {
            if (map.get(oldInstructionHandle) == null) {
                handleRangeInstruction(oldInstructionHandle, exceptionList);
                oldInstructionHandle = oldInstructionHandle.getNext();
            } else {
                Instruction oldInstruction = oldInstructionHandle.getInstruction();
                Instruction newInstruction = newInstructionHandle.getInstruction();
                if (oldInstruction instanceof InstructionBranch) {
                    handleBranchInstruction(map, oldInstruction, newInstruction);
                }
                for (InstructionTargeter targeter : oldInstructionHandle.getTargeters()) {
                    if (targeter instanceof LineNumberTag) {
                        int line = ((LineNumberTag) targeter).getLineNumber();
                        if (line != currLine) {
                            gen.addLineNumber(newInstructionHandle, line + lineNumberOffset);
                            currLine = line;
                        }
                    } else if (targeter instanceof LocalVariableTag) {
                        LocalVariableTag lvt = (LocalVariableTag) targeter;
                        LVPosition p = localVariables.get(lvt);
                        if (p == null) {
                            LVPosition newp = new LVPosition();
                            newp.start = newp.end = newInstructionHandle;
                            localVariables.put(lvt, newp);
                        } else {
                            p.end = newInstructionHandle;
                        }
                    }
                }
                oldInstructionHandle = oldInstructionHandle.getNext();
                newInstructionHandle = newInstructionHandle.getNext();
            }
        }
        addExceptionHandlers(gen, map, exceptionList);
        if (originalMethodHasLocalVariableTable || enclosingClass.getBcelObjectType().getResolvedTypeX().getWorld().generateNewLvts) {
            if (localVariables.size() == 0) {
                createNewLocalVariables(gen);
            } else {
                addLocalVariables(gen, localVariables);
            }
        }
        if (gen.getLineNumbers().length == 0) {
            gen.addLineNumber(gen.getInstructionList().getStart(), 1);
        }
    }

    private void createNewLocalVariables(MethodGen gen) {
        gen.removeLocalVariables();
        if (!getName().startsWith("<")) {
            int slot = 0;
            InstructionHandle start = gen.getInstructionList().getStart();
            InstructionHandle end = gen.getInstructionList().getEnd();
            if (!isStatic()) {
                String cname = this.enclosingClass.getClassName();
                if (cname == null) {
                    return;
                }
                Type enclosingType = BcelWorld.makeBcelType(UnresolvedType.forName(cname));
                gen.addLocalVariable("this", enclosingType, slot++, start, end);
            }
            String[] paramNames = (memberView == null ? null : memberView.getParameterNames());
            if (paramNames != null) {
                for (int i = 0; i < argumentTypes.length; i++) {
                    String pname = paramNames[i];
                    if (pname == null) {
                        pname = "arg" + i;
                    }
                    gen.addLocalVariable(pname, argumentTypes[i], slot, start, end);
                    slot += argumentTypes[i].getSize();
                }
            }
        }
    }

    private World getWorld() {
        return enclosingClass.getBcelObjectType().getResolvedTypeX().getWorld();
    }

    public void optimizedPackBody(MethodGen gen) {
        InstructionList theBody = getBody();
        InstructionHandle iHandle = theBody.getStart();
        int currLine = -1;
        int lineNumberOffset = (fromFilename == null) ? 0 : getEnclosingClass().getSourceDebugExtensionOffset(fromFilename);
        Map<LocalVariableTag, LVPosition> localVariables = new HashMap<>();
        LinkedList<ExceptionRange> exceptionList = new LinkedList<>();
        Set<InstructionHandle> forDeletion = new HashSet<>();
        Set<BranchHandle> branchInstructions = new HashSet<>();
        while (iHandle != null) {
            Instruction inst = iHandle.getInstruction();
            if (inst == Range.RANGEINSTRUCTION) {
                Range r = Range.getRange(iHandle);
                if (r instanceof ExceptionRange) {
                    ExceptionRange er = (ExceptionRange) r;
                    if (er.getStart() == iHandle) {
                        if (!er.isEmpty()) {
                            insertHandler(er, exceptionList);
                        }
                    }
                }
                forDeletion.add(iHandle);
            } else {
                if (inst instanceof InstructionBranch) {
                    branchInstructions.add((BranchHandle) iHandle);
                }
                for (InstructionTargeter targeter : iHandle.getTargetersCopy()) {
                    if (targeter instanceof LineNumberTag) {
                        int line = ((LineNumberTag) targeter).getLineNumber();
                        if (line != currLine) {
                            gen.addLineNumber(iHandle, line + lineNumberOffset);
                            currLine = line;
                        }
                    } else if (targeter instanceof LocalVariableTag) {
                        LocalVariableTag lvt = (LocalVariableTag) targeter;
                        LVPosition p = localVariables.get(lvt);
                        if (p == null) {
                            LVPosition newp = new LVPosition();
                            newp.start = newp.end = iHandle;
                            localVariables.put(lvt, newp);
                        } else {
                            p.end = iHandle;
                        }
                    }
                }
            }
            iHandle = iHandle.getNext();
        }
        for (BranchHandle branchHandle : branchInstructions) {
            handleBranchInstruction(branchHandle, forDeletion);
        }
        for (ExceptionRange r : exceptionList) {
            if (r.isEmpty()) {
                continue;
            }
            gen.addExceptionHandler(jumpForward(r.getRealStart(), forDeletion), jumpForward(r.getRealEnd(), forDeletion), jumpForward(r.getHandler(), forDeletion), (r.getCatchType() == null) ? null : (ObjectType) BcelWorld.makeBcelType(r.getCatchType()));
        }
        for (InstructionHandle handle : forDeletion) {
            try {
                theBody.delete(handle);
            } catch (TargetLostException e) {
                e.printStackTrace();
            }
        }
        gen.setInstructionList(theBody);
        if (originalMethodHasLocalVariableTable || getWorld().generateNewLvts) {
            if (localVariables.size() == 0) {
                createNewLocalVariables(gen);
            } else {
                addLocalVariables(gen, localVariables);
            }
        }
        if (gen.getLineNumbers().length == 0) {
            gen.addLineNumber(gen.getInstructionList().getStart(), 1);
        }
        wasPackedOptimally = true;
    }

    private void addLocalVariables(MethodGen gen, Map<LocalVariableTag, LVPosition> localVariables) {
        gen.removeLocalVariables();
        InstructionHandle methodStart = gen.getInstructionList().getStart();
        InstructionHandle methodEnd = gen.getInstructionList().getEnd();
        int paramSlots = gen.isStatic() ? 0 : 1;
        Type[] argTypes = gen.getArgumentTypes();
        if (argTypes != null) {
            for (Type argType : argTypes) {
                if (argType.getSize() == 2) {
                    paramSlots += 2;
                } else {
                    paramSlots += 1;
                }
            }
        }
        if (!this.enclosingClass.getWorld().generateNewLvts) {
            paramSlots = -1;
        }
        Map<InstructionHandle, Set<Integer>> duplicatedLocalMap = new HashMap<>();
        for (LocalVariableTag tag : localVariables.keySet()) {
            LVPosition lvpos = localVariables.get(tag);
            InstructionHandle start = (tag.getSlot() < paramSlots ? methodStart : lvpos.start);
            InstructionHandle end = (tag.getSlot() < paramSlots ? methodEnd : lvpos.end);
            Set<Integer> slots = duplicatedLocalMap.get(start);
            if (slots == null) {
                slots = new HashSet<>();
                duplicatedLocalMap.put(start, slots);
            } else if (slots.contains(tag.getSlot())) {
                continue;
            }
            slots.add(tag.getSlot());
            Type t = tag.getRealType();
            if (t == null) {
                t = BcelWorld.makeBcelType(UnresolvedType.forSignature(tag.getType()));
            }
            gen.addLocalVariable(tag.getName(), t, tag.getSlot(), start, end);
        }
    }

    private void addExceptionHandlers(MethodGen gen, Map<InstructionHandle, InstructionHandle> map, Iterable<ExceptionRange> exnList) {
        for (ExceptionRange r : exnList) {
            if (r.isEmpty()) {
                continue;
            }
            InstructionHandle rMappedStart = remap(r.getRealStart(), map);
            InstructionHandle rMappedEnd = remap(r.getRealEnd(), map);
            InstructionHandle rMappedHandler = remap(r.getHandler(), map);
            gen.addExceptionHandler(rMappedStart, rMappedEnd, rMappedHandler, (r.getCatchType() == null) ? null : (ObjectType) BcelWorld.makeBcelType(r.getCatchType()));
        }
    }

    private void handleBranchInstruction(Map<InstructionHandle, InstructionHandle> map, Instruction oldInstruction, Instruction newInstruction) {
        InstructionBranch oldBranchInstruction = (InstructionBranch) oldInstruction;
        InstructionBranch newBranchInstruction = (InstructionBranch) newInstruction;
        InstructionHandle oldTarget = oldBranchInstruction.getTarget();
        newBranchInstruction.setTarget(remap(oldTarget, map));
        if (oldBranchInstruction instanceof InstructionSelect) {
            InstructionHandle[] oldTargets = ((InstructionSelect) oldBranchInstruction).getTargets();
            InstructionHandle[] newTargets = ((InstructionSelect) newBranchInstruction).getTargets();
            for (int k = oldTargets.length - 1; k >= 0; k--) {
                newTargets[k] = remap(oldTargets[k], map);
                newTargets[k].addTargeter(newBranchInstruction);
            }
        }
    }

    private InstructionHandle jumpForward(InstructionHandle t, Set<InstructionHandle> handlesForDeletion) {
        InstructionHandle target = t;
        if (handlesForDeletion.contains(target)) {
            do {
                target = target.getNext();
            } while (handlesForDeletion.contains(target));
        }
        return target;
    }

    private void handleBranchInstruction(BranchHandle branchHandle, Set<InstructionHandle> handlesForDeletion) {
        InstructionBranch branchInstruction = (InstructionBranch) branchHandle.getInstruction();
        InstructionHandle target = branchInstruction.getTarget();
        if (handlesForDeletion.contains(target)) {
            do {
                target = target.getNext();
            } while (handlesForDeletion.contains(target));
            branchInstruction.setTarget(target);
        }
        if (branchInstruction instanceof InstructionSelect) {
            InstructionSelect iSelect = (InstructionSelect) branchInstruction;
            InstructionHandle[] targets = iSelect.getTargets();
            for (int k = targets.length - 1; k >= 0; k--) {
                InstructionHandle oneTarget = targets[k];
                if (handlesForDeletion.contains(oneTarget)) {
                    do {
                        oneTarget = oneTarget.getNext();
                    } while (handlesForDeletion.contains(oneTarget));
                    iSelect.setTarget(k, oneTarget);
                    oneTarget.addTargeter(branchInstruction);
                }
            }
        }
    }

    private void handleRangeInstruction(InstructionHandle ih, LinkedList<ExceptionRange> exnList) {
        Range r = Range.getRange(ih);
        if (r instanceof ExceptionRange) {
            ExceptionRange er = (ExceptionRange) r;
            if (er.getStart() == ih) {
                if (!er.isEmpty()) {
                    insertHandler(er, exnList);
                }
            }
        } else {
        }
    }

    private Map<InstructionHandle, InstructionHandle> copyAllInstructionsExceptRangeInstructionsInto(InstructionList intoList) {
        Map<InstructionHandle, InstructionHandle> map = new HashMap<>();
        for (InstructionHandle ih = getBody().getStart(); ih != null; ih = ih.getNext()) {
            if (Range.isRangeHandle(ih)) {
                continue;
            }
            Instruction inst = ih.getInstruction();
            Instruction copy = Utility.copyInstruction(inst);
            if (copy instanceof InstructionBranch) {
                map.put(ih, intoList.append((InstructionBranch) copy));
            } else {
                map.put(ih, intoList.append(copy));
            }
        }
        return map;
    }

    private static InstructionHandle remap(InstructionHandle handle, Map<InstructionHandle, InstructionHandle> map) {
        while (true) {
            InstructionHandle ret = map.get(handle);
            if (ret == null) {
                handle = handle.getNext();
            } else {
                return ret;
            }
        }
    }

    static void insertHandler(ExceptionRange fresh, List<ExceptionRange> l) {
        for (ListIterator<ExceptionRange> iter = l.listIterator(); iter.hasNext(); ) {
            ExceptionRange r = iter.next();
            if (fresh.getPriority() >= r.getPriority()) {
                iter.previous();
                iter.add(fresh);
                return;
            }
        }
        l.add(fresh);
    }

    public boolean isPrivate() {
        return Modifier.isPrivate(getAccessFlags());
    }

    public boolean isProtected() {
        return Modifier.isProtected(getAccessFlags());
    }

    public boolean isDefault() {
        return !(isProtected() || isPrivate() || isPublic());
    }

    public boolean isPublic() {
        return Modifier.isPublic(getAccessFlags());
    }

    public void assertGoodBody() {
        if (true) {
            return;
        }
        assertGoodBody(getBody(), toString());
    }

    public static void assertGoodBody(InstructionList il, String from) {
        if (true) {
            return;
        }
    }

    private static void assertTargetedBy(InstructionHandle target, InstructionTargeter targeter, String from) {
        for (InstructionTargeter instructionTargeter : target.getTargeters()) {
            if (instructionTargeter == targeter) {
                return;
            }
        }
        throw new RuntimeException("bad targeting relationship in " + from);
    }

    private static void assertTargets(InstructionTargeter targeter, InstructionHandle target, String from) {
        if (targeter instanceof Range) {
            Range r = (Range) targeter;
            if (r.getStart() == target || r.getEnd() == target) {
                return;
            }
            if (r instanceof ExceptionRange) {
                if (((ExceptionRange) r).getHandler() == target) {
                    return;
                }
            }
        } else if (targeter instanceof InstructionBranch) {
            InstructionBranch bi = (InstructionBranch) targeter;
            if (bi.getTarget() == target) {
                return;
            }
            if (targeter instanceof InstructionSelect) {
                InstructionSelect sel = (InstructionSelect) targeter;
                InstructionHandle[] itargets = sel.getTargets();
                for (int k = itargets.length - 1; k >= 0; k--) {
                    if (itargets[k] == target) {
                        return;
                    }
                }
            }
        } else if (targeter instanceof Tag) {
            return;
        }
        throw new BCException(targeter + " doesn't target " + target + " in " + from);
    }

    private static Range getRangeAndAssertExactlyOne(InstructionHandle ih, String from) {
        Range ret = null;
        Iterator<InstructionTargeter> tIter = ih.getTargeters().iterator();
        if (!tIter.hasNext()) {
            throw new BCException("range handle with no range in " + from);
        }
        while (tIter.hasNext()) {
            InstructionTargeter ts = tIter.next();
            if (ts instanceof Range) {
                if (ret != null) {
                    throw new BCException("range handle with multiple ranges in " + from);
                }
                ret = (Range) ts;
            }
        }
        if (ret == null) {
            throw new BCException("range handle with no range in " + from);
        }
        return ret;
    }

    boolean isAdviceMethod() {
        if (memberView == null) {
            return false;
        }
        return memberView.getAssociatedShadowMunger() != null;
    }

    boolean isAjSynthetic() {
        if (memberView == null) {
            return true;
        }
        return memberView.isAjSynthetic();
    }

    boolean isSynthetic() {
        if (memberView == null) {
            return false;
        }
        return memberView.isSynthetic();
    }

    public ISourceLocation getSourceLocation() {
        if (memberView != null) {
            return memberView.getSourceLocation();
        }
        return null;
    }

    public AjAttribute.EffectiveSignatureAttribute getEffectiveSignature() {
        if (effectiveSignature != null) {
            return effectiveSignature;
        }
        return memberView.getEffectiveSignature();
    }

    public void setEffectiveSignature(ResolvedMember member, Shadow.Kind kind, boolean shouldWeave) {
        this.effectiveSignature = new AjAttribute.EffectiveSignatureAttribute(member, kind, shouldWeave);
    }

    public String getSignature() {
        if (memberView != null) {
            return memberView.getSignature();
        }
        return MemberImpl.typesToSignature(BcelWorld.fromBcel(getReturnType()), BcelWorld.fromBcel(getArgumentTypes()), false);
    }

    public String getParameterSignature() {
        if (memberView != null) {
            return memberView.getParameterSignature();
        }
        return MemberImpl.typesToSignature(BcelWorld.fromBcel(getArgumentTypes()));
    }

    public BcelMethod getMemberView() {
        return memberView;
    }

    public void forcePublic() {
        markAsChanged();
        modifiers = Utility.makePublic(modifiers);
    }

    public boolean getCanInline() {
        return canInline;
    }

    public void setCanInline(boolean canInline) {
        this.canInline = canInline;
    }

    public void addAttribute(Attribute attribute) {
        attributes.add(attribute);
    }

    public String toTraceString() {
        return toShortString();
    }

    public ConstantPool getConstantPool() {
        return enclosingClass.getConstantPool();
    }

    public static boolean isConstructor(LazyMethodGen aMethod) {
        return aMethod.getName().equals("<init>");
    }
}
