package org.aspectj.weaver.loadtime;

import org.aspectj.bridge.context.CompilationAndWeavingContext;
import org.aspectj.weaver.Dump;
import org.aspectj.weaver.tools.Trace;
import org.aspectj.weaver.tools.TraceFactory;
import org.aspectj.weaver.tools.WeavingAdaptor;
import org.aspectj.weaver.tools.cache.SimpleCache;
import org.aspectj.weaver.tools.cache.SimpleCacheFactory;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

public class Aj implements ClassPreProcessor {

    private IWeavingContext weavingContext;

    public static SimpleCache laCache = SimpleCacheFactory.createSimpleCache();

    private static ReferenceQueue adaptorQueue = new ReferenceQueue();

    private static Trace trace = TraceFactory.getTraceFactory().getTrace(Aj.class);

    public Aj() {
        this(null);
    }

    public Aj(IWeavingContext context) {
        if (trace.isTraceEnabled())
            trace.enter("<init>", this, new Object[]{context, getClass().getClassLoader()});
        this.weavingContext = context;
        if (trace.isTraceEnabled())
            trace.exit("<init>");
    }

    @Override
    public void initialize() {
    }

    private final static String deleLoader = "sun.reflect.DelegatingClassLoader";

    private final static String deleLoader2 = "jdk.internal.reflect.DelegatingClassLoader";

    @Override
    public byte[] preProcess(String className, byte[] bytes, ClassLoader loader, ProtectionDomain protectionDomain) {
        if (loader == null || className == null || loader.getClass().getName().equals(deleLoader) || loader.getClass().getName().equals(deleLoader2)) {
            return bytes;
        }
        if (loadersToSkip != null) {
            if (loadersToSkip.contains(loader.getClass().getName())) {
                return bytes;
            }
        }
        if (trace.isTraceEnabled())
            trace.enter("preProcess", this, new Object[]{className, bytes, loader});
        if (trace.isTraceEnabled())
            trace.event("preProcess", this, new Object[]{loader.getParent(), Thread.currentThread().getContextClassLoader()});
        try {
            synchronized (loader) {
                if (SimpleCacheFactory.isEnabled()) {
                    byte[] cacheBytes = laCache.getAndInitialize(className, bytes, loader, protectionDomain);
                    if (cacheBytes != null) {
                        return cacheBytes;
                    }
                }
                WeavingAdaptor weavingAdaptor = WeaverContainer.getWeaver(loader, weavingContext);
                if (weavingAdaptor == null) {
                    if (trace.isTraceEnabled())
                        trace.exit("preProcess");
                    return bytes;
                }
                try {
                    weavingAdaptor.setActiveProtectionDomain(protectionDomain);
                    byte[] newBytes = weavingAdaptor.weaveClass(className, bytes, false);
                    Dump.dumpOnExit(weavingAdaptor.getMessageHolder(), true);
                    if (trace.isTraceEnabled())
                        trace.exit("preProcess", newBytes);
                    if (SimpleCacheFactory.isEnabled()) {
                        laCache.put(className, bytes, newBytes);
                    }
                    return newBytes;
                } finally {
                    weavingAdaptor.setActiveProtectionDomain(null);
                }
            }
        } catch (Throwable th) {
            trace.error(className, th);
            Dump.dumpWithException(th);
            if (trace.isTraceEnabled())
                trace.exit("preProcess", th);
            return bytes;
        } finally {
            CompilationAndWeavingContext.resetForThread();
        }
    }

    private static class AdaptorKey extends WeakReference {

        private final int loaderHashCode, sysHashCode, hashValue;

        private final String loaderClass;

        public AdaptorKey(ClassLoader loader) {
            super(loader, adaptorQueue);
            loaderHashCode = loader.hashCode();
            sysHashCode = System.identityHashCode(loader);
            loaderClass = loader.getClass().getName();
            hashValue = loaderHashCode + sysHashCode + loaderClass.hashCode();
        }

        public ClassLoader getClassLoader() {
            ClassLoader instance = (ClassLoader) get();
            return instance;
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof AdaptorKey)) {
                return false;
            }
            AdaptorKey other = (AdaptorKey) obj;
            return (other.loaderHashCode == loaderHashCode) && (other.sysHashCode == sysHashCode) && loaderClass.equals(other.loaderClass);
        }

        @Override
        public int hashCode() {
            return hashValue;
        }
    }

    public static int removeStaleAdaptors(boolean displayProgress) {
        int removed = 0;
        synchronized (WeaverContainer.weavingAdaptors) {
            if (displayProgress) {
                System.err.println("Weaver adaptors before queue processing:");
                Map<AdaptorKey, ExplicitlyInitializedClassLoaderWeavingAdaptor> m = WeaverContainer.weavingAdaptors;
                Set<AdaptorKey> keys = m.keySet();
                for (Object object : keys) {
                    System.err.println(object + " = " + WeaverContainer.weavingAdaptors.get(object));
                }
            }
            Object o = adaptorQueue.poll();
            while (o != null) {
                if (displayProgress)
                    System.err.println("Processing referencequeue entry " + o);
                AdaptorKey wo = (AdaptorKey) o;
                boolean didit = WeaverContainer.weavingAdaptors.remove(wo) != null;
                if (didit) {
                    removed++;
                } else {
                    throw new RuntimeException("Eh?? key=" + wo);
                }
                if (displayProgress)
                    System.err.println("Removed? " + didit);
                o = adaptorQueue.poll();
            }
            if (displayProgress) {
                System.err.println("Weaver adaptors after queue processing:");
                Map<AdaptorKey, ExplicitlyInitializedClassLoaderWeavingAdaptor> m = WeaverContainer.weavingAdaptors;
                Set<AdaptorKey> keys = m.keySet();
                for (Object object : keys) {
                    System.err.println(object + " = " + WeaverContainer.weavingAdaptors.get(object));
                }
            }
        }
        return removed;
    }

    public static int getActiveAdaptorCount() {
        return WeaverContainer.weavingAdaptors.size();
    }

    public static void checkQ() {
        synchronized (adaptorQueue) {
            Object o = adaptorQueue.poll();
            while (o != null) {
                AdaptorKey wo = (AdaptorKey) o;
                WeaverContainer.weavingAdaptors.remove(wo);
                o = adaptorQueue.poll();
            }
        }
    }

    public static List<String> loadersToSkip = null;

    static {
        new ExplicitlyInitializedClassLoaderWeavingAdaptor(new ClassLoaderWeavingAdaptor());
        try {
            String loadersToSkipProperty = System.getProperty("aj.weaving.loadersToSkip", "");
            StringTokenizer st = new StringTokenizer(loadersToSkipProperty, ",");
            if (loadersToSkipProperty != null && loadersToSkip == null) {
                if (st.hasMoreTokens()) {
                    loadersToSkip = new ArrayList<>();
                }
                while (st.hasMoreTokens()) {
                    String nextLoader = st.nextToken();
                    loadersToSkip.add(nextLoader);
                }
            }
        } catch (Exception e) {
        }
    }

    static class WeaverContainer {

        final static Map<AdaptorKey, ExplicitlyInitializedClassLoaderWeavingAdaptor> weavingAdaptors = Collections.synchronizedMap(new HashMap<>());

        static WeavingAdaptor getWeaver(ClassLoader loader, IWeavingContext weavingContext) {
            ExplicitlyInitializedClassLoaderWeavingAdaptor adaptor = null;
            AdaptorKey adaptorKey = new AdaptorKey(loader);
            String loaderClassName = loader.getClass().getName();
            synchronized (weavingAdaptors) {
                checkQ();
                if (loader.equals(myClassLoader)) {
                    adaptor = myClassLoaderAdaptor;
                } else {
                    adaptor = weavingAdaptors.get(adaptorKey);
                }
                if (adaptor == null) {
                    ClassLoaderWeavingAdaptor weavingAdaptor = new ClassLoaderWeavingAdaptor();
                    adaptor = new ExplicitlyInitializedClassLoaderWeavingAdaptor(weavingAdaptor);
                    if (myClassLoaderAdaptor == null && loader.equals(myClassLoader)) {
                        myClassLoaderAdaptor = adaptor;
                    } else {
                        weavingAdaptors.put(adaptorKey, adaptor);
                    }
                }
            }
            return adaptor.getWeavingAdaptor(loader, weavingContext);
        }

        private static final ClassLoader myClassLoader = WeavingAdaptor.class.getClassLoader();

        private static ExplicitlyInitializedClassLoaderWeavingAdaptor myClassLoaderAdaptor;
    }

    static class ExplicitlyInitializedClassLoaderWeavingAdaptor {

        private final ClassLoaderWeavingAdaptor weavingAdaptor;

        private boolean isInitialized;

        public ExplicitlyInitializedClassLoaderWeavingAdaptor(ClassLoaderWeavingAdaptor weavingAdaptor) {
            this.weavingAdaptor = weavingAdaptor;
            this.isInitialized = false;
        }

        private void initialize(ClassLoader loader, IWeavingContext weavingContext) {
            if (!isInitialized) {
                isInitialized = true;
                weavingAdaptor.initialize(loader, weavingContext);
            }
        }

        public ClassLoaderWeavingAdaptor getWeavingAdaptor(ClassLoader loader, IWeavingContext weavingContext) {
            initialize(loader, weavingContext);
            return weavingAdaptor;
        }
    }

    public String getNamespace(ClassLoader loader) {
        ClassLoaderWeavingAdaptor weavingAdaptor = (ClassLoaderWeavingAdaptor) WeaverContainer.getWeaver(loader, weavingContext);
        return weavingAdaptor.getNamespace();
    }

    public boolean generatedClassesExist(ClassLoader loader) {
        return ((ClassLoaderWeavingAdaptor) WeaverContainer.getWeaver(loader, weavingContext)).generatedClassesExistFor(null);
    }

    public void flushGeneratedClasses(ClassLoader loader) {
        ((ClassLoaderWeavingAdaptor) WeaverContainer.getWeaver(loader, weavingContext)).flushGeneratedClasses();
    }

    @Override
    public void prepareForRedefinition(ClassLoader loader, String className) {
        ((ClassLoaderWeavingAdaptor) WeaverContainer.getWeaver(loader, weavingContext)).flushGeneratedClassesFor(className);
    }
}
