package org.aspectj.weaver.loadtime;

import org.aspectj.bridge.AbortException;
import org.aspectj.bridge.Constants;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.util.LangUtil;
import org.aspectj.weaver.IUnwovenClassFile;
import org.aspectj.weaver.Lint;
import org.aspectj.weaver.Lint.Kind;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.World;
import org.aspectj.weaver.bcel.BcelWeakClassLoaderReference;
import org.aspectj.weaver.bcel.BcelWeaver;
import org.aspectj.weaver.bcel.BcelWorld;
import org.aspectj.weaver.bcel.Utility;
import org.aspectj.weaver.loadtime.definition.Definition;
import org.aspectj.weaver.loadtime.definition.DocumentParser;
import org.aspectj.weaver.ltw.LTWWorld;
import org.aspectj.weaver.patterns.PatternParser;
import org.aspectj.weaver.patterns.TypePattern;
import org.aspectj.weaver.tools.GeneratedClassHandler;
import org.aspectj.weaver.tools.Trace;
import org.aspectj.weaver.tools.TraceFactory;
import org.aspectj.weaver.tools.WeavingAdaptor;
import org.aspectj.weaver.tools.cache.WeavedClassCache;

import sun.misc.Unsafe;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;

public class ClassLoaderWeavingAdaptor extends WeavingAdaptor {

    private final static String AOP_XML = Constants.AOP_USER_XML + ";" + Constants.AOP_AJC_XML + ";" + Constants.AOP_OSGI_XML;

    private boolean initialized;

    private List<TypePattern> dumpTypePattern = new ArrayList<>();

    private boolean dumpBefore = false;

    private boolean dumpDirPerClassloader = false;

    private boolean hasExcludes = false;

    private List<TypePattern> excludeTypePattern = new ArrayList<>();

    private List<String> excludeStartsWith = new ArrayList<>();

    private List<String> excludeStarDotDotStar = new ArrayList<>();

    private List<String> excludeExactName = new ArrayList<>();

    private List<String> excludeEndsWith = new ArrayList<>();

    private List<String[]> excludeSpecial = new ArrayList<>();

    private boolean hasIncludes = false;

    private List<TypePattern> includeTypePattern = new ArrayList<>();

    private List<String> includeStartsWith = new ArrayList<>();

    private List<String> includeExactName = new ArrayList<>();

    private boolean includeStar = false;

    private List<TypePattern> aspectExcludeTypePattern = new ArrayList<>();

    private List<String> aspectExcludeStartsWith = new ArrayList<>();

    private List<TypePattern> aspectIncludeTypePattern = new ArrayList<>();

    private List<String> aspectIncludeStartsWith = new ArrayList<>();

    private StringBuffer namespace;

    private IWeavingContext weavingContext;

    private List<ConcreteAspectCodeGen> concreteAspects = new ArrayList<>();

    private static Trace trace = TraceFactory.getTraceFactory().getTrace(ClassLoaderWeavingAdaptor.class);

    public ClassLoaderWeavingAdaptor() {
        super();
        if (trace.isTraceEnabled()) {
            trace.enter("<init>", this);
        }
        if (trace.isTraceEnabled()) {
            trace.exit("<init>");
        }
    }

    @Deprecated
    public ClassLoaderWeavingAdaptor(final ClassLoader deprecatedLoader, final IWeavingContext deprecatedContext) {
        super();
        if (trace.isTraceEnabled()) {
            trace.enter("<init>", this, new Object[]{deprecatedLoader, deprecatedContext});
        }
        if (trace.isTraceEnabled()) {
            trace.exit("<init>");
        }
    }

    class SimpleGeneratedClassHandler implements GeneratedClassHandler {

        private BcelWeakClassLoaderReference loaderRef;

        SimpleGeneratedClassHandler(ClassLoader loader) {
            loaderRef = new BcelWeakClassLoaderReference(loader);
        }

        @Override
        public void acceptClass(String name, byte[] originalBytes, byte[] wovenBytes) {
            try {
                if (shouldDump(name.replace('/', '.'), false)) {
                    dump(name, wovenBytes, false);
                }
            } catch (Throwable throwable) {
                throwable.printStackTrace();
            }
            if (activeProtectionDomain != null) {
                defineClass(loaderRef.getClassLoader(), name, wovenBytes, activeProtectionDomain);
            } else {
                defineClass(loaderRef.getClassLoader(), name, wovenBytes);
            }
        }
    }

    public void initialize(final ClassLoader classLoader, IWeavingContext context) {
        if (initialized) {
            return;
        }
        boolean success = true;
        this.weavingContext = context;
        if (weavingContext == null) {
            weavingContext = new DefaultWeavingContext(classLoader);
        }
        createMessageHandler();
        this.generatedClassHandler = new SimpleGeneratedClassHandler(classLoader);
        List<Definition> definitions = weavingContext.getDefinitions(classLoader, this);
        if (definitions.isEmpty()) {
            disable();
            if (trace.isTraceEnabled()) {
                trace.exit("initialize", definitions);
            }
            return;
        }
        bcelWorld = new LTWWorld(classLoader, weavingContext, getMessageHandler(), null);
        weaver = new BcelWeaver(bcelWorld);
        success = registerDefinitions(weaver, classLoader, definitions);
        if (success) {
            weaver.prepareForWeave();
            enable();
            success = weaveAndDefineConceteAspects();
        }
        if (success) {
            enable();
        } else {
            disable();
            bcelWorld = null;
            weaver = null;
        }
        if (WeavedClassCache.isEnabled()) {
            initializeCache(classLoader, getAspectClassNames(definitions), generatedClassHandler, getMessageHandler());
        }
        initialized = true;
        if (trace.isTraceEnabled()) {
            trace.exit("initialize", isEnabled());
        }
    }

    List<String> getAspectClassNames(List<Definition> definitions) {
        List<String> aspects = new LinkedList<>();
        for (Definition def : definitions) {
            List<String> defAspects = def.getAspectClassNames();
            if (defAspects != null) {
                aspects.addAll(defAspects);
            }
        }
        return aspects;
    }

    List<Definition> parseDefinitions(final ClassLoader loader) {
        if (trace.isTraceEnabled()) {
            trace.enter("parseDefinitions", this);
        }
        List<Definition> definitions = new ArrayList<>();
        try {
            info("register classloader " + getClassLoaderName(loader));
            if (loader.equals(ClassLoader.getSystemClassLoader())) {
                String file = System.getProperty("aj5.def", null);
                if (file != null) {
                    info("using (-Daj5.def) " + file);
                    definitions.add(DocumentParser.parse((new File(file)).toURI().toURL()));
                }
            }
            String resourcePath = System.getProperty("org.aspectj.weaver.loadtime.configuration", AOP_XML);
            if (trace.isTraceEnabled()) {
                trace.event("parseDefinitions", this, resourcePath);
            }
            StringTokenizer st = new StringTokenizer(resourcePath, ";");
            while (st.hasMoreTokens()) {
                String nextDefinition = st.nextToken();
                if (nextDefinition.startsWith("file:")) {
                    try {
                        String fpath = new URL(nextDefinition).getFile();
                        File configFile = new File(fpath);
                        if (!configFile.exists()) {
                            warn("configuration does not exist: " + nextDefinition);
                        } else {
                            definitions.add(DocumentParser.parse(configFile.toURI().toURL()));
                        }
                    } catch (MalformedURLException mue) {
                        error("malformed definition url: " + nextDefinition);
                    }
                } else {
                    Enumeration<URL> xmls = weavingContext.getResources(nextDefinition);
                    Set<URL> seenBefore = new HashSet<>();
                    while (xmls.hasMoreElements()) {
                        URL xml = xmls.nextElement();
                        if (trace.isTraceEnabled()) {
                            trace.event("parseDefinitions", this, xml);
                        }
                        if (!seenBefore.contains(xml)) {
                            info("using configuration " + weavingContext.getFile(xml));
                            definitions.add(DocumentParser.parse(xml));
                            seenBefore.add(xml);
                        } else {
                            debug("ignoring duplicate definition: " + xml);
                        }
                    }
                }
            }
            if (definitions.isEmpty()) {
                info("no configuration found. Disabling weaver for class loader " + getClassLoaderName(loader));
            }
        } catch (Exception e) {
            definitions.clear();
            warn("parse definitions failed", e);
        }
        if (trace.isTraceEnabled()) {
            trace.exit("parseDefinitions", definitions);
        }
        return definitions;
    }

    private boolean registerDefinitions(final BcelWeaver weaver, final ClassLoader loader, List<Definition> definitions) {
        if (trace.isTraceEnabled()) {
            trace.enter("registerDefinitions", this, definitions);
        }
        boolean success = true;
        try {
            registerOptions(weaver, loader, definitions);
            registerAspectExclude(weaver, loader, definitions);
            registerAspectInclude(weaver, loader, definitions);
            success = registerAspects(weaver, loader, definitions);
            registerIncludeExclude(weaver, loader, definitions);
            registerDump(weaver, loader, definitions);
        } catch (Exception ex) {
            trace.error("register definition failed", ex);
            success = false;
            warn("register definition failed", (ex instanceof AbortException) ? null : ex);
        }
        if (trace.isTraceEnabled()) {
            trace.exit("registerDefinitions", success);
        }
        return success;
    }

    private String getClassLoaderName(ClassLoader loader) {
        return weavingContext.getClassLoaderName();
    }

    private void registerOptions(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        StringBuilder allOptions = new StringBuilder();
        for (Definition definition : definitions) {
            allOptions.append(definition.getWeaverOptions()).append(' ');
        }
        Options.WeaverOption weaverOption = Options.parse(allOptions.toString(), loader, getMessageHandler());
        World world = weaver.getWorld();
        setMessageHandler(weaverOption.messageHandler);
        world.setXlazyTjp(weaverOption.lazyTjp);
        world.setXHasMemberSupportEnabled(weaverOption.hasMember);
        world.setTiming(weaverOption.timers, true);
        world.setOptionalJoinpoints(weaverOption.optionalJoinpoints);
        world.setPinpointMode(weaverOption.pinpoint);
        weaver.setReweavableMode(weaverOption.notReWeavable);
        if (weaverOption.loadersToSkip != null && weaverOption.loadersToSkip.length() > 0) {
            Aj.loadersToSkip = LangUtil.anySplit(weaverOption.loadersToSkip, ",");
        }
        if (Aj.loadersToSkip != null) {
            MessageUtil.info(world.getMessageHandler(), "no longer creating weavers for these classloaders: " + Aj.loadersToSkip);
        }
        world.performExtraConfiguration(weaverOption.xSet);
        world.setXnoInline(weaverOption.noInline);
        world.setBehaveInJava5Way(true);
        world.setAddSerialVerUID(weaverOption.addSerialVersionUID);
        bcelWorld.getLint().loadDefaultProperties();
        bcelWorld.getLint().adviceDidNotMatch.setKind(null);
        if (weaverOption.lintFile != null) {
            InputStream resource = null;
            try {
                resource = loader.getResourceAsStream(weaverOption.lintFile);
                Exception failure = null;
                if (resource != null) {
                    try {
                        Properties properties = new Properties();
                        properties.load(resource);
                        world.getLint().setFromProperties(properties);
                    } catch (IOException e) {
                        failure = e;
                    }
                }
                if (failure != null || resource == null) {
                    warn("Cannot access resource for -Xlintfile:" + weaverOption.lintFile, failure);
                }
            } finally {
                try {
                    resource.close();
                } catch (Throwable t) {
                }
            }
        }
        if (weaverOption.lint != null) {
            if (weaverOption.lint.equals("default")) {
                bcelWorld.getLint().loadDefaultProperties();
            } else {
                bcelWorld.getLint().setAll(weaverOption.lint);
                if (weaverOption.lint.equals("ignore")) {
                    bcelWorld.setAllLintIgnored();
                }
            }
        }
    }

    private void registerAspectExclude(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        String fastMatchInfo = null;
        for (Definition definition : definitions) {
            for (String exclude : definition.getAspectExcludePatterns()) {
                TypePattern excludePattern = new PatternParser(exclude).parseTypePattern();
                aspectExcludeTypePattern.add(excludePattern);
                fastMatchInfo = looksLikeStartsWith(exclude);
                if (fastMatchInfo != null) {
                    aspectExcludeStartsWith.add(fastMatchInfo);
                }
            }
        }
    }

    private void registerAspectInclude(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        String fastMatchInfo = null;
        for (Definition definition : definitions) {
            for (String include : definition.getAspectIncludePatterns()) {
                TypePattern includePattern = new PatternParser(include).parseTypePattern();
                aspectIncludeTypePattern.add(includePattern);
                fastMatchInfo = looksLikeStartsWith(include);
                if (fastMatchInfo != null) {
                    aspectIncludeStartsWith.add(fastMatchInfo);
                }
            }
        }
    }

    protected void lint(String name, String[] infos) {
        Lint lint = bcelWorld.getLint();
        Kind kind = lint.getLintKind(name);
        kind.signal(infos, null, null);
    }

    @Override
    public String getContextId() {
        return weavingContext.getId();
    }

    private boolean registerAspects(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        if (trace.isTraceEnabled()) {
            trace.enter("registerAspects", this, new Object[]{weaver, loader, definitions});
        }
        boolean success = true;
        for (Definition definition : definitions) {
            for (String aspectClassName : definition.getAspectClassNames()) {
                if (acceptAspect(aspectClassName)) {
                    info("register aspect " + aspectClassName);
                    String requiredType = definition.getAspectRequires(aspectClassName);
                    if (requiredType != null) {
                        ((BcelWorld) weaver.getWorld()).addAspectRequires(aspectClassName, requiredType);
                    }
                    String definedScope = definition.getScopeForAspect(aspectClassName);
                    if (definedScope != null) {
                        ((BcelWorld) weaver.getWorld()).addScopedAspect(aspectClassName, definedScope);
                    }
                    weaver.addLibraryAspect(aspectClassName);
                    if (namespace == null) {
                        namespace = new StringBuffer(aspectClassName);
                    } else {
                        namespace = namespace.append(";").append(aspectClassName);
                    }
                } else {
                    lint("aspectExcludedByConfiguration", new String[]{aspectClassName, getClassLoaderName(loader)});
                }
            }
        }
        for (Definition definition : definitions) {
            for (Definition.ConcreteAspect concreteAspect : definition.getConcreteAspects()) {
                if (acceptAspect(concreteAspect.name)) {
                    info("define aspect " + concreteAspect.name);
                    ConcreteAspectCodeGen gen = new ConcreteAspectCodeGen(concreteAspect, weaver.getWorld());
                    if (!gen.validate()) {
                        error("Concrete-aspect '" + concreteAspect.name + "' could not be registered");
                        success = false;
                        break;
                    }
                    ((BcelWorld) weaver.getWorld()).addSourceObjectType(Utility.makeJavaClass(concreteAspect.name, gen.getBytes()), true);
                    concreteAspects.add(gen);
                    weaver.addLibraryAspect(concreteAspect.name);
                    if (namespace == null) {
                        namespace = new StringBuffer(concreteAspect.name);
                    } else {
                        namespace = namespace.append(";" + concreteAspect.name);
                    }
                }
            }
        }
        if (!success) {
            warn("failure(s) registering aspects. Disabling weaver for class loader " + getClassLoaderName(loader));
        } else if (namespace == null) {
            success = false;
            info("no aspects registered. Disabling weaver for class loader " + getClassLoaderName(loader));
        }
        if (trace.isTraceEnabled()) {
            trace.exit("registerAspects", success);
        }
        return success;
    }

    private boolean weaveAndDefineConceteAspects() {
        if (trace.isTraceEnabled()) {
            trace.enter("weaveAndDefineConceteAspects", this, concreteAspects);
        }
        boolean success = true;
        for (ConcreteAspectCodeGen gen : concreteAspects) {
            String name = gen.getClassName();
            byte[] bytes = gen.getBytes();
            try {
                byte[] newBytes = weaveClass(name, bytes, true);
                this.generatedClassHandler.acceptClass(name, bytes, newBytes);
            } catch (IOException ex) {
                trace.error("weaveAndDefineConceteAspects", ex);
                error("exception weaving aspect '" + name + "'", ex);
            }
        }
        if (trace.isTraceEnabled()) {
            trace.exit("weaveAndDefineConceteAspects", success);
        }
        return success;
    }

    private void registerIncludeExclude(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        String fastMatchInfo = null;
        for (Definition definition : definitions) {
            for (String value : definition.getIncludePatterns()) {
                hasIncludes = true;
                String include = value;
                fastMatchInfo = looksLikeStartsWith(include);
                if (fastMatchInfo != null) {
                    includeStartsWith.add(fastMatchInfo);
                } else if (include.equals("*")) {
                    includeStar = true;
                } else if ((fastMatchInfo = looksLikeExactName(include)) != null) {
                    includeExactName.add(fastMatchInfo);
                } else {
                    TypePattern includePattern = new PatternParser(include).parseTypePattern();
                    includeTypePattern.add(includePattern);
                }
            }
            for (String s : definition.getExcludePatterns()) {
                hasExcludes = true;
                String exclude = s;
                fastMatchInfo = looksLikeStartsWith(exclude);
                if (fastMatchInfo != null) {
                    excludeStartsWith.add(fastMatchInfo);
                } else if ((fastMatchInfo = looksLikeStarDotDotStarExclude(exclude)) != null) {
                    excludeStarDotDotStar.add(fastMatchInfo);
                } else if ((fastMatchInfo = looksLikeExactName(exclude)) != null) {
                    excludeExactName.add(exclude);
                } else if ((fastMatchInfo = looksLikeEndsWith(exclude)) != null) {
                    excludeEndsWith.add(fastMatchInfo);
                } else if (exclude.equals("org.codehaus.groovy..* && !org.codehaus.groovy.grails.web.servlet.mvc.SimpleGrailsController*")) {
                    excludeSpecial.add(new String[]{"org.codehaus.groovy.", "org.codehaus.groovy.grails.web.servlet.mvc.SimpleGrailsController"});
                } else {
                    TypePattern excludePattern = new PatternParser(exclude).parseTypePattern();
                    excludeTypePattern.add(excludePattern);
                }
            }
        }
    }

    private String looksLikeStarDotDotStarExclude(String typePattern) {
        if (!typePattern.startsWith("*..*")) {
            return null;
        }
        if (!typePattern.endsWith("*")) {
            return null;
        }
        String subPattern = typePattern.substring(4, typePattern.length() - 1);
        if (hasStarDot(subPattern, 0)) {
            return null;
        }
        return subPattern.replace('$', '.');
    }

    private String looksLikeExactName(String typePattern) {
        if (hasSpaceAnnotationPlus(typePattern, 0) || typePattern.contains("*")) {
            return null;
        }
        return typePattern.replace('$', '.');
    }

    private String looksLikeEndsWith(String typePattern) {
        if (typePattern.charAt(0) != '*') {
            return null;
        }
        if (hasSpaceAnnotationPlus(typePattern, 1) || hasStarDot(typePattern, 1)) {
            return null;
        }
        return typePattern.substring(1).replace('$', '.');
    }

    private boolean hasSpaceAnnotationPlus(String string, int pos) {
        for (int i = pos, max = string.length(); i < max; i++) {
            char ch = string.charAt(i);
            if (ch == ' ' || ch == '@' || ch == '+') {
                return true;
            }
        }
        return false;
    }

    private boolean hasStarDot(String string, int pos) {
        for (int i = pos, max = string.length(); i < max; i++) {
            char ch = string.charAt(i);
            if (ch == '*' || ch == '.') {
                return true;
            }
        }
        return false;
    }

    private String looksLikeStartsWith(String typePattern) {
        if (hasSpaceAnnotationPlus(typePattern, 0) || typePattern.charAt(typePattern.length() - 1) != '*') {
            return null;
        }
        int length = typePattern.length();
        if (typePattern.endsWith("..*") && length > 3) {
            if (typePattern.indexOf("..") == length - 3 && typePattern.indexOf('*') == length - 1) {
                return typePattern.substring(0, length - 2).replace('$', '.');
            }
        }
        return null;
    }

    private void registerDump(final BcelWeaver weaver, final ClassLoader loader, final List<Definition> definitions) {
        for (Definition definition : definitions) {
            for (String dump : definition.getDumpPatterns()) {
                TypePattern pattern = new PatternParser(dump).parseTypePattern();
                dumpTypePattern.add(pattern);
            }
            if (definition.shouldDumpBefore()) {
                dumpBefore = true;
            }
            if (definition.createDumpDirPerClassloader()) {
                dumpDirPerClassloader = true;
            }
        }
    }

    @Override
    protected boolean accept(String className, byte[] bytes) {
        if (!hasExcludes && !hasIncludes) {
            return true;
        }
        String fastClassName = className.replace('/', '.');
        for (String excludeStartsWithString : excludeStartsWith) {
            if (fastClassName.startsWith(excludeStartsWithString)) {
                return false;
            }
        }
        if (!excludeStarDotDotStar.isEmpty()) {
            for (String namePiece : excludeStarDotDotStar) {
                int index = fastClassName.lastIndexOf('.');
                if (fastClassName.indexOf(namePiece, index + 1) != -1) {
                    return false;
                }
            }
        }
        fastClassName = fastClassName.replace('$', '.');
        if (!excludeEndsWith.isEmpty()) {
            for (String lastPiece : excludeEndsWith) {
                if (fastClassName.endsWith(lastPiece)) {
                    return false;
                }
            }
        }
        if (!excludeExactName.isEmpty()) {
            for (String name : excludeExactName) {
                if (fastClassName.equals(name)) {
                    return false;
                }
            }
        }
        if (!excludeSpecial.isEmpty()) {
            for (String[] entry : excludeSpecial) {
                String excludeThese = entry[0];
                String exceptThese = entry[1];
                if (fastClassName.startsWith(excludeThese) && !fastClassName.startsWith(exceptThese)) {
                    return false;
                }
            }
        }
        boolean didSomeIncludeMatching = false;
        if (excludeTypePattern.isEmpty()) {
            if (includeStar) {
                return true;
            }
            if (!includeExactName.isEmpty()) {
                didSomeIncludeMatching = true;
                for (String exactname : includeExactName) {
                    if (fastClassName.equals(exactname)) {
                        return true;
                    }
                }
            }
            boolean fastAccept = false;
            for (String s : includeStartsWith) {
                didSomeIncludeMatching = true;
                fastAccept = fastClassName.startsWith(s);
                if (fastAccept) {
                    return true;
                }
            }
            if (includeTypePattern.isEmpty()) {
                return !didSomeIncludeMatching;
            }
        }
        boolean accept;
        try {
            ensureDelegateInitialized(className, bytes);
            ResolvedType classInfo = delegateForCurrentClass.getResolvedTypeX();
            for (TypePattern typePattern : excludeTypePattern) {
                if (typePattern.matchesStatically(classInfo)) {
                    return false;
                }
            }
            if (includeStar) {
                return true;
            }
            if (!includeExactName.isEmpty()) {
                didSomeIncludeMatching = true;
                for (String exactname : includeExactName) {
                    if (fastClassName.equals(exactname)) {
                        return true;
                    }
                }
            }
            for (String s : includeStartsWith) {
                didSomeIncludeMatching = true;
                boolean fastaccept = fastClassName.startsWith(s);
                if (fastaccept) {
                    return true;
                }
            }
            accept = !didSomeIncludeMatching;
            for (TypePattern typePattern : includeTypePattern) {
                accept = typePattern.matchesStatically(classInfo);
                if (accept) {
                    break;
                }
            }
        } finally {
            this.bcelWorld.demote();
        }
        return accept;
    }

    private boolean acceptAspect(String aspectClassName) {
        if (aspectExcludeTypePattern.isEmpty() && aspectIncludeTypePattern.isEmpty()) {
            return true;
        }
        String fastClassName = aspectClassName.replace('/', '.').replace('.', '$');
        for (String value : aspectExcludeStartsWith) {
            if (fastClassName.startsWith(value)) {
                return false;
            }
        }
        for (String s : aspectIncludeStartsWith) {
            if (fastClassName.startsWith(s)) {
                return true;
            }
        }
        ResolvedType classInfo = weaver.getWorld().resolve(UnresolvedType.forName(aspectClassName), true);
        for (TypePattern typePattern : aspectExcludeTypePattern) {
            if (typePattern.matchesStatically(classInfo)) {
                return false;
            }
        }
        boolean accept = true;
        for (TypePattern typePattern : aspectIncludeTypePattern) {
            accept = typePattern.matchesStatically(classInfo);
            if (accept) {
                break;
            }
        }
        return accept;
    }

    @Override
    protected boolean shouldDump(String className, boolean before) {
        if (before && !dumpBefore) {
            return false;
        }
        if (dumpTypePattern.isEmpty()) {
            return false;
        }
        ResolvedType classInfo = weaver.getWorld().resolve(UnresolvedType.forName(className), true);
        for (TypePattern typePattern : dumpTypePattern) {
            if (typePattern.matchesStatically(classInfo)) {
                return true;
            }
        }
        return false;
    }

    @Override
    protected String getDumpDir() {
        if (dumpDirPerClassloader) {
            StringBuilder dir = new StringBuilder();
            dir.append("_ajdump").append(File.separator).append(weavingContext.getId());
            return dir.toString();
        } else {
            return super.getDumpDir();
        }
    }

    public String getNamespace() {
        if (namespace == null) {
            return "";
        } else {
            return new String(namespace);
        }
    }

    public boolean generatedClassesExistFor(String className) {
        if (className == null) {
            return !generatedClasses.isEmpty();
        } else {
            return generatedClasses.containsKey(className);
        }
    }

    public void flushGeneratedClasses() {
        generatedClasses = new HashMap<>();
    }

    public void flushGeneratedClassesFor(String className) {
        try {
            String dottedClassName = className.replace('/', '.');
            String dottedClassNameDollar = dottedClassName + "$";
            Iterator<Map.Entry<String, IUnwovenClassFile>> iter = generatedClasses.entrySet().iterator();
            while (iter.hasNext()) {
                Entry<String, IUnwovenClassFile> next = iter.next();
                String existingGeneratedName = next.getKey();
                if (existingGeneratedName.equals(dottedClassName) || existingGeneratedName.startsWith(dottedClassNameDollar)) {
                    iter.remove();
                }
            }
        } catch (Throwable t) {
            new RuntimeException("Unexpected problem tidying up generated classes for " + className, t).printStackTrace();
        }
    }

    private Unsafe unsafe;

    private Unsafe getUnsafe() throws NoSuchFieldException, IllegalAccessException {
        if (unsafe == null) {
            Field theUnsafeField = Unsafe.class.getDeclaredField("theUnsafe");
            theUnsafeField.setAccessible(true);
            return (Unsafe) theUnsafeField.get(null);
        }
        return unsafe;
    }

    private static Method bindTo_Method, invokeWithArguments_Method = null;

    private static Object defineClassMethodHandle = null;

    private static Boolean initializedForJava11 = false;

    private static synchronized void initializeForJava11() {
        if (initializedForJava11)
            return;
        try {
            Class<?> methodType_Class = Class.forName("java.lang.invoke.MethodType");
            Method methodTypeMethodOnMethodTypeClass = methodType_Class.getDeclaredMethod("methodType", Class.class, Class[].class);
            methodTypeMethodOnMethodTypeClass.setAccessible(true);
            Object defineClassMethodType = methodTypeMethodOnMethodTypeClass.invoke(null, Class.class, new Class[]{String.class, byte[].class, int.class, int.class, ProtectionDomain.class});
            Class<?> methodHandles_Class = Class.forName("java.lang.invoke.MethodHandles");
            Method lookupMethodOnMethodHandlesClass = methodHandles_Class.getDeclaredMethod("lookup");
            lookupMethodOnMethodHandlesClass.setAccessible(true);
            Object methodHandlesLookup = lookupMethodOnMethodHandlesClass.invoke(null);
            Class<?> methodHandlesLookup_Class = Class.forName("java.lang.invoke.MethodHandles$Lookup");
            Method privateLookupMethodOnMethodHandlesClass = methodHandles_Class.getDeclaredMethod("privateLookupIn", Class.class, methodHandlesLookup_Class);
            privateLookupMethodOnMethodHandlesClass.setAccessible(true);
            Object lookup = privateLookupMethodOnMethodHandlesClass.invoke(null, ClassLoader.class, methodHandlesLookup);
            Method findVirtual_Method = methodHandlesLookup_Class.getDeclaredMethod("findVirtual", Class.class, String.class, methodType_Class);
            findVirtual_Method.setAccessible(true);
            defineClassMethodHandle = findVirtual_Method.invoke(lookup, ClassLoader.class, "defineClass", defineClassMethodType);
            Class<?> methodHandle_Class = Class.forName("java.lang.invoke.MethodHandle");
            bindTo_Method = methodHandle_Class.getDeclaredMethod("bindTo", Object.class);
            invokeWithArguments_Method = methodHandle_Class.getDeclaredMethod("invokeWithArguments", Object[].class);
            initializedForJava11 = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void defineClass(ClassLoader loader, String name, byte[] bytes, ProtectionDomain protectionDomain) {
        if (trace.isTraceEnabled()) {
            trace.enter("defineClass", this, new Object[]{loader, name, bytes});
        }
        Object clazz = null;
        debug("generating class '" + name + "'");
        if (LangUtil.is11VMOrGreater()) {
            try {
                if (!initializedForJava11) {
                    initializeForJava11();
                }
                Object o = bindTo_Method.invoke(defineClassMethodHandle, loader);
                clazz = invokeWithArguments_Method.invoke(o, new Object[]{new Object[]{name, bytes, 0, bytes.length, protectionDomain}});
            } catch (Throwable t) {
                t.printStackTrace(System.err);
                warn("define generated class failed", t);
            }
        } else {
            try {
                if (defineClassMethod == null) {
                    synchronized (lock) {
                        getUnsafe();
                        defineClassMethod = Unsafe.class.getDeclaredMethod("defineClass", String.class, byte[].class, Integer.TYPE, Integer.TYPE, ClassLoader.class, ProtectionDomain.class);
                    }
                }
                defineClassMethod.setAccessible(true);
                clazz = defineClassMethod.invoke(getUnsafe(), name, bytes, 0, bytes.length, loader, protectionDomain);
            } catch (LinkageError le) {
                le.printStackTrace();
            } catch (Exception e) {
                e.printStackTrace(System.err);
                warn("define generated class failed", e);
            }
        }
        if (trace.isTraceEnabled()) {
            trace.exit("defineClass", clazz);
        }
    }

    static Method defineClassMethod;

    private static String lock = "lock";

    private void defineClass(ClassLoader loader, String name, byte[] bytes) {
        defineClass(loader, name, bytes, null);
    }
}
