package org.aspectj.weaver.loadtime;

import org.aspectj.apache.bcel.Constants;
import org.aspectj.apache.bcel.classfile.ConstantPool;
import org.aspectj.apache.bcel.classfile.JavaClass;
import org.aspectj.apache.bcel.classfile.annotation.AnnotationGen;
import org.aspectj.apache.bcel.classfile.annotation.ClassElementValue;
import org.aspectj.apache.bcel.classfile.annotation.ElementValue;
import org.aspectj.apache.bcel.classfile.annotation.NameValuePair;
import org.aspectj.apache.bcel.classfile.annotation.SimpleElementValue;
import org.aspectj.apache.bcel.generic.FieldGen;
import org.aspectj.apache.bcel.generic.InstructionConstants;
import org.aspectj.apache.bcel.generic.InstructionFactory;
import org.aspectj.apache.bcel.generic.InstructionHandle;
import org.aspectj.apache.bcel.generic.InstructionList;
import org.aspectj.apache.bcel.generic.LocalVariableTag;
import org.aspectj.apache.bcel.generic.ObjectType;
import org.aspectj.apache.bcel.generic.Type;
import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.Message;
import org.aspectj.weaver.AjAttribute;
import org.aspectj.weaver.AnnotationAJ;
import org.aspectj.weaver.GeneratedReferenceTypeDelegate;
import org.aspectj.weaver.ReferenceType;
import org.aspectj.weaver.ResolvedMember;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.World;
import org.aspectj.weaver.bcel.BcelAnnotation;
import org.aspectj.weaver.bcel.BcelPerClauseAspectAdder;
import org.aspectj.weaver.bcel.BcelWorld;
import org.aspectj.weaver.bcel.LazyClassGen;
import org.aspectj.weaver.bcel.LazyMethodGen;
import org.aspectj.weaver.loadtime.definition.Definition;
import org.aspectj.weaver.loadtime.definition.Definition.AdviceKind;
import org.aspectj.weaver.loadtime.definition.Definition.DeclareAnnotationKind;
import org.aspectj.weaver.loadtime.definition.Definition.PointcutAndAdvice;
import org.aspectj.weaver.patterns.BasicTokenSource;
import org.aspectj.weaver.patterns.DeclareAnnotation;
import org.aspectj.weaver.patterns.ISignaturePattern;
import org.aspectj.weaver.patterns.ITokenSource;
import org.aspectj.weaver.patterns.PatternParser;
import org.aspectj.weaver.patterns.PerClause;
import org.aspectj.weaver.patterns.TypePattern;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.aspectj.apache.bcel.generic.Type.NO_ARGS;

public class ConcreteAspectCodeGen {

    private final static String[] EMPTY_STRINGS = new String[0];

    private final Definition.ConcreteAspect concreteAspect;

    private final World world;

    private boolean isValid = false;

    private ResolvedType parent;

    private PerClause perclause;

    private byte[] bytes;

    ConcreteAspectCodeGen(Definition.ConcreteAspect concreteAspect, World world) {
        this.concreteAspect = concreteAspect;
        this.world = world;
    }

    public boolean validate() {
        if (!(world instanceof BcelWorld)) {
            reportError("Internal error: world must be of type BcelWorld");
            return false;
        }
        ResolvedType current = world.lookupBySignature(UnresolvedType.forName(concreteAspect.name).getSignature());
        if (current != null && !current.isMissing()) {
            reportError("Attempt to concretize but chosen aspect name already defined: " + stringify());
            return false;
        }
        if (concreteAspect.pointcutsAndAdvice.size() != 0) {
            isValid = true;
            return true;
        }
        if (concreteAspect.declareAnnotations.size() != 0) {
            isValid = true;
            return true;
        }
        if (concreteAspect.extend == null && concreteAspect.precedence != null) {
            if (concreteAspect.pointcuts.isEmpty()) {
                isValid = true;
                parent = null;
                return true;
            } else {
                reportError("Attempt to use nested pointcuts without extends clause: " + stringify());
                return false;
            }
        }
        String parentAspectName = concreteAspect.extend;
        if (parentAspectName.contains("<")) {
            parent = world.resolve(UnresolvedType.forName(parentAspectName), true);
            if (parent.isMissing()) {
                reportError("Unable to resolve type reference: " + stringify());
                return false;
            }
            if (parent.isParameterizedType()) {
                UnresolvedType[] typeParameters = parent.getTypeParameters();
                for (UnresolvedType typeParameter : typeParameters) {
                    if (typeParameter instanceof ResolvedType && ((ResolvedType) typeParameter).isMissing()) {
                        reportError("Unablet to resolve type parameter '" + typeParameter.getName() + "' from " + stringify());
                        return false;
                    }
                }
            }
        } else {
            parent = world.resolve(concreteAspect.extend, true);
        }
        if (parent.isMissing()) {
            String fixedName = concreteAspect.extend;
            int hasDot = fixedName.lastIndexOf('.');
            while (hasDot > 0) {
                char[] fixedNameChars = fixedName.toCharArray();
                fixedNameChars[hasDot] = '$';
                fixedName = new String(fixedNameChars);
                hasDot = fixedName.lastIndexOf('.');
                parent = world.resolve(UnresolvedType.forName(fixedName), true);
                if (!parent.isMissing()) {
                    break;
                }
            }
        }
        if (parent.isMissing()) {
            reportError("Cannot find parent aspect for: " + stringify());
            return false;
        }
        if (!(parent.isAbstract() || parent.equals(ResolvedType.OBJECT))) {
            reportError("Attempt to concretize a non-abstract aspect: " + stringify());
            return false;
        }
        if (!(parent.isAspect() || parent.equals(ResolvedType.OBJECT))) {
            reportError("Attempt to concretize a non aspect: " + stringify());
            return false;
        }
        List<String> elligibleAbstractions = new ArrayList<>();
        Collection<ResolvedMember> abstractMethods = getOutstandingAbstractMethods(parent);
        for (ResolvedMember method : abstractMethods) {
            if ("()V".equals(method.getSignature())) {
                String n = method.getName();
                if (n.startsWith("ajc$pointcut")) {
                    n = n.substring(14);
                    n = n.substring(0, n.indexOf("$"));
                    elligibleAbstractions.add(n);
                } else if (hasPointcutAnnotation(method)) {
                    elligibleAbstractions.add(method.getName());
                } else {
                    reportError("Abstract method '" + method.toString() + "' cannot be concretized in XML: " + stringify());
                    return false;
                }
            } else {
                if (method.getName().startsWith("ajc$pointcut") || hasPointcutAnnotation(method)) {
                    reportError("Abstract method '" + method.toString() + "' cannot be concretized as a pointcut (illegal signature, must have no arguments, must return void): " + stringify());
                    return false;
                } else {
                    reportError("Abstract method '" + method.toString() + "' cannot be concretized in XML: " + stringify());
                    return false;
                }
            }
        }
        List<String> pointcutNames = new ArrayList<>();
        for (Definition.Pointcut abstractPc : concreteAspect.pointcuts) {
            pointcutNames.add(abstractPc.name);
        }
        for (String elligiblePc : elligibleAbstractions) {
            if (!pointcutNames.contains(elligiblePc)) {
                reportError("Abstract pointcut '" + elligiblePc + "' not configured: " + stringify());
                return false;
            }
        }
        if (concreteAspect.perclause != null) {
            String perclauseString = concreteAspect.perclause;
            if (perclauseString.startsWith("persingleton")) {
            } else if (perclauseString.startsWith("percflow")) {
            } else if (perclauseString.startsWith("pertypewithin")) {
            } else if (perclauseString.startsWith("perthis")) {
            } else if (perclauseString.startsWith("pertarget")) {
            } else if (perclauseString.startsWith("percflowbelow")) {
            } else {
                reportError("Unrecognized per clause specified " + stringify());
                return false;
            }
        }
        isValid = true;
        return isValid;
    }

    private Collection<ResolvedMember> getOutstandingAbstractMethods(ResolvedType type) {
        Map<String, ResolvedMember> collector = new HashMap<>();
        getOutstandingAbstractMethodsHelper(type, collector);
        return collector.values();
    }

    private void getOutstandingAbstractMethodsHelper(ResolvedType type, Map<String, ResolvedMember> collector) {
        if (type == null) {
            return;
        }
        if (!type.equals(ResolvedType.OBJECT)) {
            if (type.getSuperclass() != null) {
                getOutstandingAbstractMethodsHelper(type.getSuperclass(), collector);
            }
        }
        ResolvedMember[] rms = type.getDeclaredMethods();
        if (rms != null) {
            for (ResolvedMember member : rms) {
                String key = member.getName() + member.getSignature();
                if (member.isAbstract()) {
                    collector.put(key, member);
                } else {
                    collector.remove(key);
                }
            }
        }
    }

    private String stringify() {
        StringBuilder sb = new StringBuilder("<concrete-aspect name='");
        sb.append(concreteAspect.name);
        sb.append("' extends='");
        sb.append(concreteAspect.extend);
        sb.append("' perclause='");
        sb.append(concreteAspect.perclause);
        sb.append("'/> in aop.xml");
        return sb.toString();
    }

    private boolean hasPointcutAnnotation(ResolvedMember member) {
        AnnotationAJ[] as = member.getAnnotations();
        if (as == null || as.length == 0) {
            return false;
        }
        for (AnnotationAJ a : as) {
            if (a.getTypeSignature().equals("Lorg/aspectj/lang/annotation/Pointcut;")) {
                return true;
            }
        }
        return false;
    }

    public String getClassName() {
        return concreteAspect.name;
    }

    public byte[] getBytes() {
        if (!isValid) {
            throw new RuntimeException("Must validate first");
        }
        if (bytes != null) {
            return bytes;
        }
        PerClause.Kind perclauseKind = PerClause.SINGLETON;
        PerClause parentPerClause = (parent != null ? parent.getPerClause() : null);
        if (parentPerClause != null) {
            perclauseKind = parentPerClause.getKind();
        }
        String perclauseString = null;
        if (concreteAspect.perclause != null) {
            perclauseString = concreteAspect.perclause;
            if (perclauseString.startsWith("persingleton")) {
                perclauseKind = PerClause.SINGLETON;
            } else if (perclauseString.startsWith("percflow")) {
                perclauseKind = PerClause.PERCFLOW;
            } else if (perclauseString.startsWith("pertypewithin")) {
                perclauseKind = PerClause.PERTYPEWITHIN;
            } else if (perclauseString.startsWith("perthis")) {
                perclauseKind = PerClause.PEROBJECT;
            } else if (perclauseString.startsWith("pertarget")) {
                perclauseKind = PerClause.PEROBJECT;
            } else if (perclauseString.startsWith("percflowbelow")) {
                perclauseKind = PerClause.PERCFLOW;
            }
        }
        String parentName = "java/lang/Object";
        if (parent != null) {
            if (parent.isParameterizedType()) {
                parentName = parent.getGenericType().getName().replace('.', '/');
            } else {
                parentName = parent.getName().replace('.', '/');
            }
        }
        LazyClassGen cg = new LazyClassGen(concreteAspect.name.replace('.', '/'), parentName, null, Modifier.PUBLIC + Constants.ACC_SUPER, EMPTY_STRINGS, world);
        if (parent != null && parent.isParameterizedType()) {
            cg.setSuperClass(parent);
        }
        if (perclauseString == null) {
            AnnotationGen ag = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/Aspect"), Collections.<NameValuePair>emptyList(), true, cg.getConstantPool());
            cg.addAnnotation(ag);
        } else {
            List<NameValuePair> elems = new ArrayList<>();
            elems.add(new NameValuePair("value", new SimpleElementValue(ElementValue.STRING, cg.getConstantPool(), perclauseString), cg.getConstantPool()));
            AnnotationGen ag = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/Aspect"), elems, true, cg.getConstantPool());
            cg.addAnnotation(ag);
        }
        if (concreteAspect.precedence != null) {
            SimpleElementValue svg = new SimpleElementValue(ElementValue.STRING, cg.getConstantPool(), concreteAspect.precedence);
            List<NameValuePair> elems = new ArrayList<>();
            elems.add(new NameValuePair("value", svg, cg.getConstantPool()));
            AnnotationGen agprec = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/DeclarePrecedence"), elems, true, cg.getConstantPool());
            cg.addAnnotation(agprec);
        }
        LazyMethodGen init = new LazyMethodGen(Modifier.PUBLIC, Type.VOID, "<init>", NO_ARGS, EMPTY_STRINGS, cg);
        InstructionList cbody = init.getBody();
        cbody.append(InstructionConstants.ALOAD_0);
        cbody.append(cg.getFactory().createInvoke(parentName, "<init>", Type.VOID, NO_ARGS, Constants.INVOKESPECIAL));
        cbody.append(InstructionConstants.RETURN);
        cg.addMethodGen(init);
        for (Definition.Pointcut abstractPc : concreteAspect.pointcuts) {
            LazyMethodGen mg = new LazyMethodGen(Modifier.PUBLIC, Type.VOID, abstractPc.name, NO_ARGS, EMPTY_STRINGS, cg);
            SimpleElementValue svg = new SimpleElementValue(ElementValue.STRING, cg.getConstantPool(), abstractPc.expression);
            List<NameValuePair> elems = new ArrayList<>();
            elems.add(new NameValuePair("value", svg, cg.getConstantPool()));
            AnnotationGen mag = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/Pointcut"), elems, true, cg.getConstantPool());
            AnnotationAJ max = new BcelAnnotation(mag, world);
            mg.addAnnotation(max);
            InstructionList body = mg.getBody();
            body.append(InstructionConstants.RETURN);
            cg.addMethodGen(mg);
        }
        if (concreteAspect.deows.size() > 0) {
            int counter = 1;
            for (Definition.DeclareErrorOrWarning deow : concreteAspect.deows) {
                FieldGen field = new FieldGen(Modifier.FINAL, ObjectType.STRING, "rule" + (counter++), cg.getConstantPool());
                SimpleElementValue svg = new SimpleElementValue(ElementValue.STRING, cg.getConstantPool(), deow.pointcut);
                List<NameValuePair> elems = new ArrayList<>();
                elems.add(new NameValuePair("value", svg, cg.getConstantPool()));
                AnnotationGen mag = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/Declare" + (deow.isError ? "Error" : "Warning")), elems, true, cg.getConstantPool());
                field.addAnnotation(mag);
                field.setValue(deow.message);
                cg.addField(field, null);
            }
        }
        if (concreteAspect.pointcutsAndAdvice.size() > 0) {
            int adviceCounter = 1;
            for (PointcutAndAdvice paa : concreteAspect.pointcutsAndAdvice) {
                generateAdviceMethod(paa, adviceCounter, cg);
                adviceCounter++;
            }
        }
        if (concreteAspect.declareAnnotations.size() > 0) {
            int decCounter = 1;
            for (Definition.DeclareAnnotation da : concreteAspect.declareAnnotations) {
                generateDeclareAnnotation(da, decCounter++, cg);
            }
        }
        ReferenceType rt = new ReferenceType(ResolvedType.forName(concreteAspect.name).getSignature(), world);
        GeneratedReferenceTypeDelegate grtd = new GeneratedReferenceTypeDelegate(rt);
        grtd.setSuperclass(parent);
        rt.setDelegate(grtd);
        BcelPerClauseAspectAdder perClauseMunger = new BcelPerClauseAspectAdder(rt, perclauseKind);
        perClauseMunger.forceMunge(cg, false);
        JavaClass jc = cg.getJavaClass((BcelWorld) world);
        ((BcelWorld) world).addSourceObjectType(jc, true);
        bytes = jc.getBytes();
        return bytes;
    }

    private void generateDeclareAnnotation(Definition.DeclareAnnotation da, int decCounter, LazyClassGen cg) {
        AnnotationAJ constructedAnnotation = buildDeclareAnnotation_actualAnnotation(cg, da);
        if (constructedAnnotation == null) {
            return;
        }
        String nameComponent = da.declareAnnotationKind.name().toLowerCase();
        String declareName = new StringBuilder("ajc$declare_at_").append(nameComponent).append("_").append(decCounter).toString();
        LazyMethodGen declareMethod = new LazyMethodGen(Modifier.PUBLIC, Type.VOID, declareName, NO_ARGS, EMPTY_STRINGS, cg);
        InstructionList declareMethodBody = declareMethod.getBody();
        declareMethodBody.append(InstructionFactory.RETURN);
        declareMethod.addAnnotation(constructedAnnotation);
        DeclareAnnotation deca = null;
        ITokenSource tokenSource = BasicTokenSource.makeTokenSource(da.pattern, null);
        PatternParser pp = new PatternParser(tokenSource);
        if (da.declareAnnotationKind == DeclareAnnotationKind.Method || da.declareAnnotationKind == DeclareAnnotationKind.Field) {
            ISignaturePattern isp = (da.declareAnnotationKind == DeclareAnnotationKind.Method ? pp.parseCompoundMethodOrConstructorSignaturePattern(true) : pp.parseCompoundFieldSignaturePattern());
            deca = new DeclareAnnotation(da.declareAnnotationKind == DeclareAnnotationKind.Method ? DeclareAnnotation.AT_METHOD : DeclareAnnotation.AT_FIELD, isp);
        } else if (da.declareAnnotationKind == DeclareAnnotationKind.Type) {
            TypePattern tp = pp.parseTypePattern();
            deca = new DeclareAnnotation(DeclareAnnotation.AT_TYPE, tp);
        }
        deca.setAnnotationMethod(declareName);
        deca.setAnnotationString(da.annotation);
        AjAttribute attribute = new AjAttribute.DeclareAttribute(deca);
        cg.addAttribute(attribute);
        cg.addMethodGen(declareMethod);
    }

    private AnnotationAJ buildDeclareAnnotation_actualAnnotation(LazyClassGen cg, Definition.DeclareAnnotation da) {
        AnnotationGen anno = buildAnnotationFromString(cg.getConstantPool(), cg.getWorld(), da.annotation);
        if (anno == null) {
            return null;
        } else {
            AnnotationAJ bcelAnnotation = new BcelAnnotation(anno, world);
            return bcelAnnotation;
        }
    }

    private AnnotationGen buildAnnotationFromString(ConstantPool cp, World w, String annotationString) {
        int paren = annotationString.indexOf('(');
        if (paren == -1) {
            AnnotationGen aaj = buildBaseAnnotationType(cp, world, annotationString);
            return aaj;
        } else {
            String name = annotationString.substring(0, paren);
            List<String> values = new ArrayList<>();
            int pos = paren + 1;
            int depth = 0;
            int len = annotationString.length();
            int start = pos;
            while (pos < len) {
                char ch = annotationString.charAt(pos);
                if (ch == ')' && depth == 0) {
                    break;
                }
                if (ch == '(' || ch == '[') {
                    depth++;
                } else if (ch == ')' || ch == ']') {
                    depth--;
                }
                if (ch == ',' && depth == 0) {
                    values.add(annotationString.substring(start, pos).trim());
                    start = pos + 1;
                }
                pos++;
            }
            if (start != pos) {
                values.add(annotationString.substring(start, pos).trim());
            }
            AnnotationGen aaj = buildBaseAnnotationType(cp, world, name);
            if (aaj == null) {
                return null;
            }
            String typename = aaj.getTypeName();
            ResolvedType type = UnresolvedType.forName(typename).resolve(world);
            ResolvedMember[] rms = type.getDeclaredMethods();
            for (String value : values) {
                int equalsIndex = value.indexOf("=");
                String key = "value";
                if (value.charAt(0) != '\"' && equalsIndex != -1) {
                    key = value.substring(0, equalsIndex).trim();
                    value = value.substring(equalsIndex + 1).trim();
                }
                boolean keyIsOk = false;
                for (ResolvedMember rm : rms) {
                    NameValuePair nvp = null;
                    if (rm.getName().equals(key)) {
                        keyIsOk = true;
                        UnresolvedType rt = rm.getReturnType();
                        if (rt.isPrimitiveType()) {
                            switch (rt.getSignature().charAt(0)) {
                                case 'J':
                                    try {
                                        long longValue = Long.parseLong(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_LONG, cp, longValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a long");
                                        return null;
                                    }
                                    break;
                                case 'S':
                                    try {
                                        short shortValue = Short.parseShort(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_SHORT, cp, shortValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a short");
                                        return null;
                                    }
                                    break;
                                case 'F':
                                    try {
                                        float floatValue = Float.parseFloat(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_FLOAT, cp, floatValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a float");
                                        return null;
                                    }
                                    break;
                                case 'D':
                                    try {
                                        double doubleValue = Double.parseDouble(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_DOUBLE, cp, doubleValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a double");
                                        return null;
                                    }
                                    break;
                                case 'I':
                                    try {
                                        int intValue = Integer.parseInt(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_INT, cp, intValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as an integer");
                                        return null;
                                    }
                                    break;
                                case 'B':
                                    try {
                                        byte byteValue = Byte.parseByte(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_BYTE, cp, byteValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a byte");
                                        return null;
                                    }
                                    break;
                                case 'C':
                                    if (value.length() < 2) {
                                        reportError("unable to interpret annotation value '" + value + "' as a char");
                                        return null;
                                    }
                                    nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_CHAR, cp, value.charAt(1)), cp);
                                    break;
                                case 'Z':
                                    try {
                                        boolean booleanValue = Boolean.parseBoolean(value);
                                        nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.PRIMITIVE_BOOLEAN, cp, booleanValue), cp);
                                    } catch (NumberFormatException nfe) {
                                        reportError("unable to interpret annotation value '" + value + "' as a boolean");
                                        return null;
                                    }
                                    break;
                                default:
                                    reportError("not yet supporting XML setting of annotation values of type " + rt.getName());
                                    return null;
                            }
                        } else if (UnresolvedType.JL_STRING.equals(rt)) {
                            if (value.length() < 2) {
                                reportError("Invalid string value specified in annotation string: " + annotationString);
                                return null;
                            }
                            value = value.substring(1, value.length() - 1);
                            nvp = new NameValuePair(key, new SimpleElementValue(ElementValue.STRING, cp, value), cp);
                        } else if (UnresolvedType.JL_CLASS.equals(rt)) {
                            if (value.length() < 6) {
                                reportError("Not a well formed class value for an annotation '" + value + "'");
                                return null;
                            }
                            String clazz = value.substring(0, value.length() - 6);
                            boolean qualified = clazz.contains(".");
                            if (!qualified) {
                                clazz = "java.lang." + clazz;
                            }
                            nvp = new NameValuePair(key, new ClassElementValue(new ObjectType(clazz), cp), cp);
                        }
                    }
                    if (nvp != null) {
                        aaj.addElementNameValuePair(nvp);
                    }
                }
                if (!keyIsOk) {
                    reportError("annotation @" + typename + " does not have a value named " + key);
                    return null;
                }
            }
            return aaj;
        }
    }

    private AnnotationGen buildBaseAnnotationType(ConstantPool cp, World world, String typename) {
        String annoname = typename;
        if (annoname.startsWith("@")) {
            annoname = annoname.substring(1);
        }
        ResolvedType annotationType = UnresolvedType.forName(annoname).resolve(world);
        if (!annotationType.isAnnotation()) {
            reportError("declare is not specifying an annotation type :" + typename);
            return null;
        }
        if (!annotationType.isAnnotationWithRuntimeRetention()) {
            reportError("declare is using an annotation type that does not have runtime retention: " + typename);
            return null;
        }
        List<NameValuePair> elems = new ArrayList<>();
        return new AnnotationGen(new ObjectType(annoname), elems, true, cp);
    }

    private void generateAdviceMethod(PointcutAndAdvice paa, int adviceCounter, LazyClassGen cg) {
        ResolvedType delegateClass = world.resolve(UnresolvedType.forName(paa.adviceClass));
        if (delegateClass.isMissing()) {
            reportError("Class to invoke cannot be found: '" + paa.adviceClass + "'");
            return;
        }
        String adviceName = new StringBuilder("generated$").append(paa.adviceKind.toString().toLowerCase()).append("$advice$").append(adviceCounter).toString();
        AnnotationAJ aaj = buildAdviceAnnotation(cg, paa);
        String method = paa.adviceMethod;
        int paren = method.indexOf("(");
        String methodName = method.substring(0, paren);
        String signature = method.substring(paren);
        if (signature.charAt(0) != '(' || !signature.endsWith(")")) {
            reportError("Badly formatted parameter signature: '" + method + "'");
            return;
        }
        List<Type> paramTypes = new ArrayList<>();
        List<String> paramNames = new ArrayList<>();
        if (signature.charAt(1) != ')') {
            StringBuilder convertedSignature = new StringBuilder("(");
            boolean paramsBroken = false;
            int pos = 1;
            while (pos < signature.length() && signature.charAt(pos) != ')' && !paramsBroken) {
                int nextChunkEndPos = signature.indexOf(',', pos);
                if (nextChunkEndPos == -1) {
                    nextChunkEndPos = signature.indexOf(')', pos);
                }
                String nextChunk = signature.substring(pos, nextChunkEndPos).trim();
                int space = nextChunk.indexOf(" ");
                ResolvedType resolvedParamType = null;
                if (space == -1) {
                    if (nextChunk.equals("JoinPoint")) {
                        nextChunk = "org.aspectj.lang.JoinPoint";
                    } else if (nextChunk.equals("JoinPoint.StaticPart")) {
                        nextChunk = "org.aspectj.lang.JoinPoint$StaticPart";
                    } else if (nextChunk.equals("ProceedingJoinPoint")) {
                        nextChunk = "org.aspectj.lang.ProceedingJoinPoint";
                    }
                    UnresolvedType unresolvedParamType = UnresolvedType.forName(nextChunk);
                    resolvedParamType = world.resolve(unresolvedParamType);
                } else {
                    String typename = nextChunk.substring(0, space);
                    if (typename.equals("JoinPoint")) {
                        typename = "org.aspectj.lang.JoinPoint";
                    } else if (typename.equals("JoinPoint.StaticPart")) {
                        typename = "org.aspectj.lang.JoinPoint$StaticPart";
                    } else if (typename.equals("ProceedingJoinPoint")) {
                        typename = "org.aspectj.lang.ProceedingJoinPoint";
                    }
                    UnresolvedType unresolvedParamType = UnresolvedType.forName(typename);
                    resolvedParamType = world.resolve(unresolvedParamType);
                    String paramname = nextChunk.substring(space).trim();
                    paramNames.add(paramname);
                }
                if (resolvedParamType.isMissing()) {
                    reportError("Cannot find type specified as parameter: '" + nextChunk + "' from signature '" + signature + "'");
                    paramsBroken = true;
                }
                paramTypes.add(Type.getType(resolvedParamType.getSignature()));
                convertedSignature.append(resolvedParamType.getSignature());
                pos = nextChunkEndPos + 1;
            }
            convertedSignature.append(")");
            signature = convertedSignature.toString();
            if (paramsBroken) {
                return;
            }
        }
        Type returnType = Type.VOID;
        if (paa.adviceKind == AdviceKind.Around) {
            ResolvedMember[] methods = delegateClass.getDeclaredMethods();
            ResolvedMember found = null;
            for (ResolvedMember candidate : methods) {
                if (candidate.getName().equals(methodName)) {
                    UnresolvedType[] cparms = candidate.getParameterTypes();
                    if (cparms.length == paramTypes.size()) {
                        boolean paramsMatch = true;
                        for (int i = 0; i < cparms.length; i++) {
                            if (!cparms[i].getSignature().equals(paramTypes.get(i).getSignature())) {
                                paramsMatch = false;
                                break;
                            }
                        }
                        if (paramsMatch) {
                            found = candidate;
                            break;
                        }
                    }
                }
            }
            if (found != null) {
                returnType = Type.getType(found.getReturnType().getSignature());
            } else {
                reportError("Unable to find method to invoke.  In class: " + delegateClass.getName() + " cant find " + paa.adviceMethod);
                return;
            }
        }
        LazyMethodGen advice = new LazyMethodGen(Modifier.PUBLIC, returnType, adviceName, paramTypes.toArray(NO_ARGS), EMPTY_STRINGS, cg);
        InstructionList adviceBody = advice.getBody();
        int pos = 1;
        for (Type paramType : paramTypes) {
            adviceBody.append(InstructionFactory.createLoad(paramType, pos));
            pos += paramType.getSize();
        }
        adviceBody.append(cg.getFactory().createInvoke(paa.adviceClass, methodName, signature + returnType.getSignature(), Constants.INVOKESTATIC));
        if (returnType == Type.VOID) {
            adviceBody.append(InstructionConstants.RETURN);
        } else {
            if (returnType.getSignature().length() < 2) {
                String sig = returnType.getSignature();
                if (sig.equals("F")) {
                    adviceBody.append(InstructionConstants.FRETURN);
                } else if (sig.equals("D")) {
                    adviceBody.append(InstructionConstants.DRETURN);
                } else if (sig.equals("J")) {
                    adviceBody.append(InstructionConstants.LRETURN);
                } else {
                    adviceBody.append(InstructionConstants.IRETURN);
                }
            } else {
                adviceBody.append(InstructionConstants.ARETURN);
            }
        }
        advice.addAnnotation(aaj);
        InstructionHandle start = adviceBody.getStart();
        String sig = concreteAspect.name.replace('.', '/');
        start.addTargeter(new LocalVariableTag("L" + sig + ";", "this", 0, start.getPosition()));
        if (paramNames.size() > 0) {
            for (int i = 0; i < paramNames.size(); i++) {
                start.addTargeter(new LocalVariableTag(paramTypes.get(i).getSignature(), paramNames.get(i), i + 1, start.getPosition()));
            }
        }
        cg.addMethodGen(advice);
    }

    private AnnotationAJ buildAdviceAnnotation(LazyClassGen cg, PointcutAndAdvice paa) {
        SimpleElementValue svg = new SimpleElementValue(ElementValue.STRING, cg.getConstantPool(), paa.pointcut);
        List<NameValuePair> elems = new ArrayList<>();
        elems.add(new NameValuePair("value", svg, cg.getConstantPool()));
        AnnotationGen mag = new AnnotationGen(new ObjectType("org/aspectj/lang/annotation/" + paa.adviceKind.toString()), elems, true, cg.getConstantPool());
        AnnotationAJ aaj = new BcelAnnotation(mag, world);
        return aaj;
    }

    private void reportError(String message) {
        world.getMessageHandler().handleMessage(new Message(message, IMessage.ERROR, null, null));
    }
}
