package org.aspectj.weaver.patterns;

import org.aspectj.bridge.MessageUtil;
import org.aspectj.util.FuzzyBoolean;
import org.aspectj.weaver.AjcMemberMaker;
import org.aspectj.weaver.AnnotatedElement;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.CompressingDataOutputStream;
import org.aspectj.weaver.ConcreteTypeMunger;
import org.aspectj.weaver.ISourceContext;
import org.aspectj.weaver.IntMap;
import org.aspectj.weaver.Member;
import org.aspectj.weaver.NameMangler;
import org.aspectj.weaver.NewFieldTypeMunger;
import org.aspectj.weaver.ResolvedMember;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.ShadowMunger;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.VersionedDataInputStream;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.World;
import org.aspectj.weaver.ast.Literal;
import org.aspectj.weaver.ast.Test;
import org.aspectj.weaver.ast.Var;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class AnnotationPointcut extends NameBindingPointcut {

    private ExactAnnotationTypePattern annotationTypePattern;

    private String declarationText;

    public AnnotationPointcut(ExactAnnotationTypePattern type) {
        super();
        this.annotationTypePattern = type;
        this.pointcutKind = Pointcut.ANNOTATION;
        buildDeclarationText();
    }

    public AnnotationPointcut(ExactAnnotationTypePattern type, ShadowMunger munger) {
        this(type);
        buildDeclarationText();
    }

    public ExactAnnotationTypePattern getAnnotationTypePattern() {
        return annotationTypePattern;
    }

    @Override
    public int couldMatchKinds() {
        return Shadow.ALL_SHADOW_KINDS_BITS;
    }

    @Override
    public Pointcut parameterizeWith(Map<String, UnresolvedType> typeVariableMap, World w) {
        AnnotationPointcut ret = new AnnotationPointcut((ExactAnnotationTypePattern) annotationTypePattern.parameterizeWith(typeVariableMap, w));
        ret.copyLocationFrom(this);
        return ret;
    }

    @Override
    public FuzzyBoolean fastMatch(FastMatchInfo info) {
        if (info.getKind() == Shadow.StaticInitialization) {
            return annotationTypePattern.fastMatches(info.getType());
        } else {
            return FuzzyBoolean.MAYBE;
        }
    }

    @Override
    protected FuzzyBoolean matchInternal(Shadow shadow) {
        AnnotatedElement toMatchAgainst = null;
        Member member = shadow.getSignature();
        ResolvedMember rMember = member.resolve(shadow.getIWorld());
        if (rMember == null) {
            if (member.getName().startsWith(NameMangler.PREFIX)) {
                return FuzzyBoolean.NO;
            }
            shadow.getIWorld().getLint().unresolvableMember.signal(member.toString(), getSourceLocation());
            return FuzzyBoolean.NO;
        }
        Shadow.Kind kind = shadow.getKind();
        if (kind == Shadow.StaticInitialization) {
            toMatchAgainst = rMember.getDeclaringType().resolve(shadow.getIWorld());
        } else if ((kind == Shadow.ExceptionHandler)) {
            toMatchAgainst = rMember.getParameterTypes()[0].resolve(shadow.getIWorld());
        } else {
            toMatchAgainst = rMember;
            if (rMember.isAnnotatedElsewhere()) {
                if (kind == Shadow.FieldGet || kind == Shadow.FieldSet) {
                    List<ConcreteTypeMunger> mungers = rMember.getDeclaringType().resolve(shadow.getIWorld()).getInterTypeMungers();
                    for (ConcreteTypeMunger typeMunger : mungers) {
                        if (typeMunger.getMunger() instanceof NewFieldTypeMunger) {
                            ResolvedMember fakerm = typeMunger.getSignature();
                            if (fakerm.equals(member)) {
                                ResolvedMember ajcMethod = AjcMemberMaker.interFieldInitializer(fakerm, typeMunger.getAspectType());
                                ResolvedMember rmm = findMethod(typeMunger.getAspectType(), ajcMethod);
                                toMatchAgainst = rmm;
                            }
                        }
                    }
                }
            }
        }
        annotationTypePattern.resolve(shadow.getIWorld());
        return annotationTypePattern.matches(toMatchAgainst);
    }

    private ResolvedMember findMethod(ResolvedType aspectType, ResolvedMember ajcMethod) {
        ResolvedMember[] decMethods = aspectType.getDeclaredMethods();
        for (ResolvedMember member : decMethods) {
            if (member.equals(ajcMethod)) {
                return member;
            }
        }
        return null;
    }

    @Override
    protected void resolveBindings(IScope scope, Bindings bindings) {
        if (!scope.getWorld().isInJava5Mode()) {
            scope.message(MessageUtil.error(WeaverMessages.format(WeaverMessages.ATANNOTATION_ONLY_SUPPORTED_AT_JAVA5_LEVEL), getSourceLocation()));
            return;
        }
        annotationTypePattern = (ExactAnnotationTypePattern) annotationTypePattern.resolveBindings(scope, bindings, true);
    }

    @Override
    protected Pointcut concretize1(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings) {
        ExactAnnotationTypePattern newType = (ExactAnnotationTypePattern) annotationTypePattern.remapAdviceFormals(bindings);
        Pointcut ret = new AnnotationPointcut(newType, bindings.getEnclosingAdvice());
        ret.copyLocationFrom(this);
        return ret;
    }

    @Override
    protected Test findResidueInternal(Shadow shadow, ExposedState state) {
        if (annotationTypePattern instanceof BindingAnnotationFieldTypePattern) {
            if (shadow.getKind() != Shadow.MethodExecution) {
                shadow.getIWorld().getMessageHandler().handleMessage(MessageUtil.error("Annotation field binding is only supported at method-execution join points (compiler limitation)", getSourceLocation()));
                return Literal.TRUE;
            }
            BindingAnnotationFieldTypePattern btp = (BindingAnnotationFieldTypePattern) annotationTypePattern;
            ResolvedType formalType = btp.getFormalType().resolve(shadow.getIWorld());
            UnresolvedType annoType = btp.getAnnotationType();
            Var var = shadow.getKindedAnnotationVar(annoType);
            if (var == null) {
                throw new BCException("Unexpected problem locating annotation at join point '" + shadow + "'");
            }
            state.set(btp.getFormalIndex(), var.getAccessorForValue(formalType, btp.formalName));
        } else if (annotationTypePattern instanceof BindingAnnotationTypePattern) {
            BindingAnnotationTypePattern btp = (BindingAnnotationTypePattern) annotationTypePattern;
            UnresolvedType annotationType = btp.getAnnotationType();
            Var var = shadow.getKindedAnnotationVar(annotationType);
            if (var == null) {
                if (matchInternal(shadow).alwaysTrue()) {
                    return Literal.TRUE;
                } else {
                    return Literal.FALSE;
                }
            }
            state.set(btp.getFormalIndex(), var);
        }
        if (matchInternal(shadow).alwaysTrue()) {
            return Literal.TRUE;
        } else {
            return Literal.FALSE;
        }
    }

    @Override
    public List<BindingPattern> getBindingAnnotationTypePatterns() {
        if (annotationTypePattern instanceof BindingPattern) {
            List<BindingPattern> l = new ArrayList<>();
            l.add((BindingPattern) annotationTypePattern);
            return l;
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public List<BindingTypePattern> getBindingTypePatterns() {
        return Collections.emptyList();
    }

    @Override
    public void write(CompressingDataOutputStream s) throws IOException {
        s.writeByte(Pointcut.ANNOTATION);
        annotationTypePattern.write(s);
        writeLocation(s);
    }

    public static Pointcut read(VersionedDataInputStream s, ISourceContext context) throws IOException {
        AnnotationTypePattern type = AnnotationTypePattern.read(s, context);
        AnnotationPointcut ret = new AnnotationPointcut((ExactAnnotationTypePattern) type);
        ret.readLocation(context, s);
        return ret;
    }

    @Override
    public boolean equals(Object other) {
        if (!(other instanceof AnnotationPointcut)) {
            return false;
        }
        AnnotationPointcut o = (AnnotationPointcut) other;
        return o.annotationTypePattern.equals(this.annotationTypePattern);
    }

    @Override
    public int hashCode() {
        int result = 17;
        result = 37 * result + annotationTypePattern.hashCode();
        return result;
    }

    public void buildDeclarationText() {
        StringBuilder buf = new StringBuilder();
        buf.append("@annotation(");
        String annPatt = annotationTypePattern.toString();
        buf.append(annPatt.startsWith("@") ? annPatt.substring(1) : annPatt);
        buf.append(")");
        this.declarationText = buf.toString();
    }

    @Override
    public String toString() {
        return this.declarationText;
    }

    @Override
    public Object accept(PatternNodeVisitor visitor, Object data) {
        return visitor.visit(this, data);
    }
}
