package org.aspectj.weaver.patterns;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.util.FuzzyBoolean;
import org.aspectj.weaver.CompressingDataOutputStream;
import org.aspectj.weaver.ISourceContext;
import org.aspectj.weaver.IntMap;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.VersionedDataInputStream;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.World;
import org.aspectj.weaver.ast.Literal;
import org.aspectj.weaver.ast.Test;
import org.aspectj.weaver.ast.Var;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class ArgsAnnotationPointcut extends NameBindingPointcut {

    private AnnotationPatternList arguments;

    private String declarationText;

    public ArgsAnnotationPointcut(AnnotationPatternList arguments) {
        super();
        this.arguments = arguments;
        this.pointcutKind = ATARGS;
        buildDeclarationText();
    }

    public AnnotationPatternList getArguments() {
        return arguments;
    }

    public int couldMatchKinds() {
        return Shadow.ALL_SHADOW_KINDS_BITS;
    }

    public Pointcut parameterizeWith(Map typeVariableMap, World w) {
        ArgsAnnotationPointcut ret = new ArgsAnnotationPointcut(arguments.parameterizeWith(typeVariableMap, w));
        ret.copyLocationFrom(this);
        return ret;
    }

    public FuzzyBoolean fastMatch(FastMatchInfo info) {
        return FuzzyBoolean.MAYBE;
    }

    protected FuzzyBoolean matchInternal(Shadow shadow) {
        arguments.resolve(shadow.getIWorld());
        FuzzyBoolean ret = arguments.matches(shadow.getIWorld().resolve(shadow.getArgTypes()));
        return ret;
    }

    protected void resolveBindings(IScope scope, Bindings bindings) {
        if (!scope.getWorld().isInJava5Mode()) {
            scope.message(MessageUtil.error(WeaverMessages.format(WeaverMessages.ATARGS_ONLY_SUPPORTED_AT_JAVA5_LEVEL), getSourceLocation()));
            return;
        }
        arguments.resolveBindings(scope, bindings, true);
        if (arguments.ellipsisCount > 1) {
            scope.message(IMessage.ERROR, this, "uses more than one .. in args (compiler limitation)");
        }
    }

    protected Pointcut concretize1(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings) {
        if (isDeclare(bindings.getEnclosingAdvice())) {
            inAspect.getWorld().showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.ARGS_IN_DECLARE), bindings.getEnclosingAdvice().getSourceLocation(), null);
            return Pointcut.makeMatchesNothing(Pointcut.CONCRETE);
        }
        AnnotationPatternList list = arguments.resolveReferences(bindings);
        Pointcut ret = new ArgsAnnotationPointcut(list);
        ret.copyLocationFrom(this);
        return ret;
    }

    protected Test findResidueInternal(Shadow shadow, ExposedState state) {
        int len = shadow.getArgCount();
        int numArgsMatchedByEllipsis = (len + arguments.ellipsisCount) - arguments.size();
        if (numArgsMatchedByEllipsis < 0) {
            return Literal.FALSE;
        }
        if ((numArgsMatchedByEllipsis > 0) && (arguments.ellipsisCount == 0)) {
            return Literal.FALSE;
        }
        Test ret = Literal.TRUE;
        int argsIndex = 0;
        for (int i = 0; i < arguments.size(); i++) {
            if (arguments.get(i) == AnnotationTypePattern.ELLIPSIS) {
                argsIndex += numArgsMatchedByEllipsis;
            } else if (arguments.get(i) == AnnotationTypePattern.ANY) {
                argsIndex++;
            } else {
                ExactAnnotationTypePattern ap = (ExactAnnotationTypePattern) arguments.get(i);
                UnresolvedType argType = shadow.getArgType(argsIndex);
                ResolvedType rArgType = argType.resolve(shadow.getIWorld());
                if (rArgType.isMissing()) {
                    shadow.getIWorld().getLint().cantFindType.signal(new String[]{WeaverMessages.format(WeaverMessages.CANT_FIND_TYPE_ARG_TYPE, argType.getName())}, shadow.getSourceLocation(), new ISourceLocation[]{getSourceLocation()});
                }
                ResolvedType rAnnType = ap.getAnnotationType().resolve(shadow.getIWorld());
                if (ap instanceof BindingAnnotationTypePattern) {
                    BindingAnnotationTypePattern btp = (BindingAnnotationTypePattern) ap;
                    Var annvar = shadow.getArgAnnotationVar(argsIndex, rAnnType);
                    state.set(btp.getFormalIndex(), annvar);
                }
                if (!ap.matches(rArgType).alwaysTrue()) {
                    ret = Test.makeAnd(ret, Test.makeHasAnnotation(shadow.getArgVar(argsIndex), rAnnType));
                }
                argsIndex++;
            }
        }
        return ret;
    }

    public List<BindingPattern> getBindingAnnotationTypePatterns() {
        List<BindingPattern> l = new ArrayList<>();
        AnnotationTypePattern[] pats = arguments.getAnnotationPatterns();
        for (AnnotationTypePattern pat : pats) {
            if (pat instanceof BindingAnnotationTypePattern) {
                l.add((BindingPattern) pat);
            }
        }
        return l;
    }

    public List<BindingTypePattern> getBindingTypePatterns() {
        return Collections.emptyList();
    }

    public void write(CompressingDataOutputStream s) throws IOException {
        s.writeByte(Pointcut.ATARGS);
        arguments.write(s);
        writeLocation(s);
    }

    public static Pointcut read(VersionedDataInputStream s, ISourceContext context) throws IOException {
        AnnotationPatternList annotationPatternList = AnnotationPatternList.read(s, context);
        ArgsAnnotationPointcut ret = new ArgsAnnotationPointcut(annotationPatternList);
        ret.readLocation(context, s);
        return ret;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof ArgsAnnotationPointcut)) {
            return false;
        }
        ArgsAnnotationPointcut other = (ArgsAnnotationPointcut) obj;
        return other.arguments.equals(arguments);
    }

    public int hashCode() {
        return 17 + 37 * arguments.hashCode();
    }

    private void buildDeclarationText() {
        StringBuilder buf = new StringBuilder("@args");
        buf.append(arguments.toString());
        this.declarationText = buf.toString();
    }

    public String toString() {
        return this.declarationText;
    }

    public Object accept(PatternNodeVisitor visitor, Object data) {
        return visitor.visit(this, data);
    }
}
