package org.aspectj.weaver.patterns;

import org.aspectj.util.FuzzyBoolean;
import org.aspectj.util.TypeSafeEnum;
import org.aspectj.weaver.Advice;
import org.aspectj.weaver.AdviceKind;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.Checker;
import org.aspectj.weaver.CompressingDataOutputStream;
import org.aspectj.weaver.ISourceContext;
import org.aspectj.weaver.IntMap;
import org.aspectj.weaver.PoliceExtensionUse;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.ShadowMunger;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.VersionedDataInputStream;
import org.aspectj.weaver.World;
import org.aspectj.weaver.ast.Literal;
import org.aspectj.weaver.ast.Test;

import java.io.IOException;
import java.util.Map;

public abstract class Pointcut extends PatternNode {

    public static final class State extends TypeSafeEnum {

        public State(String name, int key) {
            super(name, key);
        }
    }

    public String[] m_ignoreUnboundBindingForNames = EMPTY_STRING_ARRAY;

    public static final String[] EMPTY_STRING_ARRAY = new String[0];

    public static final State SYMBOLIC = new State("symbolic", 0);

    public static final State RESOLVED = new State("resolved", 1);

    public static final State CONCRETE = new State("concrete", 2);

    protected byte pointcutKind;

    public State state;

    protected int lastMatchedShadowId;

    private FuzzyBoolean lastMatchedShadowResult;

    private String[] typeVariablesInScope = EMPTY_STRING_ARRAY;

    protected boolean hasBeenParameterized = false;

    public Pointcut() {
        super();
        this.state = SYMBOLIC;
    }

    public abstract FuzzyBoolean fastMatch(FastMatchInfo info);

    public abstract int couldMatchKinds();

    public String[] getTypeVariablesInScope() {
        return typeVariablesInScope;
    }

    public void setTypeVariablesInScope(String[] typeVars) {
        this.typeVariablesInScope = typeVars;
    }

    public final FuzzyBoolean match(Shadow shadow) {
        if (shadow.shadowId == lastMatchedShadowId) {
            return lastMatchedShadowResult;
        }
        FuzzyBoolean ret;
        if (shadow.getKind().isSet(couldMatchKinds())) {
            ret = matchInternal(shadow);
        } else {
            ret = FuzzyBoolean.NO;
        }
        lastMatchedShadowId = shadow.shadowId;
        lastMatchedShadowResult = ret;
        return ret;
    }

    protected abstract FuzzyBoolean matchInternal(Shadow shadow);

    public static final byte KINDED = 1;

    public static final byte WITHIN = 2;

    public static final byte THIS_OR_TARGET = 3;

    public static final byte ARGS = 4;

    public static final byte AND = 5;

    public static final byte OR = 6;

    public static final byte NOT = 7;

    public static final byte REFERENCE = 8;

    public static final byte IF = 9;

    public static final byte CFLOW = 10;

    public static final byte WITHINCODE = 12;

    public static final byte HANDLER = 13;

    public static final byte IF_TRUE = 14;

    public static final byte IF_FALSE = 15;

    public static final byte ANNOTATION = 16;

    public static final byte ATWITHIN = 17;

    public static final byte ATWITHINCODE = 18;

    public static final byte ATTHIS_OR_TARGET = 19;

    public static final byte NONE = 20;

    public static final byte ATARGS = 21;

    public static final byte USER_EXTENSION = 22;

    public byte getPointcutKind() {
        return pointcutKind;
    }

    protected abstract void resolveBindings(IScope scope, Bindings bindings);

    public final Pointcut resolve(IScope scope) {
        assertState(SYMBOLIC);
        Bindings bindingTable = new Bindings(scope.getFormalCount());
        IScope bindingResolutionScope = scope;
        if (typeVariablesInScope.length > 0) {
            bindingResolutionScope = new ScopeWithTypeVariables(typeVariablesInScope, scope);
        }
        this.resolveBindings(bindingResolutionScope, bindingTable);
        bindingTable.checkAllBound(bindingResolutionScope);
        this.state = RESOLVED;
        return this;
    }

    public final Pointcut concretize(ResolvedType inAspect, ResolvedType declaringType, int arity) {
        Pointcut ret = concretize(inAspect, declaringType, IntMap.idMap(arity));
        ret.m_ignoreUnboundBindingForNames = m_ignoreUnboundBindingForNames;
        return ret;
    }

    public final Pointcut concretize(ResolvedType inAspect, ResolvedType declaringType, int arity, ShadowMunger advice) {
        IntMap map = IntMap.idMap(arity);
        map.setEnclosingAdvice(advice);
        map.setConcreteAspect(inAspect);
        return concretize(inAspect, declaringType, map);
    }

    public boolean isDeclare(ShadowMunger munger) {
        if (munger == null) {
            return false;
        }
        if (munger instanceof Checker) {
            return true;
        }
        if (((Advice) munger).getKind().equals(AdviceKind.Softener)) {
            return true;
        }
        return false;
    }

    public final Pointcut concretize(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings) {
        Pointcut ret = this.concretize1(inAspect, declaringType, bindings);
        if (shouldCopyLocationForConcretize()) {
            ret.copyLocationFrom(this);
        }
        ret.state = CONCRETE;
        ret.m_ignoreUnboundBindingForNames = m_ignoreUnboundBindingForNames;
        return ret;
    }

    protected boolean shouldCopyLocationForConcretize() {
        return true;
    }

    protected abstract Pointcut concretize1(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings);

    public final Test findResidue(Shadow shadow, ExposedState state) {
        Test ret = findResidueInternal(shadow, state);
        lastMatchedShadowId = shadow.shadowId;
        return ret;
    }

    protected abstract Test findResidueInternal(Shadow shadow, ExposedState state);

    public void postRead(ResolvedType enclosingType) {
    }

    public static Pointcut read(VersionedDataInputStream s, ISourceContext context) throws IOException {
        byte kind = s.readByte();
        Pointcut ret;
        switch (kind) {
            case KINDED:
                ret = KindedPointcut.read(s, context);
                break;
            case WITHIN:
                ret = WithinPointcut.read(s, context);
                break;
            case THIS_OR_TARGET:
                ret = ThisOrTargetPointcut.read(s, context);
                break;
            case ARGS:
                ret = ArgsPointcut.read(s, context);
                break;
            case AND:
                ret = AndPointcut.read(s, context);
                break;
            case OR:
                ret = OrPointcut.read(s, context);
                break;
            case NOT:
                ret = NotPointcut.read(s, context);
                break;
            case REFERENCE:
                ret = ReferencePointcut.read(s, context);
                break;
            case IF:
                ret = IfPointcut.read(s, context);
                break;
            case CFLOW:
                ret = CflowPointcut.read(s, context);
                break;
            case WITHINCODE:
                ret = WithincodePointcut.read(s, context);
                break;
            case HANDLER:
                ret = HandlerPointcut.read(s, context);
                break;
            case IF_TRUE:
                ret = IfPointcut.makeIfTruePointcut(RESOLVED);
                break;
            case IF_FALSE:
                ret = IfPointcut.makeIfFalsePointcut(RESOLVED);
                break;
            case ANNOTATION:
                ret = AnnotationPointcut.read(s, context);
                break;
            case ATWITHIN:
                ret = WithinAnnotationPointcut.read(s, context);
                break;
            case ATWITHINCODE:
                ret = WithinCodeAnnotationPointcut.read(s, context);
                break;
            case ATTHIS_OR_TARGET:
                ret = ThisOrTargetAnnotationPointcut.read(s, context);
                break;
            case ATARGS:
                ret = ArgsAnnotationPointcut.read(s, context);
                break;
            case NONE:
                ret = makeMatchesNothing(RESOLVED);
                break;
            default:
                throw new BCException("unknown kind: " + kind);
        }
        ret.state = RESOLVED;
        ret.pointcutKind = kind;
        return ret;
    }

    public void check(ISourceContext ctx, World world) {
        PoliceExtensionUse pointcutPolice = new PoliceExtensionUse(world, this);
        this.accept(pointcutPolice, null);
        if (pointcutPolice.synchronizationDesignatorEncountered()) {
            world.setSynchronizationPointcutsInUse();
        }
    }

    public static Pointcut fromString(String str) {
        PatternParser parser = new PatternParser(str);
        return parser.parsePointcut();
    }

    static class MatchesNothingPointcut extends Pointcut {

        @Override
        protected Test findResidueInternal(Shadow shadow, ExposedState state) {
            return Literal.FALSE;
        }

        @Override
        public int couldMatchKinds() {
            return Shadow.NO_SHADOW_KINDS_BITS;
        }

        @Override
        public FuzzyBoolean fastMatch(FastMatchInfo type) {
            return FuzzyBoolean.NO;
        }

        @Override
        protected FuzzyBoolean matchInternal(Shadow shadow) {
            return FuzzyBoolean.NO;
        }

        @Override
        public void resolveBindings(IScope scope, Bindings bindings) {
        }

        @Override
        public void postRead(ResolvedType enclosingType) {
        }

        @Override
        public Pointcut concretize1(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings) {
            return makeMatchesNothing(state);
        }

        @Override
        public void write(CompressingDataOutputStream s) throws IOException {
            s.writeByte(NONE);
        }

        @Override
        public String toString() {
            return "";
        }

        @Override
        public Object accept(PatternNodeVisitor visitor, Object data) {
            return visitor.visit(this, data);
        }

        @Override
        public Pointcut parameterizeWith(Map<String, UnresolvedType> typeVariableMap, World w) {
            return this;
        }
    }

    public static Pointcut makeMatchesNothing(State state) {
        Pointcut ret = new MatchesNothingPointcut();
        ret.state = state;
        return ret;
    }

    public void assertState(State state) {
        if (this.state != state) {
            throw new BCException("expected state: " + state + " got: " + this.state);
        }
    }

    public abstract Pointcut parameterizeWith(Map<String, UnresolvedType> typeVariableMap, World w);
}
