package org.aspectj.weaver.patterns;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.util.FuzzyBoolean;
import org.aspectj.weaver.CompressingDataOutputStream;
import org.aspectj.weaver.ISourceContext;
import org.aspectj.weaver.IntMap;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.Shadow;
import org.aspectj.weaver.ShadowMunger;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.VersionedDataInputStream;
import org.aspectj.weaver.WeaverMessages;
import org.aspectj.weaver.World;
import org.aspectj.weaver.ast.Literal;
import org.aspectj.weaver.ast.Test;
import org.aspectj.weaver.ast.Var;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class ThisOrTargetAnnotationPointcut extends NameBindingPointcut {

    private boolean isThis;

    private boolean alreadyWarnedAboutDEoW = false;

    private ExactAnnotationTypePattern annotationTypePattern;

    private String declarationText;

    private static final int thisKindSet;

    private static final int targetKindSet;

    static {
        int thisFlags = Shadow.ALL_SHADOW_KINDS_BITS;
        int targFlags = Shadow.ALL_SHADOW_KINDS_BITS;
        for (int i = 0; i < Shadow.SHADOW_KINDS.length; i++) {
            Shadow.Kind kind = Shadow.SHADOW_KINDS[i];
            if (kind.neverHasThis()) {
                thisFlags -= kind.bit;
            }
            if (kind.neverHasTarget()) {
                targFlags -= kind.bit;
            }
        }
        thisKindSet = thisFlags;
        targetKindSet = targFlags;
    }

    public ThisOrTargetAnnotationPointcut(boolean isThis, ExactAnnotationTypePattern type) {
        super();
        this.isThis = isThis;
        this.annotationTypePattern = type;
        this.pointcutKind = ATTHIS_OR_TARGET;
        buildDeclarationText();
    }

    public ThisOrTargetAnnotationPointcut(boolean isThis, ExactAnnotationTypePattern type, ShadowMunger munger) {
        this(isThis, type);
    }

    public ExactAnnotationTypePattern getAnnotationTypePattern() {
        return annotationTypePattern;
    }

    @Override
    public int couldMatchKinds() {
        return isThis ? thisKindSet : targetKindSet;
    }

    @Override
    public Pointcut parameterizeWith(Map<String, UnresolvedType> typeVariableMap, World w) {
        ExactAnnotationTypePattern newPattern = (ExactAnnotationTypePattern) this.annotationTypePattern.parameterizeWith(typeVariableMap, w);
        if (newPattern.getAnnotationType() instanceof ResolvedType) {
            verifyRuntimeRetention(newPattern.getResolvedAnnotationType());
        }
        ThisOrTargetAnnotationPointcut ret = new ThisOrTargetAnnotationPointcut(isThis, (ExactAnnotationTypePattern) annotationTypePattern.parameterizeWith(typeVariableMap, w));
        ret.copyLocationFrom(this);
        return ret;
    }

    @Override
    public FuzzyBoolean fastMatch(FastMatchInfo info) {
        return FuzzyBoolean.MAYBE;
    }

    @Override
    protected FuzzyBoolean matchInternal(Shadow shadow) {
        if (!couldMatch(shadow)) {
            return FuzzyBoolean.NO;
        }
        ResolvedType toMatchAgainst = (isThis ? shadow.getThisType() : shadow.getTargetType()).resolve(shadow.getIWorld());
        annotationTypePattern.resolve(shadow.getIWorld());
        if (annotationTypePattern.matchesRuntimeType(toMatchAgainst).alwaysTrue()) {
            return FuzzyBoolean.YES;
        } else {
            return FuzzyBoolean.MAYBE;
        }
    }

    public boolean isThis() {
        return isThis;
    }

    @Override
    protected void resolveBindings(IScope scope, Bindings bindings) {
        if (!scope.getWorld().isInJava5Mode()) {
            scope.message(MessageUtil.error(WeaverMessages.format(isThis ? WeaverMessages.ATTHIS_ONLY_SUPPORTED_AT_JAVA5_LEVEL : WeaverMessages.ATTARGET_ONLY_SUPPORTED_AT_JAVA5_LEVEL), getSourceLocation()));
            return;
        }
        annotationTypePattern = (ExactAnnotationTypePattern) annotationTypePattern.resolveBindings(scope, bindings, true);
        if (annotationTypePattern.annotationType == null) {
            return;
        }
        ResolvedType rAnnotationType = (ResolvedType) annotationTypePattern.annotationType;
        if (rAnnotationType.isTypeVariableReference()) {
            return;
        }
        verifyRuntimeRetention(rAnnotationType);
    }

    private void verifyRuntimeRetention(ResolvedType rAnnotationType) {
        if (!(rAnnotationType.isAnnotationWithRuntimeRetention())) {
            IMessage m = MessageUtil.error(WeaverMessages.format(WeaverMessages.BINDING_NON_RUNTIME_RETENTION_ANNOTATION, rAnnotationType.getName()), getSourceLocation());
            rAnnotationType.getWorld().getMessageHandler().handleMessage(m);
        }
    }

    @Override
    protected Pointcut concretize1(ResolvedType inAspect, ResolvedType declaringType, IntMap bindings) {
        if (isDeclare(bindings.getEnclosingAdvice())) {
            if (!alreadyWarnedAboutDEoW) {
                inAspect.getWorld().showMessage(IMessage.ERROR, WeaverMessages.format(WeaverMessages.THIS_OR_TARGET_IN_DECLARE, isThis ? "this" : "target"), bindings.getEnclosingAdvice().getSourceLocation(), null);
                alreadyWarnedAboutDEoW = true;
            }
            return Pointcut.makeMatchesNothing(Pointcut.CONCRETE);
        }
        ExactAnnotationTypePattern newType = (ExactAnnotationTypePattern) annotationTypePattern.remapAdviceFormals(bindings);
        ThisOrTargetAnnotationPointcut ret = new ThisOrTargetAnnotationPointcut(isThis, newType, bindings.getEnclosingAdvice());
        ret.alreadyWarnedAboutDEoW = alreadyWarnedAboutDEoW;
        ret.copyLocationFrom(this);
        return ret;
    }

    @Override
    protected Test findResidueInternal(Shadow shadow, ExposedState state) {
        if (!couldMatch(shadow)) {
            return Literal.FALSE;
        }
        boolean alwaysMatches = match(shadow).alwaysTrue();
        Var var = isThis ? shadow.getThisVar() : shadow.getTargetVar();
        Var annVar = null;
        UnresolvedType annotationType = annotationTypePattern.annotationType;
        if (annotationTypePattern instanceof BindingAnnotationTypePattern) {
            BindingAnnotationTypePattern btp = (BindingAnnotationTypePattern) annotationTypePattern;
            annotationType = btp.annotationType;
            annVar = isThis ? shadow.getThisAnnotationVar(annotationType) : shadow.getTargetAnnotationVar(annotationType);
            if (annVar == null) {
                throw new RuntimeException("Impossible!");
            }
            state.set(btp.getFormalIndex(), annVar);
        }
        if (alwaysMatches && (annVar == null)) {
            return Literal.TRUE;
        } else {
            ResolvedType rType = annotationType.resolve(shadow.getIWorld());
            return Test.makeHasAnnotation(var, rType);
        }
    }

    private boolean couldMatch(Shadow shadow) {
        return isThis ? shadow.hasThis() : shadow.hasTarget();
    }

    @Override
    public List<BindingPattern> getBindingAnnotationTypePatterns() {
        if (annotationTypePattern instanceof BindingAnnotationTypePattern) {
            List<BindingPattern> l = new ArrayList<>();
            l.add((BindingPattern) annotationTypePattern);
            return l;
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public List<BindingTypePattern> getBindingTypePatterns() {
        return Collections.emptyList();
    }

    @Override
    public void write(CompressingDataOutputStream s) throws IOException {
        s.writeByte(Pointcut.ATTHIS_OR_TARGET);
        s.writeBoolean(isThis);
        annotationTypePattern.write(s);
        writeLocation(s);
    }

    public static Pointcut read(VersionedDataInputStream s, ISourceContext context) throws IOException {
        boolean isThis = s.readBoolean();
        AnnotationTypePattern type = AnnotationTypePattern.read(s, context);
        ThisOrTargetAnnotationPointcut ret = new ThisOrTargetAnnotationPointcut(isThis, (ExactAnnotationTypePattern) type);
        ret.readLocation(context, s);
        return ret;
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof ThisOrTargetAnnotationPointcut)) {
            return false;
        }
        ThisOrTargetAnnotationPointcut other = (ThisOrTargetAnnotationPointcut) obj;
        return (other.annotationTypePattern.equals(this.annotationTypePattern) && (other.isThis == this.isThis));
    }

    @Override
    public int hashCode() {
        return 17 + 37 * annotationTypePattern.hashCode() + (isThis ? 49 : 13);
    }

    private void buildDeclarationText() {
        StringBuilder buf = new StringBuilder();
        buf.append(isThis ? "@this(" : "@target(");
        String annPatt = annotationTypePattern.toString();
        buf.append(annPatt.startsWith("@") ? annPatt.substring(1) : annPatt);
        buf.append(")");
        this.declarationText = buf.toString();
    }

    @Override
    public String toString() {
        return this.declarationText;
    }

    @Override
    public Object accept(PatternNodeVisitor visitor, Object data) {
        return visitor.visit(this, data);
    }
}
