package org.aspectj.weaver.reflect;

import org.aspectj.weaver.BoundedReferenceType;
import org.aspectj.weaver.ReferenceType;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.TypeFactory;
import org.aspectj.weaver.TypeVariable;
import org.aspectj.weaver.TypeVariableReferenceType;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.World;

import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.WildcardType;
import java.util.HashMap;
import java.util.Map;

public class JavaLangTypeToResolvedTypeConverter {

    private Map<Type, TypeVariableReferenceType> typeVariablesInProgress = new HashMap<>();

    private final World world;

    public JavaLangTypeToResolvedTypeConverter(World aWorld) {
        this.world = aWorld;
    }

    private World getWorld() {
        return this.world;
    }

    public ResolvedType fromType(Type type) {
        if (type instanceof Class) {
            Class<?> clazz = (Class<?>) type;
            String name = clazz.getName();
            if (clazz.isArray()) {
                UnresolvedType ut = UnresolvedType.forSignature(name.replace('.', '/'));
                return getWorld().resolve(ut);
            } else {
                return getWorld().resolve(name);
            }
        } else if (type instanceof ParameterizedType) {
            Type ownerType = ((ParameterizedType) type).getOwnerType();
            ParameterizedType parameterizedType = (ParameterizedType) type;
            ResolvedType baseType = fromType(parameterizedType.getRawType());
            Type[] typeArguments = parameterizedType.getActualTypeArguments();
            if (baseType.isSimpleType() && typeArguments.length == 0 && ownerType != null) {
                return baseType;
            }
            ResolvedType[] resolvedTypeArguments = fromTypes(typeArguments);
            return TypeFactory.createParameterizedType(baseType, resolvedTypeArguments, getWorld());
        } else if (type instanceof java.lang.reflect.TypeVariable) {
            TypeVariableReferenceType inprogressVar = typeVariablesInProgress.get(type);
            if (inprogressVar != null) {
                return inprogressVar;
            }
            java.lang.reflect.TypeVariable<?> tv = (java.lang.reflect.TypeVariable<?>) type;
            TypeVariable rt_tv = new TypeVariable(tv.getName());
            TypeVariableReferenceType tvrt = new TypeVariableReferenceType(rt_tv, getWorld());
            typeVariablesInProgress.put(type, tvrt);
            Type[] bounds = tv.getBounds();
            ResolvedType[] resBounds = fromTypes(bounds);
            ResolvedType upperBound = resBounds[0];
            ResolvedType[] additionalBounds = ResolvedType.EMPTY_RESOLVED_TYPE_ARRAY;
            if (resBounds.length > 1) {
                additionalBounds = new ResolvedType[resBounds.length - 1];
                System.arraycopy(resBounds, 1, additionalBounds, 0, additionalBounds.length);
            }
            rt_tv.setUpperBound(upperBound);
            rt_tv.setAdditionalInterfaceBounds(additionalBounds);
            typeVariablesInProgress.remove(type);
            return tvrt;
        } else if (type instanceof WildcardType) {
            WildcardType wildType = (WildcardType) type;
            Type[] lowerBounds = wildType.getLowerBounds();
            Type[] upperBounds = wildType.getUpperBounds();
            ResolvedType bound = null;
            boolean isExtends = lowerBounds.length == 0;
            if (isExtends) {
                bound = fromType(upperBounds[0]);
            } else {
                bound = fromType(lowerBounds[0]);
            }
            return new BoundedReferenceType((ReferenceType) bound, isExtends, getWorld());
        } else if (type instanceof GenericArrayType) {
            GenericArrayType genericArrayType = (GenericArrayType) type;
            Type componentType = genericArrayType.getGenericComponentType();
            return UnresolvedType.makeArray(fromType(componentType), 1).resolve(getWorld());
        }
        return ResolvedType.MISSING;
    }

    public ResolvedType[] fromTypes(Type[] types) {
        ResolvedType[] ret = new ResolvedType[types.length];
        for (int i = 0; i < ret.length; i++) {
            ret[i] = fromType(types[i]);
        }
        return ret;
    }
}
