package org.aspectj.weaver.reflect;

import org.aspectj.bridge.AbortException;
import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.IMessageHandler;
import org.aspectj.weaver.BCException;
import org.aspectj.weaver.IWeavingSupport;
import org.aspectj.weaver.ReferenceType;
import org.aspectj.weaver.ReferenceTypeDelegate;
import org.aspectj.weaver.ResolvedType;
import org.aspectj.weaver.UnresolvedType;
import org.aspectj.weaver.WeakClassLoaderReference;
import org.aspectj.weaver.World;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public class ReflectionWorld extends World implements IReflectionWorld {

    private static Map<WeakClassLoaderReference, ReflectionWorld> rworlds = Collections.synchronizedMap(new HashMap<>());

    private WeakClassLoaderReference classLoaderReference;

    private AnnotationFinder annotationFinder;

    private boolean mustUseOneFourDelegates = false;

    private Map<String, Class<?>> inProgressResolutionClasses = new HashMap<>();

    public static ReflectionWorld getReflectionWorldFor(WeakClassLoaderReference classLoaderReference) {
        return new ReflectionWorld(classLoaderReference);
    }

    public static void cleanUpWorlds() {
        synchronized (rworlds) {
            rworlds.clear();
        }
    }

    private ReflectionWorld() {
    }

    public ReflectionWorld(WeakClassLoaderReference classloaderRef) {
        this.setMessageHandler(new ExceptionBasedMessageHandler());
        setBehaveInJava5Way(true);
        classLoaderReference = classloaderRef;
        annotationFinder = makeAnnotationFinderIfAny(classLoaderReference.getClassLoader(), this);
    }

    public ReflectionWorld(ClassLoader aClassLoader) {
        super();
        this.setMessageHandler(new ExceptionBasedMessageHandler());
        setBehaveInJava5Way(true);
        classLoaderReference = new WeakClassLoaderReference(aClassLoader);
        annotationFinder = makeAnnotationFinderIfAny(classLoaderReference.getClassLoader(), this);
    }

    public ReflectionWorld(boolean forceUseOf14Delegates, ClassLoader aClassLoader) {
        this(aClassLoader);
        this.mustUseOneFourDelegates = forceUseOf14Delegates;
        if (forceUseOf14Delegates) {
            this.setBehaveInJava5Way(false);
        }
    }

    public static AnnotationFinder makeAnnotationFinderIfAny(ClassLoader loader, World world) {
        AnnotationFinder annotationFinder = null;
        try {
            Class<?> java15AnnotationFinder = Class.forName("org.aspectj.weaver.reflect.Java15AnnotationFinder");
            annotationFinder = (AnnotationFinder) java15AnnotationFinder.getDeclaredConstructor().newInstance();
            annotationFinder.setClassLoader(loader);
            annotationFinder.setWorld(world);
        } catch (ClassNotFoundException ex) {
        } catch (IllegalAccessException | InstantiationException | NoSuchMethodException ex) {
            throw new BCException("AspectJ internal error", ex);
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        return annotationFinder;
    }

    public ClassLoader getClassLoader() {
        return classLoaderReference.getClassLoader();
    }

    public AnnotationFinder getAnnotationFinder() {
        return annotationFinder;
    }

    public ResolvedType resolve(Class aClass) {
        return resolve(this, aClass);
    }

    public static ResolvedType resolve(World world, Class<?> aClass) {
        String className = aClass.getName();
        if (aClass.isArray()) {
            return world.resolve(UnresolvedType.forSignature(className.replace('.', '/')));
        } else {
            return world.resolve(className);
        }
    }

    public ResolvedType resolveUsingClass(Class<?> clazz) {
        String signature = UnresolvedType.forName(clazz.getName()).getSignature();
        try {
            inProgressResolutionClasses.put(signature, clazz);
            return resolve(clazz.getName());
        } finally {
            inProgressResolutionClasses.remove(signature);
        }
    }

    protected ReferenceTypeDelegate resolveDelegate(ReferenceType ty) {
        ReferenceTypeDelegate result;
        if (mustUseOneFourDelegates) {
            result = ReflectionBasedReferenceTypeDelegateFactory.create14Delegate(ty, this, classLoaderReference.getClassLoader());
        } else {
            result = ReflectionBasedReferenceTypeDelegateFactory.createDelegate(ty, this, classLoaderReference.getClassLoader());
        }
        if (result == null && inProgressResolutionClasses.size() != 0) {
            Class<?> clazz = inProgressResolutionClasses.get(ty.getSignature());
            if (clazz != null) {
                result = ReflectionBasedReferenceTypeDelegateFactory.createDelegate(ty, this, clazz);
            }
        }
        return result;
    }

    public static class ReflectionWorldException extends RuntimeException {

        private static final long serialVersionUID = -3432261918302793005L;

        public ReflectionWorldException(String message) {
            super(message);
        }
    }

    private static class ExceptionBasedMessageHandler implements IMessageHandler {

        public boolean handleMessage(IMessage message) throws AbortException {
            throw new ReflectionWorldException(message.toString());
        }

        public boolean isIgnoring(org.aspectj.bridge.IMessage.Kind kind) {
            if (kind == IMessage.INFO) {
                return true;
            } else {
                return false;
            }
        }

        public void dontIgnore(org.aspectj.bridge.IMessage.Kind kind) {
        }

        public void ignore(org.aspectj.bridge.IMessage.Kind kind) {
        }
    }

    public IWeavingSupport getWeavingSupport() {
        return null;
    }

    public boolean isLoadtimeWeaving() {
        return true;
    }
}
