package org.aspectj.weaver.tools.cache;

import org.aspectj.util.FileUtil;
import org.aspectj.util.LangUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.StreamCorruptedException;
import java.util.Map;
import java.util.TreeMap;

public class FlatFileCacheBacking extends AsynchronousFileCacheBacking {

    private static final AsynchronousFileCacheBackingCreator<FlatFileCacheBacking> defaultCreator = new AsynchronousFileCacheBackingCreator<FlatFileCacheBacking>() {

        public FlatFileCacheBacking create(File cacheDir) {
            return new FlatFileCacheBacking(cacheDir);
        }
    };

    public FlatFileCacheBacking(File cacheDir) {
        super(cacheDir);
    }

    public static final FlatFileCacheBacking createBacking(File cacheDir) {
        return createBacking(cacheDir, defaultCreator);
    }

    @Override
    protected Map<String, byte[]> readClassBytes(Map<String, IndexEntry> indexMap, File cacheDir) {
        return readClassBytes(indexMap, cacheDir.listFiles());
    }

    protected Map<String, byte[]> readClassBytes(Map<String, IndexEntry> indexMap, File[] files) {
        Map<String, byte[]> result = new TreeMap<>();
        if (LangUtil.isEmpty(files)) {
            return result;
        }
        for (File file : files) {
            if (!file.isFile()) {
                continue;
            }
            String key = file.getName();
            if (INDEX_FILE.equalsIgnoreCase(key)) {
                continue;
            }
            IndexEntry entry = indexMap.get(key);
            if ((entry == null) || entry.ignored) {
                if ((logger != null) && logger.isTraceEnabled()) {
                    logger.info("readClassBytes(" + key + ") remove orphan/ignored: " + file.getAbsolutePath());
                }
                FileUtil.deleteContents(file);
                continue;
            }
            try {
                byte[] bytes = FileUtil.readAsByteArray(file);
                long crc = crc(bytes);
                if (crc != entry.crcWeaved) {
                    throw new StreamCorruptedException("Mismatched CRC - expected=" + entry.crcWeaved + "/got=" + crc);
                }
                result.put(key, bytes);
                if ((logger != null) && logger.isTraceEnabled()) {
                    logger.debug("readClassBytes(" + key + ") cached from " + file.getAbsolutePath());
                }
            } catch (IOException e) {
                if ((logger != null) && logger.isTraceEnabled()) {
                    logger.error("Failed (" + e.getClass().getSimpleName() + ")" + " to read bytes from " + file.getAbsolutePath() + ": " + e.getMessage());
                }
                indexMap.remove(key);
                FileUtil.deleteContents(file);
                continue;
            }
        }
        return result;
    }

    @Override
    protected IndexEntry resolveIndexMapEntry(File cacheDir, IndexEntry ie) {
        File cacheEntry = new File(cacheDir, ie.key);
        if (ie.ignored || cacheEntry.canRead()) {
            return ie;
        } else {
            return null;
        }
    }

    @Override
    protected void writeClassBytes(String key, byte[] bytes) throws Exception {
        File dir = getCacheDirectory(), file = new File(dir, key);
        FileOutputStream out = new FileOutputStream(file);
        try {
            out.write(bytes);
        } finally {
            out.close();
        }
    }

    @Override
    protected void removeClassBytes(String key) throws Exception {
        File dir = getCacheDirectory(), file = new File(dir, key);
        if (file.exists() && (!file.delete())) {
            throw new StreamCorruptedException("Failed to delete " + file.getAbsolutePath());
        }
    }
}
