package org.aspectj.weaver.tools.cache;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.IMessageHandler;
import org.aspectj.bridge.Message;
import org.aspectj.bridge.MessageUtil;
import org.aspectj.weaver.tools.GeneratedClassHandler;

import java.util.LinkedList;
import java.util.List;

public class WeavedClassCache {

    public static final String WEAVED_CLASS_CACHE_ENABLED = "aj.weaving.cache.enabled";

    public static final String CACHE_IMPL = SimpleCacheFactory.CACHE_IMPL;

    private static CacheFactory DEFAULT_FACTORY = new DefaultCacheFactory();

    public static final byte[] ZERO_BYTES = new byte[0];

    private final IMessageHandler messageHandler;

    private final GeneratedCachedClassHandler cachingClassHandler;

    private final CacheBacking backing;

    private final CacheStatistics stats;

    private final CacheKeyResolver resolver;

    private final String name;

    private static final List<WeavedClassCache> cacheRegistry = new LinkedList<>();

    protected WeavedClassCache(GeneratedClassHandler existingClassHandler, IMessageHandler messageHandler, String name, CacheBacking backing, CacheKeyResolver resolver) {
        this.resolver = resolver;
        this.name = name;
        this.backing = backing;
        this.messageHandler = messageHandler;
        cachingClassHandler = new GeneratedCachedClassHandler(this, existingClassHandler);
        this.stats = new CacheStatistics();
        synchronized (cacheRegistry) {
            cacheRegistry.add(this);
        }
    }

    public static WeavedClassCache createCache(ClassLoader loader, List<String> aspects, GeneratedClassHandler existingClassHandler, IMessageHandler messageHandler) {
        CacheKeyResolver resolver = DEFAULT_FACTORY.createResolver();
        String name = resolver.createClassLoaderScope(loader, aspects);
        if (name == null) {
            return null;
        }
        CacheBacking backing = DEFAULT_FACTORY.createBacking(name);
        if (backing != null) {
            return new WeavedClassCache(existingClassHandler, messageHandler, name, backing, resolver);
        }
        return null;
    }

    public String getName() {
        return name;
    }

    public static void setDefaultCacheFactory(CacheFactory factory) {
        DEFAULT_FACTORY = factory;
    }

    public CachedClassReference createGeneratedCacheKey(String className) {
        return resolver.generatedKey(className);
    }

    public CachedClassReference createCacheKey(String className, byte[] originalBytes) {
        return resolver.weavedKey(className, originalBytes);
    }

    public GeneratedClassHandler getCachingClassHandler() {
        return cachingClassHandler;
    }

    public static boolean isEnabled() {
        String enabled = System.getProperty(WEAVED_CLASS_CACHE_ENABLED);
        String impl = System.getProperty(CACHE_IMPL);
        return (enabled != null && (impl == null || !SimpleCache.IMPL_NAME.equalsIgnoreCase(impl)));
    }

    public void put(CachedClassReference ref, byte[] classBytes, byte[] weavedBytes) {
        CachedClassEntry.EntryType type = CachedClassEntry.EntryType.WEAVED;
        if (ref.getKey().matches(resolver.getGeneratedRegex())) {
            type = CachedClassEntry.EntryType.GENERATED;
        }
        backing.put(new CachedClassEntry(ref, weavedBytes, type), classBytes);
        stats.put();
    }

    public CachedClassEntry get(CachedClassReference ref, byte[] classBytes) {
        CachedClassEntry entry = backing.get(ref, classBytes);
        if (entry == null) {
            stats.miss();
        } else {
            stats.hit();
            if (entry.isGenerated())
                stats.generated();
            if (entry.isWeaved())
                stats.weaved();
            if (entry.isIgnored())
                stats.ignored();
        }
        return entry;
    }

    public void ignore(CachedClassReference ref, byte[] classBytes) {
        stats.putIgnored();
        backing.put(new CachedClassEntry(ref, ZERO_BYTES, CachedClassEntry.EntryType.IGNORED), classBytes);
    }

    public void remove(CachedClassReference ref) {
        backing.remove(ref);
    }

    public void clear() {
        backing.clear();
    }

    public CacheStatistics getStats() {
        return stats;
    }

    public static List<WeavedClassCache> getCaches() {
        synchronized (cacheRegistry) {
            return new LinkedList<>(cacheRegistry);
        }
    }

    protected void error(String message, Throwable th) {
        messageHandler.handleMessage(new Message(message, IMessage.ERROR, th, null));
    }

    protected void error(String message) {
        MessageUtil.error(messageHandler, message);
    }

    protected void info(String message) {
        MessageUtil.info(message);
    }
}
