package com.jpro.webapi;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.HashSet;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * Represents a JavaScript variable managed within a Java environment. This class provides mechanisms to interact
 * with JavaScript variables via a WebAPI and manage their lifecycle, including automatic cleanup using a cleanup
 * function upon garbage collection.
 */
public class JSVariable {

    WebAPI webAPI;
    String name;
    CompletableFuture<Void> future;

    /**
     * Constructs a new JSVariable with the specified WebAPI and variable name.
     *
     * @param webAPI the {@link WebAPI} object through which the JavaScript will be executed
     * @param name   the name of the JavaScript variable
     */
    public JSVariable(WebAPI webAPI, String name) {
        this(webAPI, name, name + " = undefined;");
    }

    /**
     * Constructs a new JSVariable with the specified WebAPI, variable name, and cleanup script.
     *
     * @param webAPI  the {@link WebAPI} object through which the JavaScript will be executed
     * @param name    the name of the JavaScript variable
     * @param cleanup JavaScript code to be executed for cleaning up the variable
     */
    public JSVariable(WebAPI webAPI, String name, String cleanup) {
        this(webAPI, name, cleanup, CompletableFuture.completedFuture(null));
    }

    /**
     * Constructs a JSVariable with the specified WebAPI, variable name, cleanup script,
     * and future representing the completion of the variable.
     *
     * @param webAPI  the {@link WebAPI} object through which the JavaScript will be executed
     * @param name    the name of the JavaScript variable
     * @param cleanup JavaScript code to be executed for cleaning up the variable
     * @param future  a {@link CompletableFuture} which will be completed when the variable is set to a value.
     *                If an error occurs, the future will be completed exceptionally.
     */
    public JSVariable(WebAPI webAPI, String name, String cleanup, CompletableFuture<Void> future) {
        this.webAPI = webAPI;
        this.name = name;
        this.future = future;
        onCleanup(this, new CleanupFunction(name, cleanup, webAPI));
    }

    private static class CleanupFunction implements Runnable {
        String name;
        String code;
        WeakReference<WebAPI> webAPI;

        private CleanupFunction(String name, String code, WebAPI webAPI) {
            this.name = name;
            this.code = code;
            this.webAPI = new WeakReference(webAPI);
        }

        @Override
        public void run() {
            WebAPI realWebAPI = webAPI.get();
            if (realWebAPI != null) {
                realWebAPI.executeScript(code);
            }
        }
    }


    /**
     * Executes a script to get the string representation of the JavaScript variable.
     *
     * @return a {@link CompletableFuture}.
     */
    public CompletableFuture<String> getString() {
        return webAPI.executeScriptWithFuture(getName() + ";");
    }

    /**
     * Returns a Future, which will be completed when the JSVariable is set to a value.
     * The Future doesn't contain the value itself.
     * But if something went wrong, the Future will be completed exceptionally.
     *
     * @return A Future, which will be completed when the JSVariable is set to a value.
     */
    public CompletableFuture<Void> onComplete() {
        return future;
    }

    /**
     * This method registers an error handler which will be called when an error occurs.
     *
     * @param onError The error handler which will be called when an error occurs.
     */
    public void onError(Consumer<Throwable> onError) {
        future.exceptionally(e -> {
            onError.accept(e);
            return null;
        });
    }

    /**
     * Gets the WebAPI associated with this JSVariable.
     *
     * @return the {@link WebAPI} object
     */
    public WebAPI getWebAPI() {
        return webAPI;
    }

    /**
     * Gets the code to access the JavaScript variable.
     * This name can be used in JavaScript code to refer to the variable.
     *
     * @return the code to access this variable
     */
    public String getName() {
        return name;
    }

    private static final HashSet<WeakReferenceWithRunnable> references = new HashSet<>();
    private static final ReferenceQueue<?> queue = new ReferenceQueue<>();

    static {
        Thread cleanupDetectorThread = new Thread(() -> {
            while (true) {
                try {
                    WeakReferenceWithRunnable r = (WeakReferenceWithRunnable) queue.remove();
                    references.remove(r);
                    r.r.run();
                } catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }, "JPro-JSVariable-cleanup-detector");
        cleanupDetectorThread.setDaemon(true);
        cleanupDetectorThread.start();
    }

    private static class WeakReferenceWithRunnable extends WeakReference {
        Runnable r = null;

        WeakReferenceWithRunnable(Object ref, Runnable r) {
            super(ref, queue);
            this.r = r;
        }
    }

    private static void onCleanup(Object obj, Runnable r) {
        onCleanup(new WeakReferenceWithRunnable(obj, r));
    }

    private static void onCleanup(WeakReferenceWithRunnable weakref) {
        references.add(weakref);
    }

    /**
     * This method converts a CompletableFuture&lt;JSVariable&gt; to a Promise which can be used in JS.
     * The Promise will be resolved when the CompletableFuture is resolved.
     *
     * @param webAPI The WebAPI to use
     * @param future The CompletableFuture to convert
     * @return a {@link JSVariable} containing a Promise which will be resolved when the CompletableFuture is resolved.
     * @since 2023.3.2
     */
    public static JSVariable futureToPromise(WebAPI webAPI, CompletableFuture<JSVariable> future) {
        JSVariable var1 = webAPI.executeScriptWithVariable("(function() {" +
                "var myresolve; " +
                "var myPromise = new Promise(function(resolve, reject) { " +
                "    myresolve = resolve;" +
                "}); " +
                "return {resolve: myresolve, promise: myPromise};" +
                "})();");
        future.thenAccept(var2 -> {
            var2.webAPI.executeScript(var1.getName() + ".resolve(" + var2.getName() + ");");
        });
        return webAPI.executeScriptWithVariable(var1.getName() + ".promise;");
    }


    /**
     * This method checks if the JSVariable is a Promise.
     *
     * @return a {@link CompletableFuture} which will be resolved with <code>true</code>
     * if the JSVariable is a Promise and <code>false</code> if not.
     * @since 2023.3.4
     */
    public CompletableFuture<Boolean> isPromise() {
        return webAPI.executeScriptWithFuture(name + " instanceof Promise;")
                .thenApply(v -> {
                    if (v.equals("true")) {
                        return true;
                    } else if (v.equals("false")) {
                        return false;
                    } else {
                        throw new RuntimeException("Unexpected result: " + v);
                    }
                });
    }

    /**
     * This method converts a JSVariable of a promise to a CompletableFuture&lt;JSVariable&gt;
     * which will be resolved when the promise is resolved.
     *
     * @param webAPI The WebAPI to use
     * @param myvar  The JSVariable to convert
     * @return A CompletableFuture which will be resolved when the promise is resolved
     * @since 2023.3.2
     */
    public static CompletableFuture<JSVariable> promiseToFuture(WebAPI webAPI, JSVariable myvar) {
        return webAPI.js().evalFuture("return await " + myvar.getName() + ";");
    }

    /**
     * Wraps the JSVariable in a PromiseJSVariable, which provides additional promise-specific functionalities.
     *
     * @return a {@link PromiseJSVariable} object
     */
    public PromiseJSVariable toPromise() {
        return new PromiseJSVariable(this);
    }
}
