package com.jpro.webapi;

import com.jpro.webapi.annotation.Experimental;
import com.jpro.webapi.js.JS;
import javafx.beans.property.*;
import javafx.collections.ObservableList;
import javafx.collections.ObservableMap;
import javafx.geometry.Rectangle2D;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.SelectionMode;
import javafx.scene.image.Image;
import javafx.stage.Stage;
import javafx.stage.Window;

import java.io.File;
import java.net.URI;
import java.net.URL;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * <p>Used to interface with the browser.</p>
 * <p>
 * JPro Documentation: <a href="https://www.jpro.one/">JPro</a><br>
 * JPro Link: <a href="https://www.jpro.one/">JPro</a><br>
 * HelloWorld: <a href="https://github.com/JPro-one/HelloJPro">HelloWorld</a><p>
 * <p>
 * The WebAPI is typically used for the following use-cases:
 * <ul>
 * <li>Checking, whether this session is currently running in a browser, with the method {@link #isBrowser()}
 * <li>Executing javascript with the method {@link #executeScript(String)} or {@link #executeScriptWithListener(String, ScriptResultListener)}
 * <li>Register values in the browser with the method {@link #registerValue(String, Node)}
 * <li>Registering a java function which can be called from the browser with javascript. This can be done with the method {@link #registerJavaFunction(String, WebCallback)} ()}
 * <li>Handle FileEvents with the method {@link #makeFileUploadNode(Node)}
 * <li>Get information about the browser with {@link #getBrowserSize()}, {@link #isMobile()}, {@link #getCookies()}, {@link #getLanguage()} or {@link #getLanguages()}
 * <li>Open a new Stage with {@link #openStageAsTab(Stage)} or {@link #openStageAsPopup(Stage)}
 * </ul>
 * <br>
 * <b>Example:</b><br>
 *
 * <p>The webAPI is usually accessed the following way:</p>
 *
 * <pre>if(WebAPI.isBrowser()) {
 *    WebAPI webAPI = WebAPI.getWebAPI(stage);
 *    webAPI.executeScript(*javascript-code*)
 * }</pre>
 * <p>
 *  @author Florian Kirmaier
 */
public abstract class WebAPI {

    private static Class<?> webAPIImplClass = null;
    private static BiConsumer<Node, WebAPIConsumer> getWebAPIImpl = null;
    private static WebAPIStatic webAPIStaticImpl = null;

    /**
     * Returns <code>true</code> if the implementation is for a browser.
     * An alternative way to check isBrowser is checking the property <code>Boolean.getBoolean("jpro.isbrowser")</code>
     *
     * @return <code>true</code> if implementation for browser; <code>false</code> otherwise.
     */
    public static boolean isBrowser() {
        return webAPIImplClass != null;
    }

    /**
     * Returns the WebAPI implementation interface for the platform the application is currently running on.
     *
     * @param window The window, which is associated with a Session.
     * @return An instance of WebAPI which can be used to interface with the browser.
     * @throws RuntimeException if <code>isBrowser</code> is <code>false</code>
     */
    public static WebAPI getWebAPI(Window window) throws RuntimeException {
        if (!isBrowser()) {
            throw new RuntimeException("createWebAPI is not implemented on this Platform!");
        }
        try {
            java.lang.reflect.Method method = webAPIImplClass.getMethod("getWebAPI", Window.class);
            return (WebAPI) method.invoke(null, window);
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException("Unable to create the WebAPI!", e);
        }
    }

    /**
     * Returns the WebAPI implementation interface for the platform the application is currently running on.
     *
     * @param scene The scene, which is associated with a Session.
     * @return An instance of WebAPI which can be used to interface with the browser.
     * @throws RuntimeException if <code>isBrowser</code> is <code>false</code>
     */
    public static WebAPI getWebAPI(Scene scene) throws RuntimeException {
        if (scene.getWindow() == null) {
            throw new RuntimeException("getWebAPI: The window of the provided scene is null!");
        }
        return getWebAPI(scene.getWindow());
    }

    /**
     * Returns the WebAPI implementation interface for the platform the application is currently running on.
     *
     * @param node     A node, which gets added to a scene which is associated with a Session.
     * @param consumer Gets executed, when the scene of the node is set.
     * @since 2019.1.0
     */
    public static void getWebAPI(Node node, WebAPIConsumer consumer) {
        getWebAPIImpl.accept(node, consumer);
    }

    /**
     * Internal method. Do not use.
     *
     * @param clazz The class of the WebAPI implementation.
     */
    public static void setWebAPIImpl(Class<?> clazz, BiConsumer<Node, WebAPIConsumer> _getWebAPIImpl, WebAPIStatic staticImpl) {
        webAPIImplClass = clazz;
        getWebAPIImpl = _getWebAPIImpl;
        webAPIStaticImpl = staticImpl;
    }

    /**
     * Returns information about this running instance.
     *
     * @return an {@link InstanceInfo} object containing instance information about this session.
     */
    public abstract InstanceInfo getInstanceInfo();

    /**
     * Returns the url, which was used to call this session.
     *
     * @return The name of the server on which this session is currently running.
     * @deprecated Use {@link #getBrowserURL()} instead.
     */
    @Deprecated
    public String getServerName() {
        return getBrowserURL();
    }

    /**
     * Returns the URL, which was used to call this session.
     *
     * @return The URL string on which this session is currently running.
     */
    public abstract String getBrowserURL();

    /**
     * Sets a new cookie in the current browser session.
     * After this method is called, the map returned by {@link #getCookies()} will be updated to reflect the new cookie.
     * The cookie will not expire unless manually removed.
     *
     * @param key   The name of the cookie to set.
     * @param value The value of the cookie.
     */
    public abstract void setCookie(String key, String value);

    /**
     * Deletes a cookie from the current browser session.
     * After this method is called, the map returned by {@link #getCookies()} will be updated to reflect the removal of
     * the cookie.
     *
     * @param key The name of the cookie to delete.
     */
    public abstract void deleteCookie(String key);

    /**
     * Retrieves the cookies of the current browser window.
     * The returned cookies are identical to the cookies accessible via {@code document.cookie} in the browser.
     *
     * @return An observable map of the browser window's cookies.
     */
    public abstract ObservableMap<String, String> getCookies();

    /**
     * Retrieves the cookies associated with the WebSocket connection.
     * The content of these cookies may differ from those retrieved by {@link #getCookies()} due to different domains.
     *
     * @return A map containing the WebSocket connection's cookies.
     */
    public abstract Map<String, String> getWebsocketCookies();

    /**
     * Determines whether the current session is running on a mobile device (Android or iOS).
     *
     * @return {@code true} if the session is running on a mobile device, {@code false} otherwise.
     */
    public abstract boolean isMobile();

    /**
     * Retrieves the preferred language of the currently hosting browser.
     *
     * @return The preferred language of the hosting browser.
     */
    public abstract String getLanguage();

    /**
     * Retrieves the list of preferred languages of the currently hosting browser.
     * The first element in the list represents the language with the highest priority.
     *
     * @return A list of preferred languages of the hosting browser.
     */
    public abstract List<String> getLanguages();

    /**
     * Retrieves the String returned by the `navigator.platform` property.
     * This JavaScript property is deprecated but widely supported.
     * If it is not available, this method returns `null`.
     * It is recommended to use `getPlatform()` and `getPlatformOld()` together for compatibility.
     * A more elegant API is provided in the jpro-platform library.
     *
     * @return the value of the `navigator.platform` property, or `null` if unavailable.
     */
    public abstract String getPlatformOld();

    /**
     * Retrieves the String returned by the `navigator.userAgentData.platform` property.
     * This JavaScript property is recommended but not widely supported.
     * If it is not available, this method returns `null`.
     * It is recommended to use `getPlatform()` and `getPlatformOld()` together for compatibility.
     * A more elegant API is provided in the jpro-platform library.
     *
     * @return the value of the `navigator.userAgentData.platform` property, or `null` if unavailable.
     */
    public abstract String getPlatform();

    /**
     * Retrieves the preferred locale of the currently hosting browser.
     *
     * @return The locale of the hosting browser.
     */
    public abstract Locale getLocale();

    /**
     * Retrieves the preferred time zone of the currently hosting browser.
     *
     * @return The time zone of the hosting browser.
     */
    public abstract TimeZone getTimeZone();

    /**
     * Retrieves all query parameters from the URL that was used to create the current session.
     *
     * @return A map containing the query parameters and their values from the session URL.
     *         If no query parameters are present, an empty map is returned.
     */
    public Map<String, String> getURLQueryParams() {
        String url = getBrowserURL();
        Map<String, String> result = new HashMap<>();

        URI uri = URI.create(url);

        if (uri.getQuery() == null) {
            return result;
        }

        for (String param : uri.getQuery().split("&")) {
            if (param != null) {
                String[] pair = param.split("=");
                if (pair.length > 1) {
                    result.put(pair[0], pair[1]);
                } else {
                    result.put(pair[0], "");
                }
            }
        }
        return result;
    }

    /**
     * Retrieves all the headers provided to this session.
     *
     * @return A map containing all the headers for this session.
     */
    public abstract Map<String, List<String>> getHeaders();

    /**
     * Retrieves the window size of the currently hosting browser.
     *
     * @return The size of the hosting browser window as a {@link Rectangle2D} object.
     */
    public abstract Rectangle2D getBrowserSize();

    /**
     * A boolean property that changes value when the dark mode is activated or deactivated.
     */
    public abstract ReadOnlyBooleanProperty darkMode();

    /**
     * Retrieves the current value of the property {@link #darkMode()}.
     *
     * @return {@code true} if dark mode is active, {@code false} otherwise.
     */
    public abstract boolean isDarkMode();

    /**
     * Retrieves the device pixel ratio in the current browser tab.
     * The value can change when the tab is moved to a different screen or when the browser's zoom level is adjusted.
     */
    public abstract ReadOnlyDoubleProperty devicePixelRatio();

    /**
     * Retrieves the current value of the device pixel ratio in the current browser tab.
     *
     * @return The current device pixel ratio as a {@code double}.
     */
    public abstract double getDevicePixelRatio();

    /**
     * Returns the ID of the current session.
     *
     * @return an ID string which represents the current session.
     */
    public abstract String getInstanceID();

    /**
     * Returns the server address used to instantiate the currently running session. It hast the form: "host:port".
     *
     * @return the server address string.
     */
    public abstract String getServer();

    /**
     * Adds a listener to this WebAPI, to get triggered as this session is closing.
     * If this session is already closed, the listener gets triggered immediately.
     *
     * @param listener The listener to add.
     */
    public abstract void addInstanceCloseListener(InstanceCloseListener listener);

    /**
     * Removes a listener from this WebAPI.  This listener will no longer get triggered when the session gets closed.
     *
     * @param listener The listener to be removed.
     */
    public abstract void removeInstanceCloseListener(InstanceCloseListener listener);

    /**
     * Loads and executes the js file in the currently hosting browser.
     * Each url is only loaded once, even when called multiple times.
     *
     * @param url The url of the JavaScript file, which gets loaded. The js file must be accessible from the server.
     */
    public abstract void loadJSFile(URL url);

    /**
     * Loads a CSS file into the browser.
     * Each url is only loaded once, even when called multiple times.
     *
     * @param url The url of the CSS file, which gets loaded. The CSS file must be accessible from the server.
     */
    public abstract void loadCSSFile(URL url);

    /**
     * Returns as JS object, with which JavaScript code can be executed.
     * @return a JS object, with which JavaScript code can be executed.
     */
    public abstract JS js();

    /**
     * Executes JavaScript <code>code</code> in the currently hosting browser.
     * use {@link JS#eval(String)} instead.
     *
     * @param code The code to be executed.
     */
    @Deprecated
    public abstract void executeScript(String code);

    /**
     * Executes Javascript <code>code</code> in the currently hosting browser.
     * The execution will block until it returns.
     *
     * @param code The code to be executed.
     * @return The return value of the code.
     */
    public abstract String executeScriptWithReturn(String code);

    /**
     * Creates a unique name, which can be used in JavaScript code.
     *
     * @return return an unused name, which can be used in JavaScript.
     */
    public abstract String createUniqueJSName();

    /**
     * Creates a unique name with the given prefix, which can be used in JavaScript code.
     *
     * @param prefix A prefix for generated name.
     * @return return an unused name, which can be used in JavaScript.
     */
    public abstract String createUniqueJSName(String prefix);

    /**
     * Executes Javascript code in the currently hosting browser and returns the result as a JSVariable.
     * Use {@link JS#eval(String)} instead.
     *
     * @param code The JavaScript code to be executed.
     * @return JSVariable, containing the result of the code.
     */
    @Deprecated
    public abstract JSVariable executeScriptWithVariable(String code);

    /**
     * Executes Javascript <code>code</code> in the currently hosting browser and assigns a listener to the execution
     * result. Use {@link JS#eval(String)} instead.
     *
     * @param code     The code which gets executed.
     * @param listener The listener which executed with the return value of the code.
     */
    @Deprecated
    public abstract void executeScriptWithListener(String code, ScriptResultListener listener);

    /**
     * Executes Javascript <code>code</code> in the currently hosting browser. The result will be returned as a
     * CompletableFuture. Use {@link JS#evalFuture(String)}.getString().thenAccept(listener) instead.
     *
     * @param code The code which gets executed.
     * @return A CompletableFuture, which contains the result of the code.
     * @since 2023.3.2
     */
    @Deprecated
    public abstract CompletableFuture<String> executeScriptWithFuture(String code);

    /**
     * Creates a public profile for a given <code>url</code>.
     *
     * @param url The url for the resource to be profiled. The url must accessible from the server.
     * @return url The url or the resource to be profiled. The url must be callable from the client.
     */
    public abstract String createPublicFile(URL url);


    /**
     * Downloads content from the given <code>url</code>.
     *
     * @param url The url from where to download. The url must be accessible from the server.
     */
    public abstract void downloadURL(URL url);

    /**
     * Downloads content from the given <code>url</code>.
     *
     * @param url     The url from where to download. The url must be accessible from the server.
     * @param cleanup gets executed after the resource was read by the server.
     */
    public abstract void downloadURL(URL url, Runnable cleanup);

    /**
     * Downloads content from the given resource <code>url</code>.
     *
     * @param resource The resource from where to download. The resource must be accessible from the server.
     */
    public abstract void downloadResource(String resource);

    /**
     * Downloads content from the given resource <code>url</code>.
     *
     * @param resource The resource from where to download. The resource must be accessible from the server.
     * @param cleanup  gets executed after the resource was read by the server.
     */
    public abstract void downloadResource(String resource, Runnable cleanup);

    /**
     * Opens content from the given <code>url</code>.
     *
     * @param url The url to be opened. The url must be accessible from the server.
     */
    public abstract void openLocalURL(URL url);

    /**
     * Opens content from the given resource <code>url</code>.
     *
     * @param resource The resource to be opened. The resource must be accessible from the server.
     */
    public abstract void openLocalResource(String resource);

    /**
     * Writes the id of the node into the javascript-variable jpro.*name*.
     * use {@link WebAPI#getElement(Node)} instead.
     *
     * @param name  The name of the node.
     * @param value The node to be given an alias.
     */
    @Deprecated
    public abstract void registerValue(String name, Node value);

    /**
     * Writes the value as json into the javascript-variable jpro.*name*.
     *
     * @param name  The name of the variable.
     * @param value The node in the scenegraph.
     */
    @Deprecated
    public abstract void registerValue(String name, String value);

    /**
     * This return a JSVariable, which reference the HTMLElement of the given node.
     *
     * @param node The node, which HTMLElement should be referenced.
     * @return a JSVariable, which reference the HTMLElement of the given node.
     * @since 2023.2.1
     */
    public abstract JSVariable getElement(Node node);

    /**
     * This return a JSVariable, which reference the HTMLElement of the given HTMLView.
     *
     * @param node The node, which HTMLElement should be referenced.
     * @return a JSVariable, which reference the HTMLElement of the given node.
     * @since 2023.2.2
     */
    public abstract JSVariable getHTMLViewElement(HTMLView node);

    /**
     * Wraps a node in a tag. Make sure to not change anything regarding rendering.
     * This can be used to better confirm to the HTML standard.
     *
     * @param tag  The tag, which should be used to wrap the node.
     * @param node The node, which should be wrapped.
     * @return a JSVariable, which reference the HTMLElement of the given node.
     * @since 2023.2.2
     */
    @Experimental
    public abstract JSVariable wrapNode(String tag, Node node);

    /**
     * This loads the node into the browser.
     * The node and all are children will be sent to the browser.
     * Afterward, showing the node in the browser is very fast and no further communication with the server is needed.
     *
     * @param node The JavaFX node, which should be loaded.
     * @since 2023.2.2
     */
    @Experimental
    public abstract void loadNode(Node node);

    /**
     * Writes a function into the javascript-variable jpro.*name*.
     * When this function is called, it's argument is sent to the JPro server, and used to call the callback.
     *
     * @param name     The name of the function. It will be accessible in JavaScript with jpro.*name*.
     * @param callback The java function, which will get called when the JavaScript function is called.
     */
    public abstract void registerJavaFunction(String name, WebCallback callback);

    /**
     * Writes a function which can be accessed with the js-variable.
     * When this function is called, it's argument is sent to the JPro server, and used to call the callback.
     * It will be cleaned up, when the JSVariable is no longer referenced.
     *
     * @param callback The java function, which will get called when the javascript function is called.
     */
    public abstract JSVariable registerJavaFunction(WebCallback callback);

    /**
     * Writes a function which can be accessed with the js-variable.
     * When this function is called, the argument will be mapped to a JSVariable and sent to the JPro server and used
     * to call the callback. The function will be cleaned up, when the JSVariable is no longer referenced.
     *
     * @param callback The java function, which will get called when the javascript function is called.
     *                 It is called with an JSVariable.
     * @return a JSVariable, which contains the function.
     * @since 2023.3.2
     */
    public abstract JSVariable registerJavaFunctionWithVariable(Consumer<JSVariable> callback);

    /**
     * Writes a function which contains asynchronous code and can be accessed with the JSVariable after it's finished.
     * The JSVariable is accessible after the CompletableFuture is finished and contains the result of the code.
     * The code should use a return statement. Otherwise, the result will be undefined.
     *
     * @param code The JavaScript code to be executed.
     * @return a CompletableFuture, which contains the result of the code.
     * @since 2023.3.2
     */
    public abstract CompletableFuture<JSVariable> executeJSAsync(String code);

    /**
     * Writes a function which contains asynchronous code. The result can be accessed with the JSVariable as a js-promise.
     * The code should use a return statement. Otherwise, the result will be undefined.
     *
     * @param code The JavaScript code to be executed.
     * @return a JSVariable, which contains a js-promise.
     * @since 2023.3.2
     */
    public abstract JSVariable executeJSAsyncPromise(String code);

    /**
     * Updates the width and height of the provided Scene.
     * It is usually called by the root of the Scene.
     *
     * @param scene The scene which gets layout.
     */
    public abstract void layoutRoot(Scene scene);

    /**
     * Updates the width and height of the provided Scene.
     * It is usually called by the root of the Scene.
     * <p>
     * This method is deprecated. Use {@link #layoutRoot(Scene)} instead.
     *
     * @param scene The scene which gets layout.
     */
    @Deprecated
    public abstract void requestLayout(Scene scene);

    /**
     * Opens the given <code>url</code> in the current tab.
     *
     * @param url The url which will be opened.
     * @since 2019.2.5
     */
    public abstract void openURL(String url);

    /**
     * This method returns an id, which can be used to start the specified window in a new JProTag.
     * It's used like this &lt;jpro-app href=&quot;/app/$id&quot;&gt;&lt;/jpro-app&gt;
     *
     * @param window The window which should be opened.
     * @return The id which can be used for a jpro-tag.
     * @since 2022.1.6
     */
    public abstract String registerWindow(Window window);

    /**
     * Opens the given <code>url</code> in a new tab.
     *
     * @param url The url which will be opened.
     * @since 2019.2.5
     */
    public abstract void openURLAsTab(String url);

    /**
     * Opens the given <code>url</code> in a new popup.
     *
     * @param url The url which will be opened.
     * @since 2019.2.5
     */
    public abstract void openURLAsPopup(String url);

    /**
     * Opens the given stage in a new popup.
     *
     * @param stage The stage which gets opened as a popup.
     *              It's important that the owner of the Stage is null,
     *              because otherwise the stage is shown as a frame in the owner.
     * @throws IllegalStateException The owner of the stage must be null.
     * @since 2019.1.0
     */
    public abstract void openStageAsPopup(Stage stage) throws IllegalStateException;

    /**
     * Opens the given stage in a new tab.
     *
     * @param stage The stage which gets opened as a tab.
     *              It's important that the owner of the Stage is null,
     *              because otherwise the stage is shown as a frame in the owner.
     * @throws IllegalStateException The owner of the stage must be null.
     * @since 2019.1.0
     */
    public abstract void openStageAsTab(Stage stage) throws IllegalStateException;

    /**
     * Opens the given stage in a new tab. The target specifies the name of the browsing context.
     *
     * @param stage The stage which gets opened as a tab.
     *              It's important that the owner of the Stage is null,
     *              because otherwise the stage is shown as a frame in the owner.
     * @param target A string, without whitespace, specifying the name of the browsing context the stage is being loaded
     *               into. If the name doesn't identify an existing context, a new context is created and given the
     *               specified name. The special target keywords, _self, _blank, _parent, _top, and can also be used.
     * @throws IllegalStateException The owner of the stage must be null.
     * @since 2024.3.3
     */
    public abstract void openStageAsTab(Stage stage, String target) throws IllegalStateException;

    /**
     * @param runnable The task which gets executed next.
     *                 The runnable is executed, after the current state of the scene graph is sent to the client.
     *                 This can be useful to load a page incrementally
     * @since 2019.1.0
     */
    public abstract void runAfterUpdate(Runnable runnable);

    /**
     * Returns a {@link MultiFileUploader} for a specific <code>node</code>.
     *
     * @param node The node which will be used for accessing files.
     * @return {@link FileUploader} of a specific node.
     */
    public abstract MultiFileUploader makeMultiFileUploadNode(Node node);

    /**
     * Returns a <code>FileUploader</code> for a specific <code>node</code>.
     *
     * @param node The node which will be used for accessing files.
     * @return {@link FileUploader} of a specific node.
     */
    public abstract FileUploader makeFileUploadNode(Node node);

    /**
     * Creates a {@link JSFile} object.
     *
     * @param objectURL The object URL of the file.
     * @param filename  The filename of the file.
     * @param size      The size of the file in bytes.
     * @return the {@link JSFile} object.
     */
    public abstract JSFile createJSFile(String objectURL, String filename, long size);

    /**
     * This is a functional interface, which is used by the FileUploader.
     *
     * @author Florian Kirmaier
     * @since 2018.1.1
     */
    @FunctionalInterface
    public interface FileSelectedListener {
        void handle(String value);
    }

    public static abstract class FileUploader implements SingleFileUploader {
    }

    /**
     * This class is used for uploading a file. It's associated with a node.
     *
     * @author Florian Kirmaier
     * @since 2018.1.1
     */
    public interface SingleFileUploader extends FileSelector {

        /**
         * Contains the filename, which was selected by the client.
         */
        ReadOnlyStringProperty selectedFileProperty();

        /**
         * Contains the size in bytes of the selected file.
         */
        ReadOnlyIntegerProperty selectedFileSizeProperty();

        /**
         * Defines the progress of uploading the file.
         * When the value is 1.0, then the upload is finished.
         */
        ReadOnlyDoubleProperty progressProperty();

        /**
         * The current JSFile, which was selected by the client.
         */
        ReadOnlyObjectProperty<JSFile> jsFileProperty();

        /**
         * Starts uploading the selected file.
         * During upload, the progress property will slowly reach 1.0.
         */
        void uploadFile();

        /**
         * Contains the file, which was uploaded. It is null until progress reaches 1.0.
         */
        ReadOnlyObjectProperty<File> uploadedFileProperty();

        /**
         * Defines a function to be called when a file is dropped.
         */
        ObjectProperty<FileSelectedListener> onFileSelectedProperty();

        /**
         * Defines whether an object url should be created.
         */
        BooleanProperty shouldCreateObjectURLProperty();

        /**
         * @return a JSVariable containing the object URL.
         */
        ObjectProperty<JSVariable> objectURLProperty();

        /**
         * Gets the value of the property uploadedFile.
         */
        default File getUploadedFile() {
            return uploadedFileProperty().getValue();
        }

        /**
         * Gets the value of the property selectedFile.
         */
        default String getSelectedFile() {
            return selectedFileProperty().getValue();
        }

        /**
         * Gets the value of the property selectedFileSize
         */
        default int getSelectedFileSize() {
            return selectedFileSizeProperty().getValue();
        }

        /**
         * Gets the value of the property progress.
         */
        default double getProgress() {
            return progressProperty().getValue();
        }

        /**
         * Gets the value of the property jsFile.
         */
        default JSFile getJSFile() {
            return jsFileProperty().getValue();
        }

        /**
         * @since 2018.1.1
         * Gets the value of the property onFileSelected
         */
        default FileSelectedListener getOnFileSelected() {
            return onFileSelectedProperty().getValue();
        }

        /**
         * @since 2018.1.1
         * Sets the value of the property onFileSelected.
         */
        default void setOnFileSelected(FileSelectedListener value) {
            onFileSelectedProperty().setValue(value);
        }
    }

    public interface MultiFileUploader extends FileSelector {

        /**
         * Defines a function to be called when a files are selected.
         *
         * @since 2022.1.6
         */
        ObjectProperty<Consumer<List<JSFile>>> onFilesSelectedProperty();

        /**
         * Sets the value of the property onFileSelected.
         *
         * @since 2022.1.6
         */
        default void setOnFilesSelected(Consumer<List<JSFile>> value) {
            onFilesSelectedProperty().setValue(value);
        }
    }

    public interface FileSelector {
        /**
         * When true, dropping a file over this node selects the dropped file.
         */
        BooleanProperty selectFileOnDropProperty();

        /**
         * This value changes to true, when the cursor is dragging a file and is above the associated node.
         */
        ReadOnlyBooleanProperty fileDragOverProperty();

        /**
         * The MIME types of the files which are currently dragged above the node.
         *
         * @since 2023.3.1
         */
        ObservableList<String> getFilesDragOverTypes();

        /**
         * When true, clicking the node will open a file chooser.
         */
        BooleanProperty selectFileOnClickProperty();

        /**
         * Gets the value of the property fileDragOver.
         *
         * @since 2018.1.1
         */
        default boolean getFileDragOver() {
            return fileDragOverProperty().getValue();
        }

        /**
         * Sets the value of the property selectFileOnClick.
         */
        default void setSelectFileOnClick(Boolean value) {
            selectFileOnClickProperty().setValue(value);
        }

        /**
         * Gets the value of the property selectFileOnClick
         */
        default boolean getSelectFileOnClick() {
            return selectFileOnClickProperty().getValue();
        }

        /**
         * Sets the value of the property selectFileOnDrop.
         */
        default void setSelectFileOnDrop(Boolean value) {
            selectFileOnDropProperty().setValue(value);
        }

        /**
         * Gets the value of the property selectFileOnDrop
         */
        default boolean getSelectFileOnDrop() {
            return selectFileOnDropProperty().getValue();
        }

        /**
         * Makes sure only files are selectable, which filename ends with at least one of the Strings.
         * Every entry should start with "." like ".jpg".
         * The file chooser might provide the option to open any file.
         * This is limited to the file chooser and doesn't work with drag and drop due to limitations in the browser.
         * The list is by default empty. If it's empty, then any filename is accepted.
         *
         * @since 2019.1.1
         */
        ObservableList<String> supportedExtensions();

        /**
         * Defines the selection mode of the file chooser.
         * The default value is SelectionMode.SINGLE.
         *
         * @since 2023.3.0
         */
        ObjectProperty<SelectionMode> selectionModeProperty();

        /**
         * Gets the value of the property selectionMode.
         *
         * @since 2023.3.0
         */
        default SelectionMode getSelectionMode() {
            return selectionModeProperty().getValue();
        }

        /**
         * Sets the value of the property selectionMode.
         *
         * @since 2023.3.0
         */
        default void setSelectionMode(SelectionMode value) {
            selectionModeProperty().setValue(value);
        }
    }


    public interface JSFile {

        String getFilename();

        /**
         * Returns the size of the file in bytes.
         */
        long getFileSize();

        /**
         * Returns the Object URL representing the file.
         * The Object URL is only valid as long as the JSFile is not garbage collected.
         */
        JSVariable getObjectURL();

        /**
         * Defines the progress of uploading the file.
         * When the value is 1.0, then the upload is finished.
         */
        ReadOnlyDoubleProperty progressProperty();

        /**
         * Contains the file, which was uploaded. It is null until progress reaches 1.0.
         */
        ReadOnlyObjectProperty<File> uploadedFileProperty();

        /**
         * Gets the value of the property progress.
         */
        default double getProgress() {
            return progressProperty().getValue();
        }

        /**
         * Gets the value of the property uploadedFile.
         */
        default File getUploadedFile() {
            return uploadedFileProperty().getValue();
        }

        /**
         * Starts uploading the selected file.
         * During upload, the progress property will slowly reach 1.0.
         */
        void uploadFile();

        /**
         * This returns the file when the upload is finished.
         * Calling this method, will trigger the upload, if it hasn't started yet.
         *
         * @return future of the uploaded file
         */
        CompletableFuture<File> getUploadedFileFuture();
    }

    /**
     * Sets whether the Image must be transmitted lossless.
     * The default value is true.
     * It must be called before the image is serialized by JPro.
     * This is useful for optimizing the size of images.
     * A typical use case would be to set lossless to false for a dynamically created WritableImage.
     * Then this Image would be sent to the client in a more compressed image format.
     *
     * @since 2019.2.2
     */
    public abstract void setLossless(Image image, boolean value);

    /**
     * Returns a <code>FileUploader</code> for a specific <code>node</code>
     *
     * @param node The node which will be used for accessing files.
     * @return <code>FileUploader</code> of a specific node.
     */
    public static FileUploader makeFileUploadNodeStatic(Node node) {
        return webAPIStaticImpl.makeFileUploadNodeStatic(node);
    }

    /**
     * Returns a <code>MultiFileUploader</code> for a specific <code>node</code>
     *
     * @param node The node which will be used for accessing files.
     * @return <code>MultiFileUploader</code> of a specific node.
     */
    public static MultiFileUploader makeMultiFileUploadNodeStatic(Node node) {
        return webAPIStaticImpl.makeMultiFileUploadNodeStatic(node);
    }

    /**
     * Sets whether the Image must be transmitted lossless.
     * The default value is true.
     * It must be called before the image is serialized by JPro.
     * This is useful for optimizing the size of images.
     * A typical use case would be to set lossless to false for a dynamically created WritableImage.
     * Then this Image would be sent to the client in a more compressed image format.
     *
     * @since 2022.1.0
     */
    public static void setLosslessStatic(Image image, boolean value) {
        webAPIStaticImpl.setLosslessStatic(image, value);
    }

    /**
     * Creates a virtual image that does not allocate any memory in the RAM.
     * When rendered in the browser, the specified resource is downloaded directly.
     * HTTP URLs are downloaded directly by the client, whereas file resources are accessed through the JPro server.
     *
     * @param url The URL of the image resource.
     * @param w   The width of the virtual image.
     * @param h   The height of the virtual image.
     * @return A virtual image object representing the specified image.
     * @since 2022.1.0
     */
    public static Image createVirtualImage(String url, int w, int h) {
        return webAPIStaticImpl.createVirtualImage(url, w, h);
    }

    /**
     * Creates a virtual image that does not allocate any memory in the RAM.
     * The image resource is downloaded directly when rendered in the browser.
     * If {@code jproServerAsProxy} is {@code true}, the image URL is handled by the JPro server,
     * allowing the server to act as a proxy. If {@code jproServerAsProxy} is {@code false},
     * the image URL is accessed directly by the client.
     *
     * @param url               The URL of the image resource.
     * @param w                 The width of the virtual image.
     * @param h                 The height of the virtual image.
     * @param jproServerAsProxy determines whether the JPro server will act as a proxy for the URL
     * @return a virtual image object representing the specified image
     * @since 2022.1.0
     */
    public static Image createVirtualImage(String url, int w, int h, boolean jproServerAsProxy) {
        return webAPIStaticImpl.createVirtualImage(url, w, h, jproServerAsProxy);
    }

    /**
     * Calling this method closes the current instance.
     * This can be used to trigger a reconnect, to create a fresh instance.
     */
    public abstract void closeInstance();

    /**
     * Returns information about the current running server.
     *
     * @return an {@link ServerInfo} object containing server information about this session.
     */
    public static ServerInfo getServerInfo() {
        return webAPIStaticImpl.getServerInfo();
    }

    /**
     * Internal class
     */
    public interface WebAPIStatic {
        FileUploader makeFileUploadNodeStatic(Node node);

        MultiFileUploader makeMultiFileUploadNodeStatic(Node node);

        void setLosslessStatic(Image image, boolean value);

        Image createVirtualImage(String url, int w, int h);

        Image createVirtualImage(String url, int w, int h, boolean jproServerAsProxy);

        ServerInfo getServerInfo();
    }
}
