package com.jpro.webapi.server;

import com.jpro.webapi.annotation.Experimental;

/**
 * Represents a server response.
 * <p>
 * If no mime type is set, then a default mimetype based on the file extension will be used.
 *
 * @author Florian Kirmaier
 */
@Experimental
public class Response {

    private final boolean empty;
    private byte[] body;

    private String mimeType;

    /**
     * Creates an empty response.
     * This private constructor is used internally by the {@link #empty()} method.
     */
    private Response() {
        empty = true;
    }

    /**
     * Creates a response with the given body.
     * This constructor sets the response as non-empty.
     *
     * @param body the response body
     */
    public Response(byte[] body) {
        this.body = body;
        empty = false;
    }

    /**
     * Checks if the response is empty.
     *
     * @return {@code true} if the response is empty, {@code false} otherwise
     */
    public boolean isEmpty() {
        return empty;
    }

    /**
     * Gets the body of the response.
     *
     * @return the response body, or {@code null} if the response is empty
     */
    public byte[] getBody() {
        return body;
    }

    /**
     * Gets the MIME type of the response.
     *
     * @return the MIME type, or {@code null} if not specified
     */
    public String getMimeType() {
        return mimeType;
    }

    /**
     * Creates an empty response.
     *
     * @return an empty response
     */
    public static Response empty() {
        return new Response();
    }

    /**
     * Creates a response with the given body.
     *
     * @param body the response body
     * @return a response containing the specified body
     */
    public static Response of(byte[] body) {
        return new Response(body);
    }

    /**
     * Creates a response with the given body and MIME type.
     *
     * @param body the response body
     * @param mimeType the MIME type of the response
     * @return a response containing the specified body and MIME type
     */
    public static Response of(byte[] body, String mimeType) {
        Response response = new Response(body);
        response.mimeType = mimeType;
        return response;
    }
}
