package com.jpro.webapi;

import javafx.beans.property.*;

/**
 * This interface can be implemented by a JavaFX {@link javafx.scene.Node}.
 * When an TextInputControl is used with JPro, in the Browser a hidden input field is used to capture the input.
 * Using this interface creates the same behavior in JavaFX without the need to extend TextInputControl.
 */
public interface JProTextInputControl {

    /**
     * Determines whether the text input control is a TextArea.
     *
     * @return <code>true</code> if the text input control is a TextArea, <code>false</code> otherwise.
     */
    boolean isTextArea();

    /**
     * Retrieves the current text from the text input control.
     *
     * @return the current text as a String.
     */
    default String getText() {
        return textProperty().get();
    }

    /**
     * Sets the text in the text input control to the specified string.
     *
     * @param text the new text to set in the control.
     */
    default void setText(String text) {
        textProperty().set(text);
    }

    /**
     * The text that is displayed in the control.
     */
    StringProperty textProperty();

    /**
     * Retrieves the type of virtual keyboard that is used with this control.
     *
     * @return the type of virtual keyboard as a String.
     */
    default String getVkType() {
        return vkTypeProperty().get();
    }

    /**
     * Sets the type of virtual keyboard to be used with this control.
     *
     * @param vkType the type of virtual keyboard to set.
     */
    default void setVkType(String vkType) {
        vkTypeProperty().set(vkType);
    }

    /**
     * The text that is displayed in the control.
     */
    StringProperty vkTypeProperty();

    /**
     * Gets the position of the anchor in the text control.
     * The anchor, together with the caret, defines the text selection.
     *
     * @return the position of the anchor as an integer.
     */
    default int getAnchor() {
        return anchorProperty().get();
    }

    /**
     * Sets the position of the anchor in the text control. The anchor, v
     * together with the caret, defines the text selection.
     *
     * @param anchor the new position of the anchor.
     */
    default void setAnchor(int anchor) {
        anchorProperty().set(anchor);
    }

    /**
     * The anchor of the selection.
     */
    IntegerProperty anchorProperty();

    /**
     * Retrieves the current position of the caret (cursor) in the text control.
     *
     * @return the current caret position as an integer.
     */
    default int getCaretPosition() {
        return caretPositionProperty().get();
    }

    /**
     * Sets the position of the caret (cursor) in the text control.
     *
     * @param caretPosition the new position of the caret.
     */
    default void setCaretPosition(int caretPosition) {
        caretPositionProperty().set(caretPosition);
    }

    /**
     * Provides access to the caret position property.
     * The caret position indicates the current position of the caret (cursor) within the text.
     *
     * @return an IntegerProperty representing the current caret position in the text control.
     */
    IntegerProperty caretPositionProperty();

    /**
     * Checks if the text input control is editable.
     *
     * @return <code>true</code> if the control is editable, <code>false</code> otherwise.
     */
    default boolean isEditable() {
        return editableProperty().get();
    }

    /**
     * Whether the control is editable or not.
     */
    ReadOnlyBooleanProperty editableProperty();

    /**
     * Selects a range of text in the text input control. The range is specified by the anchor and caret position,
     * where anchor defines the start of the selection and caretPosition defines the end of the selection.
     *
     * @param anchor the starting index of the text selection.
     * @param caretPosition the ending index of the text selection.
     */
    void selectRange(int anchor, int caretPosition);
}
