package com.jpro.webapi;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * This class represents a variable in the JS-scope of the client, which is a promise.
 * It provides the same functionality as a JSVariable, but has additional methods to handle the promise.
 */
public class PromiseJSVariable extends JSVariable {

    private final JSVariable jsVariable;

    /**
     * Constructs a new PromiseJSVariable using an existing JSVariable.
     *
     * @param jsVariable the JSVariable that is being wrapped as a promise
     */
    public PromiseJSVariable(JSVariable jsVariable) {
        super(jsVariable.getWebAPI(), jsVariable.getName(), "", jsVariable.onComplete());
        this.jsVariable = jsVariable;
    }

    /**
     * Resolves the JavaScript promise and retrieves its value as a string.
     * This method first ensures that the promise is resolved by calling {@link #resolveVariable()},
     * then converts the resolved value to a string.
     *
     * @return a {@link CompletableFuture} that, when completed, provides the string value of the resolved promise.
     */
    public CompletableFuture<String> getPromiseString() {
        return resolveVariable().thenCompose(JSVariable::getString);
    }

    /**
     * Returns a CompletableFuture that is completed when the underlying JavaScript promise is resolved.
     * This does not provide the value of the promise but rather signifies its completion.
     * If an error occurs, the future will be completed exceptionally.
     *
     * @return a {@link CompletableFuture} that is completed when the promise is resolved
     */
    public CompletableFuture<Void> onPromiseComplete() {
        return resolveVariable().thenAccept(JSVariable::getString);
    }

    /**
     * Registers an error handler for the promise. This handler is invoked if the promise
     * encounters an error during its resolution.
     *
     * @param errorHandler the consumer to handle errors
     */
    public void onPromiseError(Consumer<Throwable> errorHandler) {
        resolveVariable()
                .thenAccept((jsVariable) -> jsVariable.onError(errorHandler))
                .whenComplete((result, error) -> {
                    if (error != null) {
                        errorHandler.accept(error);
                    }
                });
    }

    /**
     * Resolves the underlying JavaScript promise and returns a CompletableFuture that contains
     * the JSVariable representing the resolved promise.
     *
     * @return a {@link CompletableFuture} containing the JSVariable once the promise is resolved
     */
    public CompletableFuture<JSVariable> resolveVariable() {
        return JSVariable.promiseToFuture(jsVariable.webAPI, this);
    }
}
