package com.jpro.webapi;

import com.jpro.webapi.annotation.Experimental;

import java.time.Duration;
import java.util.*;

/**
 * Holds information for the current running JPro server.
 */
@Experimental
public interface ServerInfo {

    /**
     * Returns the start time of the server in milliseconds.
     *
     * @return a long value representing the start time.
     */
    long getStartTime();

    /**
     * Returns the running time of the server.
     *
     * @return a {@link Duration} object representing the running time.
     */
    Duration getRunningTime();

    /**
     * Returns the number of instances created.
     *
     * @return an integer value representing the number of instances created.
     */
    int getInstancesCreatedCount();

    /**
     * Returns the number of instances active.
     *
     * @return an integer value representing the number of instances active.
     */
    int getInstancesActiveCount();

    /**
     * Returns the number of instances not active or afk (away from keyboard).
     *
     * @return an integer value representing the number of instances not active.
     */
    int getInstancesAfkCount();

    /**
     * Returns the number of views created.
     *
     * @return an integer value representing the number of views created.
     */
    int getViewsCreatedCount();

    /**
     * Returns the number of views active.
     *
     * @return an integer value representing the number of views active.
     */
    int getViewsActiveCount();

    /**
     * Returns the number of views not active or afk (away from keyboard).
     *
     * @return an integer value representing the number of views not active or afk.
     */
    int getViewsAfkCount();

    /**
     * Returns the number of active windows opened.
     *
     * @return an integer value representing the number of active windows opened.
     */
    int getActiveWindowsCount();

    /**
     * Returns the current framerate.
     *
     * @return an integer value representing the current framerate.
     */
    int getFramerate();

    /**
     * Returns the max memory available for the server.
     *
     * @return a long value representing the max memory available.
     */
    long getMaxMemory();

    /**
     * Returns the heap memory used by the server.
     *
     * @return a long value representing the total memory available.
     */
    long getUsedHeapMemory();

    /**
     * Returns the heap memory used by the server in percentage.
     *
     * @return a double value representing the heap memory used in percentage.
     */
    double getUsedHeapMemoryPercent();

    /**
     * Returns the non heap memory used by the server.
     *
     * @return a long value representing the non heap memory used.
     */
    long getUsedNonHeapMemory();

    /**
     * Returns the non heap memory commited by the server.
     *
     * @return a double value representing the non heap memory commited.
     */
    long getCommittedNonHeapMemory();

    /**
     * Returns the overall JavaFX CPU usage by the server.
     *
     * @return a double value representing the overall JavaFX CPU usage.
     */
    double getJavaFXCpuUsage();

    /**
     * Returns the current Java version the server is running on.
     *
     * @return a string value representing the current Java version.
     */
    String getJavaVersion();

    /**
     * Returns the current JavaFX version the server is running on.
     *
     * @return a string value representing the current JavaFX version.
     */
    String getJavaFXVersion();

    /**
     * Returns the current version of the JPro server.
     *
     * @return a string value representing the current version of the JPro server.
     */
    String getJProVersion();

    /**
     * Returns the latest Git commit hash for this JPro server.
     *
     * @return a string value representing the latest Git commit hash for this JPro server.
     */
    String getJProGitCommit();

    /**
     * Returns the build time of this JPro server.
     *
     * @return a long value representing the build time of this JPro server.
     */
    long getJProBuildTime();

    /**
     * Returns the current mode the server is running on:
     * <ul>
     *     <li>
     *         dev = development mode
     *     </li>
     *     <li>
     *         prod = production mode
     *     </li>
     * </ul>
     *
     * @return a string value representing the current JPro mode.
     */
    String getJProMode();

    /**
     * Returns deployment method for this JPro server, either via Gradle or Maven.
     *
     * @return a string value representing the deployment method for this JPro server.
     */
    String getJProDeployment();

    /**
     * Returns the free memory of the system the server is running on.
     *
     * @return
     */
    long getFreeSystemMemory();

    /**
     * Returns the total memory of the system the server is running on.
     *
     * @return a long value representing the total memory of the system.
     */
    long getTotalSystemMemory();

    /**
     * Returns the CPU usage of the system the server is running on.
     *
     * @return a double value representing the CPU usage of the system.
     */
    double getCpuSystemLoad();

    /**
     * Returns the JVM CPU usage of the system the server is running on.
     *
     * @return a double value representing the JVM CPU usage of the system.
     */
    double getJvmSystemLoad();

    /**
     * Returns the free disk space of the system the server is running on.
     *
     * @return a long value representing the free disk space of the system.
     */
    long getFreeDiskSpace();

    /**
     * Returns the total disk space of the system the server is running on.
     *
     * @return a long value representing the total disk space of the system.
     */
    long getTotalDiskSpace();

    /**
     * Returns the default file encoding of the system the server is running on.
     *
     * @return a string value representing the default file encoding of the system.
     */
    String getDefaultJavaEncoding();

    /**
     * Returns the default locale of the system the server is running on.
     *
     * @return a string value representing the default locale of the system.
     */
    Locale getDefaultJavaLocale();

    /**
     * Returns the default timezone of the system the server is running on.
     *
     * @return a string value representing the default timezone of the system.
     */
    TimeZone getDefaultJavaTimeZone();

    /**
     * Returns the default country of the system the server is running on.
     *
     * @return a string value representing the default country of the system.
     */
    List<String> getOpenInstances();

    /**
     * Returns stats information provided as a map.
     *
     * @return a map containing status information.
     */
    Map<String, Object> getStats();
}
