package one.lfa.epubsquash.api;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Configuration values for EPUB squashers.
 */
@SuppressWarnings({"all"})
public final class EPUBSquasherConfiguration implements EPUBSquasherConfigurationType {
  private final Path inputFile;
  private final Path temporaryDirectory;
  private final Path outputFile;
  private final double scale;
  private final double maximumImageWidth;
  private final double maximumImageHeight;

  private EPUBSquasherConfiguration(EPUBSquasherConfiguration.Builder builder) {
    this.inputFile = builder.inputFile;
    this.temporaryDirectory = builder.temporaryDirectory;
    this.outputFile = builder.outputFile;
    this.maximumImageWidth = builder.maximumImageWidth;
    this.maximumImageHeight = builder.maximumImageHeight;
    this.scale = builder.scaleIsSet()
        ? builder.scale
        : EPUBSquasherConfigurationType.super.scale();
  }

  private EPUBSquasherConfiguration(
      Path inputFile,
      Path temporaryDirectory,
      Path outputFile,
      double scale,
      double maximumImageWidth,
      double maximumImageHeight) {
    this.inputFile = inputFile;
    this.temporaryDirectory = temporaryDirectory;
    this.outputFile = outputFile;
    this.scale = scale;
    this.maximumImageWidth = maximumImageWidth;
    this.maximumImageHeight = maximumImageHeight;
  }

  /**
   * @return The input file
   */
  @Override
  public Path inputFile() {
    return inputFile;
  }

  /**
   * @return The temporary directory used to unpack files
   */
  @Override
  public Path temporaryDirectory() {
    return temporaryDirectory;
  }

  /**
   * @return The output file
   */
  @Override
  public Path outputFile() {
    return outputFile;
  }

  /**
   * Each image will be scaled by a given amount, where {@code 1.0} keeps the original size,
   * {@code 0.5} scales by 50%, etc.
   * @return The amount by which to scale the image
   */
  @Override
  public double scale() {
    return scale;
  }

  /**
   * @return The maximum image width in pixels
   */
  @Override
  public double maximumImageWidth() {
    return maximumImageWidth;
  }

  /**
   * @return The maximum image height in pixels
   */
  @Override
  public double maximumImageHeight() {
    return maximumImageHeight;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#inputFile() inputFile} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for inputFile
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withInputFile(Path value) {
    if (this.inputFile == value) return this;
    Path newValue = Objects.requireNonNull(value, "inputFile");
    return new EPUBSquasherConfiguration(
        newValue,
        this.temporaryDirectory,
        this.outputFile,
        this.scale,
        this.maximumImageWidth,
        this.maximumImageHeight);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#temporaryDirectory() temporaryDirectory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for temporaryDirectory
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withTemporaryDirectory(Path value) {
    if (this.temporaryDirectory == value) return this;
    Path newValue = Objects.requireNonNull(value, "temporaryDirectory");
    return new EPUBSquasherConfiguration(
        this.inputFile,
        newValue,
        this.outputFile,
        this.scale,
        this.maximumImageWidth,
        this.maximumImageHeight);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#outputFile() outputFile} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for outputFile
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withOutputFile(Path value) {
    if (this.outputFile == value) return this;
    Path newValue = Objects.requireNonNull(value, "outputFile");
    return new EPUBSquasherConfiguration(
        this.inputFile,
        this.temporaryDirectory,
        newValue,
        this.scale,
        this.maximumImageWidth,
        this.maximumImageHeight);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#scale() scale} attribute.
   * A value strict bits equality used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for scale
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withScale(double value) {
    if (Double.doubleToLongBits(this.scale) == Double.doubleToLongBits(value)) return this;
    return new EPUBSquasherConfiguration(
        this.inputFile,
        this.temporaryDirectory,
        this.outputFile,
        value,
        this.maximumImageWidth,
        this.maximumImageHeight);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#maximumImageWidth() maximumImageWidth} attribute.
   * A value strict bits equality used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for maximumImageWidth
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withMaximumImageWidth(double value) {
    if (Double.doubleToLongBits(this.maximumImageWidth) == Double.doubleToLongBits(value)) return this;
    return new EPUBSquasherConfiguration(
        this.inputFile,
        this.temporaryDirectory,
        this.outputFile,
        this.scale,
        value,
        this.maximumImageHeight);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EPUBSquasherConfigurationType#maximumImageHeight() maximumImageHeight} attribute.
   * A value strict bits equality used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for maximumImageHeight
   * @return A modified copy of the {@code this} object
   */
  public final EPUBSquasherConfiguration withMaximumImageHeight(double value) {
    if (Double.doubleToLongBits(this.maximumImageHeight) == Double.doubleToLongBits(value)) return this;
    return new EPUBSquasherConfiguration(
        this.inputFile,
        this.temporaryDirectory,
        this.outputFile,
        this.scale,
        this.maximumImageWidth,
        value);
  }

  /**
   * This instance is equal to all instances of {@code EPUBSquasherConfiguration} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof EPUBSquasherConfiguration
        && equalTo((EPUBSquasherConfiguration) another);
  }

  private boolean equalTo(EPUBSquasherConfiguration another) {
    return inputFile.equals(another.inputFile)
        && temporaryDirectory.equals(another.temporaryDirectory)
        && outputFile.equals(another.outputFile)
        && Double.doubleToLongBits(scale) == Double.doubleToLongBits(another.scale)
        && Double.doubleToLongBits(maximumImageWidth) == Double.doubleToLongBits(another.maximumImageWidth)
        && Double.doubleToLongBits(maximumImageHeight) == Double.doubleToLongBits(another.maximumImageHeight);
  }

  /**
   * Computes a hash code from attributes: {@code inputFile}, {@code temporaryDirectory}, {@code outputFile}, {@code scale}, {@code maximumImageWidth}, {@code maximumImageHeight}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + inputFile.hashCode();
    h += (h << 5) + temporaryDirectory.hashCode();
    h += (h << 5) + outputFile.hashCode();
    h += (h << 5) + Double.hashCode(scale);
    h += (h << 5) + Double.hashCode(maximumImageWidth);
    h += (h << 5) + Double.hashCode(maximumImageHeight);
    return h;
  }

  /**
   * Prints the immutable value {@code EPUBSquasherConfiguration} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "EPUBSquasherConfiguration{"
        + "inputFile=" + inputFile
        + ", temporaryDirectory=" + temporaryDirectory
        + ", outputFile=" + outputFile
        + ", scale=" + scale
        + ", maximumImageWidth=" + maximumImageWidth
        + ", maximumImageHeight=" + maximumImageHeight
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link EPUBSquasherConfigurationType} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable EPUBSquasherConfiguration instance
   */
  public static EPUBSquasherConfiguration copyOf(EPUBSquasherConfigurationType instance) {
    if (instance instanceof EPUBSquasherConfiguration) {
      return (EPUBSquasherConfiguration) instance;
    }
    return EPUBSquasherConfiguration.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link EPUBSquasherConfiguration EPUBSquasherConfiguration}.
   * <pre>
   * EPUBSquasherConfiguration.builder()
   *    .setInputFile(java.nio.file.Path) // required {@link EPUBSquasherConfigurationType#inputFile() inputFile}
   *    .setTemporaryDirectory(java.nio.file.Path) // required {@link EPUBSquasherConfigurationType#temporaryDirectory() temporaryDirectory}
   *    .setOutputFile(java.nio.file.Path) // required {@link EPUBSquasherConfigurationType#outputFile() outputFile}
   *    .setScale(double) // optional {@link EPUBSquasherConfigurationType#scale() scale}
   *    .setMaximumImageWidth(double) // required {@link EPUBSquasherConfigurationType#maximumImageWidth() maximumImageWidth}
   *    .setMaximumImageHeight(double) // required {@link EPUBSquasherConfigurationType#maximumImageHeight() maximumImageHeight}
   *    .build();
   * </pre>
   * @return A new EPUBSquasherConfiguration builder
   */
  public static EPUBSquasherConfiguration.Builder builder() {
    return new EPUBSquasherConfiguration.Builder();
  }

  /**
   * Builds instances of type {@link EPUBSquasherConfiguration EPUBSquasherConfiguration}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_INPUT_FILE = 0x1L;
    private static final long INIT_BIT_TEMPORARY_DIRECTORY = 0x2L;
    private static final long INIT_BIT_OUTPUT_FILE = 0x4L;
    private static final long INIT_BIT_MAXIMUM_IMAGE_WIDTH = 0x8L;
    private static final long INIT_BIT_MAXIMUM_IMAGE_HEIGHT = 0x10L;
    private static final long OPT_BIT_SCALE = 0x1L;
    private long initBits = 0x1fL;
    private long optBits;

    private Path inputFile;
    private Path temporaryDirectory;
    private Path outputFile;
    private double scale;
    private double maximumImageWidth;
    private double maximumImageHeight;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code EPUBSquasherConfigurationType} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(EPUBSquasherConfigurationType instance) {
      Objects.requireNonNull(instance, "instance");
      setInputFile(instance.inputFile());
      setTemporaryDirectory(instance.temporaryDirectory());
      setOutputFile(instance.outputFile());
      setScale(instance.scale());
      setMaximumImageWidth(instance.maximumImageWidth());
      setMaximumImageHeight(instance.maximumImageHeight());
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#inputFile() inputFile} attribute.
     * @param inputFile The value for inputFile 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setInputFile(Path inputFile) {
      this.inputFile = Objects.requireNonNull(inputFile, "inputFile");
      initBits &= ~INIT_BIT_INPUT_FILE;
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#temporaryDirectory() temporaryDirectory} attribute.
     * @param temporaryDirectory The value for temporaryDirectory 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setTemporaryDirectory(Path temporaryDirectory) {
      this.temporaryDirectory = Objects.requireNonNull(temporaryDirectory, "temporaryDirectory");
      initBits &= ~INIT_BIT_TEMPORARY_DIRECTORY;
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#outputFile() outputFile} attribute.
     * @param outputFile The value for outputFile 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setOutputFile(Path outputFile) {
      this.outputFile = Objects.requireNonNull(outputFile, "outputFile");
      initBits &= ~INIT_BIT_OUTPUT_FILE;
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#scale() scale} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link EPUBSquasherConfigurationType#scale() scale}.</em>
     * @param scale The value for scale 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setScale(double scale) {
      this.scale = scale;
      optBits |= OPT_BIT_SCALE;
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#maximumImageWidth() maximumImageWidth} attribute.
     * @param maximumImageWidth The value for maximumImageWidth 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setMaximumImageWidth(double maximumImageWidth) {
      this.maximumImageWidth = maximumImageWidth;
      initBits &= ~INIT_BIT_MAXIMUM_IMAGE_WIDTH;
      return this;
    }

    /**
     * Initializes the value for the {@link EPUBSquasherConfigurationType#maximumImageHeight() maximumImageHeight} attribute.
     * @param maximumImageHeight The value for maximumImageHeight 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder setMaximumImageHeight(double maximumImageHeight) {
      this.maximumImageHeight = maximumImageHeight;
      initBits &= ~INIT_BIT_MAXIMUM_IMAGE_HEIGHT;
      return this;
    }

    /**
     * Builds a new {@link EPUBSquasherConfiguration EPUBSquasherConfiguration}.
     * @return An immutable instance of EPUBSquasherConfiguration
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public EPUBSquasherConfiguration build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new EPUBSquasherConfiguration(this);
    }

    private boolean scaleIsSet() {
      return (optBits & OPT_BIT_SCALE) != 0;
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_INPUT_FILE) != 0) attributes.add("inputFile");
      if ((initBits & INIT_BIT_TEMPORARY_DIRECTORY) != 0) attributes.add("temporaryDirectory");
      if ((initBits & INIT_BIT_OUTPUT_FILE) != 0) attributes.add("outputFile");
      if ((initBits & INIT_BIT_MAXIMUM_IMAGE_WIDTH) != 0) attributes.add("maximumImageWidth");
      if ((initBits & INIT_BIT_MAXIMUM_IMAGE_HEIGHT) != 0) attributes.add("maximumImageHeight");
      return "Cannot build EPUBSquasherConfiguration, some of required attributes are not set " + attributes;
    }
  }
}
