package one.payout.payment.api

import one.payout.payment.api.PayoutClient.Builder
import one.payout.payment.api.model.*
import one.payout.payment.impl.PayoutClientImpl
import java.net.http.HttpClient
import java.util.function.Consumer

/**
 * PayoutClient is a client for the [Payout API](https://postman.payout.one/).
 * It wraps the API and provides a convenient way to interact with it.
 *
 * Authorization is done automatically before each request (it is not required to call [authorize] method manually).
 * The client is responsible for refreshing the token when it expires.
 *
 * Use [PayoutClient.newBuilder] to create an instance.
 * The client is stateless and thread-safe. It is recommended to create a single instance of the client and reuse it.
 *
 * Client creation example:
 * ```
 *     val client = PayoutClient
 *         .newBuilder()
 *         .sandboxBaseUrl()
 *         .clientId(System.getenv("PAYOUT_CLIENT_ID"))
 *         .clientSecret(System.getenv("PAYOUT_CLIENT_SECRET"))
 *         .build()
 * ```
 *
 * For more customization see [Builder].
 * For more information about API see [Payout API documentation](https://postman.payout.one/).
 *
 * @see Builder
 */
interface PayoutClient {
    companion object {
        const val PROD_BASE_URL = "https://app.payout.one"
        const val SANDBOX_BASE_URL = "https://sandbox.payout.one"

        @JvmStatic
        fun newBuilder(): Builder = PayoutClientImpl.newBuilder()
    }

    fun authorize(): ApiResponse<AuthorizeResponse>

    fun createCheckout(createCheckoutRequest: CreateCheckoutRequest): ApiResponse<Checkout>

    fun retrieveCheckout(checkoutId: Long): ApiResponse<Checkout?>

    fun parseWebhook(body: String): Webhook

    /**
     * Builder for [PayoutClient].
     *
     * Builders are not thread-safe and should not be used concurrently from multiple threads without external synchronization.
     */
    interface Builder {
        /** Sets the client ID. It is used for authorization. */
        fun clientId(clientId: String): Builder

        /**
         * Sets the client secret. It is used for authorization, signing and signature verification.
         *
         * Client secret will not be stored in memory. It will be requested from the supplier each time it is needed.
         */
        fun clientSecretSupplier(clientSecretSupplier: ClientSecretSupplier): Builder

        /**
         * Sets the client secret statically. It is used for authorization, signing and signature verification.
         *
         * To provide more secure usage, use [clientSecretSupplier] instead (no need to store secret in memory).
         * @see clientSecretSupplier
         */
        fun clientSecret(clientSecret: String): Builder = clientSecretSupplier { clientSecret }

        /** Sets the base URL of the API. Default is [PROD_BASE_URL]. */
        fun baseUrl(baseUrl: String): Builder

        /**
         * Sets the [SANDBOX_BASE_URL] as base URL.
         * @see baseUrl
         */
        fun sandboxBaseUrl(): Builder = baseUrl(SANDBOX_BASE_URL)

        /**
         * Sets the [NonceGenerator] for generating nonces.
         * If not supplied, `UUID.randomUUID().toString()` will be used.
         */
        fun nonceGenerator(nonceGenerator: NonceGenerator): Builder

        /**
         * Method to adjust the [HttpClient] used by the client.
         * @see HttpClient.Builder
         */
        fun adjustHttp(block: HttpClient.Builder.() -> Unit): Builder

        /**
         * Method to adjust the [HttpClient] used by the client to use easier in Java.
         * @see HttpClient.Builder
         */
        fun adjustHttp(consumer: Consumer<HttpClient.Builder>): Builder = adjustHttp { consumer.accept(this) }

        /** Builds the [PayoutClient]. */
        fun build(): PayoutClient
    }

    fun interface ClientSecretSupplier {
        fun get(): String
    }

    fun interface NonceGenerator {
        fun generate(): String
    }

    open class PayoutClientException : RuntimeException {
        constructor(message: String, cause: Throwable?) : super(message, cause)
        constructor(message: String) : super(message)
    }

    class AuthorizeException : PayoutClientException {
        constructor(message: String, cause: Throwable?) : super(message, cause)
        constructor(message: String) : super(message)
    }
}
