package one.payout.payment.impl

import io.github.oshai.kotlinlogging.KotlinLogging
import one.payout.payment.api.PayoutClient
import one.payout.payment.api.model.ApiResponse
import one.payout.payment.api.model.AuthorizeResponse
import java.util.*
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ConcurrentLinkedQueue
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicReference
import kotlin.time.Duration.Companion.seconds

private val logger = KotlinLogging.logger {}

internal class PayoutAuthorizer(private val authorizeMethod: AuthorizeMethod) {
    private val cache = AtomicReference<State>(State.Empty)
    private val manualLoaders: ConcurrentLinkedQueue<State.Loading> = ConcurrentLinkedQueue()

    @Throws(PayoutClient.AuthorizeException::class)
    fun authorize(): ApiResponse<AuthorizeResponse> =
        createLoadingState()
            .also {
                manualLoaders.add(it)
                it.future.whenComplete { _, _ -> manualLoaders.remove(it) }
            }
            .future
            .runCatching { get() }
            .getOrElse { throw PayoutClient.AuthorizeException("Failed to authorize: $it", it) }

    @Throws(PayoutClient.AuthorizeException::class)
    fun getToken(): Token {
        val currentState = cache.get()
        if (currentState.isValid()) {
            logger.debug { "Reusing valid${if (currentState is State.Loading) " loading" else ""} token" }
            currentState
                .getToken()
                .onSuccess { return Token(it, false) }
                .onFailure { logger.debug { "Retrying - previous authorization failed: $it" } }
        } else {
            val manualLoader = manualLoaders.peek()
            if (manualLoader != null && cache.compareAndSet(currentState, manualLoader)) {
                logger.debug { "Reusing - manual loader is loading" }
                manualLoader
                    .getToken()
                    .onSuccess { return Token(it, false) }
                    .onFailure { logger.debug { "Retrying - previous manual loader failed: $it" } }
            }
        }
        return cache
            .updateAndGet { if (it == currentState) createLoadingState() else it }
            .getToken()
            .getOrElse { throw PayoutClient.AuthorizeException("Failed to authorize: $it", it) }
            .let { Token(it, true) }
    }

    fun invalidateToken(token: String) {
        val currentState = cache.get()
        if (currentState is State.Authorization && currentState.authorizeResponse.token == token) {
            cache.compareAndSet(currentState, State.Empty)
        }
    }

    private fun createLoadingState(): State.Loading {
        val future = CompletableFuture
            .supplyAsync {
                val startTime = System.currentTimeMillis()
                authorizeMethod.authorize().also { response ->
                    response.getOrNull()?.let { authorization ->
                        cache.updateAndGet {
                            if (it is State.Authorization && it.startTime >= startTime) it
                            else State.Authorization(startTime, authorization)
                        }
                    }
                }
            }
            .orTimeout(20, TimeUnit.SECONDS)
        return State.Loading(future)
    }

    fun interface AuthorizeMethod {
        fun authorize(): ApiResponse<AuthorizeResponse>
    }

    data class Token(val value: String, val newCreated: Boolean)

    private sealed class State {
        abstract fun isValid(): Boolean
        abstract fun getToken(): Result<String>

        data object Empty : State() {
            override fun isValid(): Boolean = false
            override fun getToken(): Result<String> = Result.failure(IllegalStateException("Token not available"))
        }

        data class Authorization(val startTime: Long, val authorizeResponse: AuthorizeResponse) : State() {
            val validUntil = startTime + authorizeResponse.validFor.seconds.inWholeMilliseconds

            override fun isValid(): Boolean = System.currentTimeMillis() < validUntil
            override fun getToken(): Result<String> = Result.success(authorizeResponse.token)
        }

        data class Loading(val future: CompletableFuture<ApiResponse<AuthorizeResponse>>) : State() {
            override fun isValid(): Boolean = !future.isDone
            override fun getToken(): Result<String> = future.runCatching { get().result.getOrThrow().token }
        }
    }
}
