package one.payout.payment.impl

import io.github.oshai.kotlinlogging.KotlinLogging
import one.payout.payment.api.PayoutClient
import one.payout.payment.api.model.*
import one.payout.payment.impl.method.ClientMethod
import one.payout.payment.impl.method.WebhookParser
import one.payout.payment.impl.method.api.AuthorizeMethod
import one.payout.payment.impl.method.api.CreateCheckoutMethod
import one.payout.payment.impl.method.api.RetrieveCheckoutMethod
import java.net.http.HttpClient
import java.util.*

private val logger = KotlinLogging.logger {}

internal class PayoutClientImpl private constructor(
    private val httpClient: HttpClient,
    private val baseUrl: String,
    private val clientId: String,
    private val clientSecretSupplier: PayoutClient.ClientSecretSupplier,
    nonceGenerator: PayoutClient.NonceGenerator,
) : PayoutClient {
    private val authorizer = PayoutAuthorizer(::authorizeInternal)
    private val signer = PayoutSigner(clientSecretSupplier, nonceGenerator)

    init {
        logger.info { "PayoutClient created for baseUrl \"$baseUrl\" and clientId \"$clientId\"" }
    }

    companion object {
        fun newBuilder(): PayoutClient.Builder = BuilderImpl()
    }

    override fun authorize(): ApiResponse<AuthorizeResponse> = authorizer.authorize()

    override fun createCheckout(createCheckoutRequest: CreateCheckoutRequest): ApiResponse<Checkout> =
        CreateCheckoutMethod.execute(createCheckoutRequest)

    override fun retrieveCheckout(checkoutId: Long): ApiResponse<Checkout?> = RetrieveCheckoutMethod.execute(checkoutId)

    override fun parseWebhook(body: String): Webhook = WebhookParser.execute(body)

    private fun <T, R> ClientMethod<T, R>.execute(data: R): T =
        execute(
            data,
            ClientMethod.ClientContext(authorizer, httpClient, baseUrl, clientId, clientSecretSupplier, signer)
        )

    private fun <T> ClientMethod<T, Unit>.execute(): T = execute(Unit)

    private fun authorizeInternal(): ApiResponse<AuthorizeResponse> = AuthorizeMethod.execute()

    private class BuilderImpl : PayoutClient.Builder {
        private var clientId: String? = null
        private var clientSecretSupplier: PayoutClient.ClientSecretSupplier? = null
        private var baseUrl = PayoutClient.PROD_BASE_URL
        private var nonceGenerator: PayoutClient.NonceGenerator =
            PayoutClient.NonceGenerator { UUID.randomUUID().toString() }
        private val httpBuilder = HttpClient.newBuilder()

        override fun clientId(clientId: String): PayoutClient.Builder = apply {
            this.clientId = clientId
        }

        override fun clientSecretSupplier(
            clientSecretSupplier: PayoutClient.ClientSecretSupplier
        ): PayoutClient.Builder = apply {
            this.clientSecretSupplier = clientSecretSupplier
        }

        override fun baseUrl(baseUrl: String): PayoutClient.Builder = apply {
            this.baseUrl = baseUrl
        }

        override fun nonceGenerator(nonceGenerator: PayoutClient.NonceGenerator): PayoutClient.Builder = apply {
            this.nonceGenerator = nonceGenerator
        }

        override fun adjustHttp(block: HttpClient.Builder.() -> Unit): PayoutClient.Builder = apply {
            httpBuilder.apply(block)
        }

        override fun build(): PayoutClientImpl {
            val clientId = clientId ?: throw IllegalStateException("clientId must be set")
            val clientSecretSupplier = clientSecretSupplier
                ?: throw IllegalStateException("clientSecretSupplier must be set")
            return PayoutClientImpl(httpBuilder.build(), baseUrl, clientId, clientSecretSupplier, nonceGenerator)
        }
    }

}
