package one.payout.payment.impl.method

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.module.kotlin.readValue
import com.fasterxml.jackson.module.kotlin.treeToValue
import io.github.oshai.kotlinlogging.KotlinLogging
import one.payout.payment.api.PayoutClient.PayoutClientException
import one.payout.payment.api.model.Webhook
import one.payout.payment.impl.PayoutSigner

private val logger = KotlinLogging.logger {}

internal object WebhookParser : ClientMethod<Webhook, String>(logger) {
    override fun execute(data: String, clientContext: ClientContext): Webhook =
        kotlin
            .runCatching {
                objectMapper.readValue<WebhookInternal>(data)
                    .apply { checkSignature(clientContext.signer) }
                    .toWebhook()
            }
            .getOrElse {
                throw if (it is PayoutClientException) it
                else PayoutClientException("Failed to parse webhook: $it", it)
            }

    private fun WebhookInternal.checkSignature(signer: PayoutSigner) {
        if (!signer.isSignatureValid(nonce, signature, externalId, type))
            throw PayoutClientException("Signature of webhook is invalid")
    }

    private fun WebhookInternal.toWebhook() = Webhook(
        externalId = externalId,
        `object` = `object`,
        type = type,
        data = parseData(),
    )

    private fun WebhookInternal.parseData(): Result<Any> =
        kotlin.runCatching {
            if (`object` != "webhook") throw PayoutClientException("Unsupported object: $`object`")
            when (val dataObject = data["object"]?.asText()) {
                "checkout" -> objectMapper.treeToValue<Webhook.WebhookCheckout>(data)
                "payu_token" -> objectMapper.treeToValue<Webhook.PayuToken>(data)
                else -> throw PayoutClientException("Unsupported data object: $dataObject")
            }
        }

    private data class WebhookInternal(
        val externalId: String,
        val `object`: String,
        val type: String,
        val nonce: String,
        val signature: String,
        val data: JsonNode,
    )
}
