package one.payout.payment.impl.method.api

import com.fasterxml.jackson.databind.json.JsonMapper
import io.github.oshai.kotlinlogging.KLogger
import one.payout.payment.api.PayoutClient
import one.payout.payment.api.PayoutClient.AuthorizeException
import one.payout.payment.api.PayoutClient.PayoutClientException
import one.payout.payment.api.model.ApiResponse
import one.payout.payment.impl.PayoutAuthorizer
import one.payout.payment.impl.PayoutSigner
import one.payout.payment.impl.method.ClientMethod
import java.net.URI
import java.net.http.HttpClient
import java.net.http.HttpRequest
import java.net.http.HttpResponse

internal abstract class ClientApiMethod<T, R>(
    private val logger: KLogger,
    private val withAuthorization: Boolean = true,
    private val requestMethod: String = "GET",
    private val on404: (() -> T)? = null,
) : ClientMethod<ApiResponse<T>, R>(logger) {
    override fun execute(data: R, clientContext: ClientContext): ApiResponse<T> {
        val requestContext = getRequestContextOrThrow(data, clientContext)
        return kotlin
            .runCatching {
                executeAuthorizationAware(
                    ExecutionContext(
                        clientContext,
                        getApiPath(data),
                        requestContext,
                        data
                    )
                )
            }
            .getOrElse { throw PayoutClientException("Failed to $requestContext: $it", it) }
    }

    protected abstract fun getRequestContext(data: R, clientContext: ClientContext): String

    protected abstract fun getApiPath(data: R): String

    protected open fun logCallingStart(executionContext: ExecutionContext<R>) {
        logger.info { "Calling ${executionContext.requestContext}..." }
    }

    protected open fun ExecutionContext<R>.createRequestObject(): Any? = null

    protected open fun logRequestBody(requestBody: String, executionContext: ExecutionContext<R>) {
        logger.info { "${executionContext.requestContext} body: $requestBody" }
    }

    protected open fun HttpRequest.Builder.customize(executionContext: ExecutionContext<R>) {}

    protected open fun logResponse(statusCode: Int, body: String?, executionContext: ExecutionContext<R>) {
        logger.info { "${executionContext.requestContext} response: $statusCode: $body" }
    }

    protected abstract fun JsonMapper.readResponse(body: String): T

    protected open fun T.onSuccess(executionContext: ExecutionContext<R>) {}

    private fun getRequestContextOrThrow(data: R, clientContext: ClientContext): String =
        kotlin
            .runCatching { getRequestContext(data, clientContext) }
            .getOrElse {
                throw PayoutClientException(
                    "Failed to determine request context for ${this::class.simpleName}: $it",
                    it
                )
            }

    private fun executeAuthorizationAware(executionContext: ExecutionContext<R>): ApiResponse<T> {
        logCallingStart(executionContext)
        if (!withAuthorization) {
            return execute(null, executionContext)
        } else {
            val token = executionContext.authorizer.getToken()
            val firstTry = execute(token.value, executionContext)
            if (firstTry.result.exceptionOrNull() !is AuthorizeException) return firstTry
            executionContext.authorizer.invalidateToken(token.value)
            if (token.newCreated) return firstTry
            return execute(executionContext.authorizer.getToken().value, executionContext)
        }
    }

    private fun execute(token: String?, executionContext: ExecutionContext<R>): ApiResponse<T> =
        executionContext.httpClient
            .send(
                createRequest(token, executionContext),
                HttpResponse.BodyHandlers.ofString()
            )
            .processResponse(executionContext)

    private fun createRequest(token: String?, executionContext: ExecutionContext<R>): HttpRequest {
        val requestBody = executionContext.createRequestObject()
            ?.let { objectMapper.writeValueAsString(it) }
            ?.also { logRequestBody(it, executionContext) }
        return HttpRequest.newBuilder()
            .uri(URI.create(executionContext.url))
            .method(requestMethod, requestBody)
            .header("Content-Type", "application/json")
            .header("Accept", "application/json")
            .also { if (token != null) it.header("Authorization", "Bearer $token") }
            .apply { customize(executionContext) }
            .build()
    }

    private fun HttpRequest.Builder.method(method: String, body: String?): HttpRequest.Builder = apply {
        if (body != null) {
            method(method, HttpRequest.BodyPublishers.ofString(body))
        } else when (method) {
            "GET" -> GET()
            "DELETE" -> DELETE()
            else -> method(method, HttpRequest.BodyPublishers.noBody())
        }
    }

    private fun HttpResponse<String>.processResponse(executionContext: ExecutionContext<R>): ApiResponse<T> {
        logResponse(statusCode(), body(), executionContext)

        return kotlin
            .runCatching {
                when {
                    statusCode() in 200 until 300 -> apiResponseSuccess(objectMapper.readResponse(body()))

                    statusCode() == 401 -> apiResponseFailure(
                        AuthorizeException(
                            "${executionContext.requestContext} failed: ${readError()}"
                        )
                    )

                    statusCode() == 404 && on404 != null -> apiResponseSuccess(on404.invoke())

                    else -> apiResponseFailure(
                        PayoutClientException(
                            "${executionContext.requestContext} failed: ${readError()}"
                        )
                    )
                }
            }
            .mapCatching { it.getOrNull()?.apply { onSuccess(executionContext) }; it }
            .getOrElse {
                apiResponseFailure(
                    PayoutClientException("${executionContext.requestContext} failed: $it", it),
                )
            }
    }

    private fun HttpResponse<String>.readError(): String = kotlin
        .runCatching { objectMapper.readTree(body()).get("errors").asText()!! }
        .getOrElse { body() }

    private fun HttpResponse<String>.apiResponseSuccess(value: T) = apiResponse(Result.success(value))

    private fun HttpResponse<String>.apiResponseFailure(exception: Throwable) = apiResponse(Result.failure(exception))

    private fun HttpResponse<String>.apiResponse(result: Result<T>) =
        ApiResponse(statusCode(), body(), result)

    protected data class ExecutionContext<R>(
        private val clientContext: ClientContext,
        val apiPath: String,
        val requestContext: String,
        val data: R,
    ) {
        val url: String = clientContext.baseUrl + apiPath
        val authorizer: PayoutAuthorizer get() = clientContext.authorizer
        val httpClient: HttpClient get() = clientContext.httpClient
        val clientId: String get() = clientContext.clientId
        val clientSecretSupplier: PayoutClient.ClientSecretSupplier get() = clientContext.clientSecretSupplier
        val signer: PayoutSigner get() = clientContext.signer
    }
}
