package one.payout.payment.impl.method.api

import com.fasterxml.jackson.databind.json.JsonMapper
import com.fasterxml.jackson.module.kotlin.readValue
import io.github.oshai.kotlinlogging.KotlinLogging
import one.payout.payment.api.PayoutClient.PayoutClientException
import one.payout.payment.api.model.Checkout
import one.payout.payment.api.model.CreateCheckoutRequest
import one.payout.payment.impl.PayoutSigner
import one.payout.payment.impl.model.CreateCheckoutApiRequest
import java.net.http.HttpRequest

private val logger = KotlinLogging.logger {}

internal object CreateCheckoutMethod : ClientApiMethod<Checkout, CreateCheckoutRequest>(
    logger = logger,
    requestMethod = "POST",
) {
    override fun getRequestContext(data: CreateCheckoutRequest, clientContext: ClientContext): String =
        "${data.javaClass.simpleName} for externalId \"${data.externalId}\"" +
                " and idempotencyKey \"${data.idempotencyKey}\""

    override fun getApiPath(data: CreateCheckoutRequest): String = "/api/v1/checkouts"

    override fun ExecutionContext<CreateCheckoutRequest>.createRequestObject(): CreateCheckoutApiRequest =
        data.toApi(signer)

    override fun HttpRequest.Builder.customize(executionContext: ExecutionContext<CreateCheckoutRequest>) {
        executionContext.data.idempotencyKey?.let { header("Idempotency-Key", it) }
    }

    override fun JsonMapper.readResponse(body: String): Checkout = readValue(body)

    override fun Checkout.onSuccess(executionContext: ExecutionContext<CreateCheckoutRequest>) {
        if (!executionContext.signer.isSignatureValid(nonce, signature, amount, currency, externalId)) {
            throw PayoutClientException("Signature of retrieved checkout is invalid")
        }
    }

    private fun CreateCheckoutRequest.toApi(signer: PayoutSigner): CreateCheckoutApiRequest {
        val mode: CreateCheckoutApiRequest.CheckoutMode =
            when {
                recurrentCardToken != null -> CreateCheckoutApiRequest.CheckoutMode.RECURRENT
                rememberedCardToken != null -> CreateCheckoutApiRequest.CheckoutMode.CARD_ON_FILE
                storeCardMode != CreateCheckoutRequest.StoreCardMode.NO_STORE -> CreateCheckoutApiRequest.CheckoutMode.STORE_CARD
                else -> CreateCheckoutApiRequest.CheckoutMode.STANDARD
            }

        val signature = signer.sign(amount, currency, externalId)
        return CreateCheckoutApiRequest(
            amount = amount,
            currency = currency,
            customer = customer,
            billingAddress = billingAddress,
            shippingAddress = shippingAddress,
            products = products,
            externalId = externalId,
            metadata = objectMapper.valueToTree(metadata),
            mode = mode,
            recurring = storeCardMode == CreateCheckoutRequest.StoreCardMode.STORE_CARD_FOR_RECURRENT_PAYMENT,
            nonce = signature.nonce,
            cardToken = rememberedCardToken,
            recurrentToken = recurrentCardToken,
            redirectUrl = redirectUrl,
            signature = signature.value,
        )
    }

}
