/*
 * Decompiled with CFR 0.152.
 */
package one.tomorrow.transactionaloutbox.repository;

import jakarta.persistence.EntityManager;
import jakarta.persistence.LockModeType;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.TypedQuery;
import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import lombok.Generated;
import one.tomorrow.transactionaloutbox.model.OutboxLock;
import org.hibernate.dialect.lock.LockingStrategyException;
import org.hibernate.exception.ConstraintViolationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

@Repository
public class OutboxLockRepository {
    private static final Logger logger = LoggerFactory.getLogger(OutboxLockRepository.class);
    @PersistenceContext
    private EntityManager entityManager;

    @Transactional(propagation=Propagation.REQUIRES_NEW)
    public boolean acquireOrRefreshLock(String ownerId, Duration timeout) {
        OutboxLock lock = null;
        Instant now = Instant.now();
        try {
            lock = (OutboxLock)this.entityManager.find(OutboxLock.class, (Object)"outboxLock");
            if (lock == null) {
                logger.debug("No outbox lock found. Creating one for {}", (Object)ownerId);
                lock = new OutboxLock(ownerId, now.plus(timeout));
            } else if (ownerId.equals(lock.getOwnerId())) {
                logger.debug("Found outbox lock with requested owner {}, valid until {} - updating lock", (Object)lock.getOwnerId(), (Object)lock.getValidUntil());
                this.entityManager.lock((Object)lock, LockModeType.PESSIMISTIC_WRITE, Map.of("jakarta.persistence.lock.timeout", 0));
                lock.setValidUntil(now.plus(timeout));
            } else {
                if (lock.getValidUntil().isAfter(now)) {
                    logger.debug("Found outbox lock with owner {}, valid until {}", (Object)lock.getOwnerId(), (Object)lock.getValidUntil());
                    this.tryRollback();
                    return false;
                }
                logger.info("Found expired outbox lock with owner {}, which was valid until {} - grabbing lock for {}", new Object[]{lock.getOwnerId(), lock.getValidUntil(), ownerId});
                this.entityManager.lock((Object)lock, LockModeType.PESSIMISTIC_WRITE, Map.of("jakarta.persistence.lock.timeout", 0));
                lock.setOwnerId(ownerId);
                lock.setValidUntil(now.plus(timeout));
            }
            this.entityManager.persist((Object)lock);
            this.entityManager.flush();
            logger.info("Acquired or refreshed outbox lock for owner {}, valid until {}", (Object)ownerId, (Object)lock.getValidUntil());
            return true;
        }
        catch (LockingStrategyException e) {
            return this.handleException(e, ownerId, lock);
        }
        catch (ConstraintViolationException e) {
            return this.handleException(e, ownerId);
        }
        catch (Throwable e) {
            Throwable throwable = e.getCause();
            if (throwable instanceof ConstraintViolationException) {
                ConstraintViolationException constraintViolationException = (ConstraintViolationException)throwable;
                return this.handleException(constraintViolationException, ownerId);
            }
            throwable = e.getCause();
            if (throwable instanceof LockingStrategyException) {
                LockingStrategyException lockingStrategyException = (LockingStrategyException)throwable;
                return this.handleException(lockingStrategyException, ownerId, lock);
            }
            logger.warn("Outbox lock selection/acquisition for owner {} failed", (Object)ownerId, (Object)e);
            this.tryRollback();
            throw e;
        }
    }

    private boolean handleException(LockingStrategyException e, String ownerId, OutboxLock lock) {
        String reason = e.getCause() != null ? e.getCause().toString() : e.toString();
        logger.info("Could not grab lock {} for owner {} - database row is locked: {}", new Object[]{lock, ownerId, reason});
        this.tryRollback();
        return false;
    }

    private boolean handleException(ConstraintViolationException e, String ownerId) {
        String reason = e.getCause() != null ? e.getCause().toString() : e.toString();
        logger.info("Outbox lock for owner {} could not be created, another one has been created concurrently: {}", (Object)ownerId, (Object)reason);
        this.tryRollback();
        return false;
    }

    private void tryRollback() {
        try {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }
        catch (Exception ex) {
            logger.info("Caught exception while rolling back OutBox transaction", (Throwable)ex);
        }
    }

    public boolean preventLockStealing(String ownerId) {
        Optional lock = this.queryByOwnerId(ownerId).setLockMode(LockModeType.PESSIMISTIC_READ).getResultStream().findFirst();
        return lock.isPresent();
    }

    @Transactional
    public void releaseLock(String ownerId) {
        this.queryByOwnerId(ownerId).getResultStream().findFirst().ifPresentOrElse(lock -> {
            this.entityManager.remove(lock);
            this.entityManager.flush();
            logger.info("Released outbox lock for owner {}", (Object)ownerId);
        }, () -> logger.debug("Outbox lock for owner {} not found", (Object)ownerId));
    }

    private TypedQuery<OutboxLock> queryByOwnerId(String ownerId) {
        return this.entityManager.createQuery("FROM OutboxLock WHERE ownerId = :ownerId", OutboxLock.class).setParameter("ownerId", (Object)ownerId);
    }

    @Generated
    public OutboxLockRepository() {
    }

    @Generated
    public OutboxLockRepository(EntityManager entityManager) {
        this.entityManager = entityManager;
    }
}

