/*
 * Decompiled with CFR 0.152.
 */
package one.tomorrow.transactionaloutbox.repository;

import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import one.tomorrow.transactionaloutbox.model.OutboxLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.jdbc.UncategorizedSQLException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.StringUtils;

@Repository
public class OutboxLockRepository {
    private static final Logger logger = LoggerFactory.getLogger(OutboxLockRepository.class);
    private static final ResultSetExtractor<OutboxLock> resultSetExtractor = rs -> rs.next() ? new OutboxLock(rs.getString("owner_id"), rs.getTimestamp("valid_until").toInstant()) : null;
    private final JdbcTemplate jdbcTemplate;
    private final TransactionTemplate transactionTemplate;

    public OutboxLockRepository(JdbcTemplate jdbcTemplate, PlatformTransactionManager transactionManager) {
        this.jdbcTemplate = jdbcTemplate;
        this.transactionTemplate = new TransactionTemplate(transactionManager);
    }

    @Transactional(propagation=Propagation.REQUIRES_NEW)
    public boolean acquireOrRefreshLock(String ownerId, Duration timeout) {
        return (Boolean)this.transactionTemplate.execute(status -> this.acquireOrRefreshLock(ownerId, timeout, status));
    }

    private boolean acquireOrRefreshLock(String ownerId, Duration timeout, TransactionStatus txStatus) {
        Instant now = Instant.now();
        try {
            Instant validUntil;
            OutboxLock lock = (OutboxLock)this.jdbcTemplate.query("select * from outbox_kafka_lock where id = 'outboxLock'", resultSetExtractor);
            if (lock == null) {
                logger.info("No outbox lock found. Creating one for {}", (Object)ownerId);
                validUntil = now.plus(timeout);
                this.jdbcTemplate.update("insert into outbox_kafka_lock (id, owner_id, valid_until) values (?, ?, ?)", new Object[]{"outboxLock", ownerId, Timestamp.from(validUntil)});
            } else if (ownerId.equals(lock.getOwnerId())) {
                logger.debug("Found outbox lock with requested owner {}, valid until {} - updating lock", (Object)lock.getOwnerId(), (Object)lock.getValidUntil());
                this.jdbcTemplate.execute("select * from outbox_kafka_lock where id = 'outboxLock' for update nowait");
                validUntil = now.plus(timeout);
                this.jdbcTemplate.update("update outbox_kafka_lock set valid_until = ? where id = 'outboxLock'", new Object[]{Timestamp.from(validUntil)});
            } else {
                if (lock.getValidUntil().isAfter(now)) {
                    logger.debug("Found outbox lock with owner {}, valid until {}", (Object)lock.getOwnerId(), (Object)lock.getValidUntil());
                    this.tryRollback();
                    return false;
                }
                logger.info("Found expired outbox lock with owner {}, which was valid until {} - grabbing lock for {}", new Object[]{lock.getOwnerId(), lock.getValidUntil(), ownerId});
                this.jdbcTemplate.execute("select * from outbox_kafka_lock where id = 'outboxLock' for update nowait");
                validUntil = now.plus(timeout);
                this.jdbcTemplate.update("update outbox_kafka_lock set owner_id = ?, valid_until = ? where id = 'outboxLock'", new Object[]{ownerId, Timestamp.from(validUntil)});
            }
            txStatus.flush();
            logger.debug("Acquired or refreshed outbox lock for owner {}, valid until {}", (Object)ownerId, (Object)validUntil);
            return true;
        }
        catch (UncategorizedSQLException e) {
            return this.handleException(e, ownerId);
        }
        catch (DuplicateKeyException e) {
            return this.handleException(e, ownerId);
        }
        catch (Throwable e) {
            Throwable throwable = e.getCause();
            if (throwable instanceof DuplicateKeyException) {
                DuplicateKeyException duplicateKeyException = (DuplicateKeyException)throwable;
                return this.handleException(duplicateKeyException, ownerId);
            }
            logger.warn("Outbox lock selection/acquisition for owner {} failed", (Object)ownerId, (Object)e);
            this.tryRollback();
            throw e;
        }
    }

    private boolean handleException(UncategorizedSQLException e, String ownerId) {
        if (e.getMessage().contains("could not obtain lock")) {
            String reason = e.getCause() != null ? e.getCause().toString() : e.toString();
            logger.info("Could not grab lock for owner {} - database row is locked: {}", (Object)ownerId, (Object)reason);
        } else {
            logger.warn("Failed to grab lock for owner {} - uncategorized exception", (Object)ownerId, (Object)e);
        }
        this.tryRollback();
        return false;
    }

    private boolean handleException(DuplicateKeyException e, String ownerId) {
        String reason = e.getCause() != null ? e.getCause().toString() : e.toString();
        logger.info("Outbox lock for owner {} could not be created, another one has been created concurrently: {}", (Object)ownerId, (Object)reason);
        this.tryRollback();
        return false;
    }

    private void tryRollback() {
        try {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
        }
        catch (Exception ex) {
            logger.info("Caught exception while rolling back OutBox transaction", (Throwable)ex);
        }
    }

    public boolean preventLockStealing(String ownerId) {
        Optional<OutboxLock> lock = this.queryByOwnerId(ownerId, " for share");
        return lock.isPresent();
    }

    @Transactional
    public void releaseLock(String ownerId) {
        this.transactionTemplate.executeWithoutResult(status -> this.queryByOwnerId(ownerId).ifPresentOrElse(lock -> {
            this.jdbcTemplate.update("delete from outbox_kafka_lock where owner_id = ?", new Object[]{ownerId});
            status.flush();
            logger.info("Released outbox lock for owner {}", (Object)ownerId);
        }, () -> logger.debug("Outbox lock for owner {} not found", (Object)ownerId)));
    }

    private Optional<OutboxLock> queryByOwnerId(String ownerId) {
        return this.queryByOwnerId(ownerId, null);
    }

    private Optional<OutboxLock> queryByOwnerId(String ownerId, String lock) {
        Object sql = "select * from outbox_kafka_lock where owner_id = ? ";
        if (StringUtils.hasLength((String)lock)) {
            sql = (String)sql + lock;
        }
        return Optional.ofNullable((OutboxLock)this.jdbcTemplate.query((String)sql, resultSetExtractor, new Object[]{ownerId}));
    }
}

