package one.veriph.sdk.repository

import dagger.Component
import kotlinx.coroutines.flow.FlowCollector
import one.veriph.sdk.data.SessionStyle
import one.veriph.sdk.data.VerificationEvent
import one.veriph.sdk.data.VerificationStatusParams
import one.veriph.sdk.data.getErrorObject
import one.veriph.sdk.network.RetrofitInstance
import one.veriph.sdk.network.VerificationDataSource
import one.veriph.sdk.ui.util.StateBundle
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
@Component(
    modules = [
        VerificationDataSource::class,
        RetrofitInstance::class
    ]
)
internal interface VerificationRepositoryFactory {
    fun verificationRepo(): VerificationRepository
}

class VerificationRepository @Inject constructor(private val verificationDataSource: VerificationDataSource) {

    suspend fun getSessionStyle(apiKey: String, sessionUuid: String): StateBundle<SessionStyle> {
        val result = verificationDataSource.getSessionStyle(apiKey, sessionUuid)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject())
    }

    suspend fun getCountryCodes(apiKey: String): StateBundle<List<one.veriph.sdk.data.CountryCode>?> {
        val result = verificationDataSource.getCountryCodes(apiKey)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject())
    }

    suspend fun createAttempt(
        apiKey: String,
        sessionUuid: String,
        fingerprint: String,
        phoneNumber: one.veriph.sdk.data.PhoneBundle?,
        cancellationReason: Int?
    ): StateBundle<one.veriph.sdk.data.AttemptBundle> {
        val attemptInput =
            one.veriph.sdk.data.AttemptInput(
                fingerprint,
                sessionUuid,
                phoneNumber,
                cancellationReason
            )
        val result = verificationDataSource.createAttempt(apiKey, attemptInput)

        val errorObj = if (!result.isSuccessful) result.errorBody()?.getErrorObject() else null
        val errorCode = errorObj?.internalErrorCode

        val bundle = one.veriph.sdk.data.AttemptBundle(result.body(), result.code(), errorCode)

        return when (bundle.status) {
            one.veriph.sdk.data.AttemptStatus.OPENED -> StateBundle(bundle)

            one.veriph.sdk.data.AttemptStatus.CLOSED,
            one.veriph.sdk.data.AttemptStatus.EXCEEDED,
            one.veriph.sdk.data.AttemptStatus.NEEDS_INPUT -> StateBundle(
                bundle
            )

            one.veriph.sdk.data.AttemptStatus.ERROR -> StateBundle(null, errorObj)
        }
    }

    suspend fun submitAttempt(
        apiKey: String,
        input: one.veriph.sdk.data.AttemptSubmission
    ): StateBundle<one.veriph.sdk.data.AttemptStatusResponse?> {
        val result = verificationDataSource.submitAttempt(apiKey, input)
        return if (result.isSuccessful)
            StateBundle(result.body())
        else
            StateBundle(error = result.errorBody()?.getErrorObject())
    }

    suspend fun subscribeToStatus(
        uuid: String,
        apiKey: String,
        params: VerificationStatusParams,
        collector: FlowCollector<VerificationEvent>
    ) {
        // TODO Integrate Server-Sent Events properly and use both
        //verificationDataSource.subscribeToEvents(uuid).collect(collector)
        verificationDataSource.subscribeViaPolling(apiKey, params).collect(collector)
    }
}