package one.veriph.sdk.ui.verification

import android.content.Intent
import android.os.Bundle
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.lifecycleScope
import androidx.lifecycle.repeatOnLifecycle
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import kotlinx.coroutines.launch
import one.veriph.sdk.R
import one.veriph.sdk.ui.util.FlowScreen
import one.veriph.sdk.ui.util.getFingerprint
import one.veriph.sdk.ui.verification.screen.SetupScreen
import one.veriph.sdk.ui.verification.screen.createAttempt
import one.veriph.sdk.ui.verification.screen.executeAttempt
import one.veriph.sdk.ui.verification.screen.navigateToAttemptExecution
import one.veriph.sdk.ui.verification.screen.navigateToCreateAttempt
import one.veriph.sdk.ui.verification.screen.navigateToPhoneForm
import one.veriph.sdk.ui.verification.screen.openPhoneForm

class VerificationActivity : ComponentActivity() {
    private val viewModel: VerificationViewModel by viewModels()

    private fun onVerificationEnded() {
        val i = Intent()
        val session = viewModel.sessionUuid.value
        if (session != null)
            i.putExtra(EXTRA_SESSION_UUID, session)
        setResult(RESULT_OK, i)
        finish()
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val extFingerprint = getFingerprint(applicationContext)
        val sessionUuid = intent.getStringExtra(EXTRA_SESSION_UUID)
        val apiKey = intent.getStringExtra(EXTRA_API_KEY)

        if (sessionUuid == null || apiKey == null) {
            Toast.makeText(this, R.string.error_no_params, Toast.LENGTH_LONG).show()
            setResult(RESULT_CANCELED)
            finish()
        } else {
            lifecycleScope.launch {
                repeatOnLifecycle(Lifecycle.State.RESUMED) {
                    viewModel.sessionClosed.collect {
                        if (it.state != null && it.state) {
                            onVerificationEnded()
                        }
                    }
                }
            }

            setContent {
                val navController = rememberNavController()
                NavHost(
                    navController = navController,
                    startDestination = FlowScreen.Setup.route
                ) {
                    composable(FlowScreen.Setup.route) {
                        SetupScreen(
                            viewModel = viewModel,
                            sessionUuid = sessionUuid,
                            apiKey = apiKey,
                            fingerprint = extFingerprint,
                            onFinish = {
                                navController.navigateToCreateAttempt()
                            }
                        )
                    }

                    openPhoneForm(
                        viewModel = viewModel,
                        onSubmit = {
                            navController.navigateToCreateAttempt()
                        }
                    )

                    createAttempt(
                        viewModel = viewModel,
                        onExecution = {
                            navController.navigateToAttemptExecution()
                        },
                        onInputNeeded = {
                            navController.navigateToPhoneForm()
                        }, onFinish = {
                            onVerificationEnded()
                        })
                    executeAttempt(
                        viewModel = viewModel,
                        onExpiration = {
                            viewModel.invalidateAttempt()
                            navController.navigateToCreateAttempt()
                        },
                        onCancellation = {
                            viewModel.invalidateAttempt(it)
                            navController.navigateToCreateAttempt()
                        },
                        onEditNumber = {
                            navController.navigateToPhoneForm()
                        }
                    )
                }
            }
        }
    }

    companion object {
        const val EXTRA_SESSION_UUID: String = "sessionUuid"
        const val EXTRA_API_KEY: String = "apiKey"
    }
}