package one.veriph.sdk.ui.verification.composable

import android.content.Context
import android.content.Intent
import android.net.Uri
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import one.veriph.sdk.R
import one.veriph.sdk.data.VerificationMethod

@Composable
fun InverseOTPAttempt(
    attempt: one.veriph.sdk.data.AttemptCreationResponse,
    onExpiration: () -> Unit
) {
    val context = LocalContext.current
    val messageContent =
        stringResource(id = R.string.attempt_execution_inverse_message, attempt.payload.hash!!)
    val typeLabel = stringResource(
        id = when (attempt.type) {
            VerificationMethod.INVERSE_OTP_SMS -> R.string.v1_sdk_generic_channel_sms
            VerificationMethod.INVERSE_OTP_WHATSAPP -> R.string.v1_sdk_generic_channel_whatsapp
            else -> R.string.v1_sdk_generic_channel_unknown
        }
    )

    Text(
        stringResource(id = R.string.attempt_execution_inverse_title),
        modifier = Modifier.padding(bottom = 12.dp),
        textAlign = TextAlign.Center,
        color = MaterialTheme.colorScheme.secondary,
        fontSize = 24.sp,
    )
    Text(
        stringResource(
            id = R.string.attempt_execution_inverse_description,
            typeLabel
        ),
        modifier = Modifier.padding(bottom = 24.dp),
        textAlign = TextAlign.Center,
    )
    Button(
        modifier = Modifier.padding(bottom = 24.dp),
        onClick = {
            sendMessage(context, attempt, messageContent)
        }) {
        Text(stringResource(id = R.string.attempt_execution_inverse_btn_send))
    }
    CircularProgressIndicator(
        modifier = Modifier
            .width(32.dp)
            .padding(bottom = 12.dp),
        color = MaterialTheme.colorScheme.primary,
        trackColor = MaterialTheme.colorScheme.surfaceVariant,
    )
    CountdownTimer(context, attempt, onExpiration)
}

fun sendMessage(
    context: Context,
    attempt: one.veriph.sdk.data.AttemptCreationResponse,
    content: String
) {
    val destination = attempt.payload.destination!!
    when (attempt.type) {
        VerificationMethod.INVERSE_OTP_WHATSAPP -> {
            sendWhatsApp(context, destination, content)
        }

        else -> {
            sendSMS(context, destination, content)
        }
    }
}

fun sendSMS(context: Context, destination: String, message: String) {
    val intent = Intent(Intent.ACTION_SENDTO).apply {
        data = Uri.parse("smsto:$destination")
        putExtra("sms_body", message)
    }

    if (intent.resolveActivity(context.packageManager) != null) {
        context.startActivity(intent)
    } else try {
        context.startActivity(intent)
    } catch (e: Exception) {
        e.printStackTrace()
    }
}

fun sendWhatsApp(context: Context, destination: String, message: String) {
    val url = "https://wa.me/${destination}?text=${Uri.encode(message)}"
    val webpage: Uri = Uri.parse(url)
    val intent = Intent(Intent.ACTION_VIEW, webpage)
    if (intent.resolveActivity(context.packageManager) != null) {
        context.startActivity(intent)
    } else try {
        context.startActivity(intent)
    } catch (e: Exception) {
        e.printStackTrace()
    }
}