package one.veriph.sdk.ui.verification.composable

import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Button
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import one.veriph.sdk.R
import one.veriph.sdk.data.VerificationMethod
import one.veriph.sdk.ui.util.onlyContainsNumbers
import one.veriph.sdk.ui.verification.VerificationViewModel

@Composable
fun TraditionalOTPAttempt(
    viewModel: VerificationViewModel,
    attempt: one.veriph.sdk.data.AttemptCreationResponse,
    onExpiration: () -> Unit,
) {
    val context = LocalContext.current

    val submission = viewModel.submissionRes.collectAsState()
    var input by rememberSaveable { mutableStateOf("") }
    var inputError by rememberSaveable { mutableStateOf(false) }
    var canRetry by rememberSaveable { mutableStateOf(false) }
    var errorBundle by rememberSaveable { mutableStateOf<one.veriph.sdk.data.ErrorBundle?>(null) }
    val typeLabel = stringResource(
        id = when (attempt.type) {
            VerificationMethod.OTP_SMS -> R.string.v1_sdk_generic_channel_sms
            VerificationMethod.OTP_WHATSAPP -> R.string.v1_sdk_generic_channel_whatsapp
            VerificationMethod.ROBOCALL -> R.string.v1_sdk_generic_channel_robocall
            else -> R.string.v1_sdk_generic_channel_unknown
        }
    )

    val submissionRunning = submission.value.isLoading

    Text(
        stringResource(id = R.string.attempt_execution_traditional_title),
        modifier = Modifier.padding(bottom = 12.dp),
        textAlign = TextAlign.Center,
        color = MaterialTheme.colorScheme.secondary,
        fontSize = 24.sp,
    )
    Text(
        stringResource(
            id = R.string.attempt_execution_traditional_description,
            typeLabel,
            attempt.userPhoneNumberHint ?: ""
        ),
        modifier = Modifier.padding(bottom = 24.dp),
        textAlign = TextAlign.Center,
    )

    OutlinedTextField(
        value = input,
        singleLine = true,
        enabled = !submissionRunning,
        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
        modifier = Modifier
            .fillMaxWidth()
            .padding(bottom = 12.dp),
        onValueChange = {
            if (it.isEmpty() || onlyContainsNumbers(it) || it.length <= 6) {
                input = it
            }

            inputError = input.length != 6

            if (input.length == 6) {
                viewModel.submitAttempt(input)
            }
        },
        label = { Text(stringResource(id = R.string.attempt_execution_traditional_hint)) },
        isError = inputError
    )

    if (errorBundle != null)
        ErrorMessage(errorBundle = errorBundle!!)

    Button(
        modifier = Modifier.padding(bottom = 24.dp),
        enabled = canRetry && !submissionRunning,
        onClick = {
            onExpiration()
        }) {
        Text(stringResource(id = R.string.attempt_execution_traditional_btn_retry))
    }
    CountdownTimer(context, attempt) {
        canRetry = true
    }

    if (submission.value.error != null) {
        errorBundle = submission.value.error
        input = ""
        viewModel.resetAttemptSubmission()
    }
}

@Composable
fun ErrorMessage(errorBundle: one.veriph.sdk.data.ErrorBundle) {
    val errorMsgRes = if (errorBundle.internalErrorCode == 20015)
        R.string.attempt_execution_traditional_error_code
    else
        R.string.attempt_execution_traditional_error_generic
    Text(
        stringResource(id = errorMsgRes),
        color = MaterialTheme.colorScheme.error,
        modifier = Modifier.padding(bottom = 24.dp, top = 12.dp),
        textAlign = TextAlign.Center
    )
}